/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: AbstractFilteredPage.java 548 2010-06-18 13:09:39Z fdesbois $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.4.1/wao-ui/src/main/java/fr/ifremer/wao/ui/base/AbstractFilteredPage.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.base;

import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.List;

/**
 * AbstractFilteredPage
 *
 * Created: 14 janv. 2010
 *
 * This class uses the BoatFilter bean (also SamplingFilter and UserFilter which
 * are supertypes).
 *
 * Known implementations are {@link fr.ifremer.wao.ui.pages.Synthesis }
 * and {@link fr.ifremer.wao.ui.pages.Contacts }
 *
 * You must override the three abstract methods and call {@link #initSelectFilters(boolean, boolean, boolean)} () }
 * in the setupRender of the page.
 * <pre>
 * - BoatName uses an Autocomplete mixins :
 *      <t:textfield t:id="boatName" t:value="filter.boatName" t:mixins="Autocomplete"/>
 * - SampleRow select value is sampleRowId :
 *      <input t:type="select" t:id="sampleRow" t:model="sampleRowSelectModel" value="sampleRowId" />
 * - Company select value is companyId :
 *      <input t:type="select" t:id="company" t:model="companySelectModel" value="companyId" />
 * - Observer select value is observerId :
 *      <input t:type="select" t:id="observer" t:model="observerSelectModel" value="observerId" />
 *
 * Use submit button with specific ids to refresh data depends on select change:
 * - id="refreshByCompany" : refresh facadeSelect, sectorSelect, sampleRowSelect 
 *   and observerSelect
 * - id="refreshByFacade" : refresh sectorSelect and sampleRowSelect
 * - id="refreshBySector" : refresh sampleRowSelect
 *
 * </pre>
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
public abstract class AbstractFilteredPage {

    @Inject
    private Logger log;

    @SessionState
    private ConnectedUser user;

    @Inject
    private PropertyAccess propertyAccess;
    
    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private ServiceUser serviceUser;

    @Persist
    private GenericSelectModel<SampleRow> sampleRowSelectModel;

    @Property
    private String sampleRowId;
    
    @Persist
    private SelectModel facadeSelectModel;

    @Persist
    private SelectModel sectorSelectModel;

    @Persist
    private GenericSelectModel<Company> companySelectModel;

    @Property
    private String companyId;

    @Persist
    private GenericSelectModel<WaoUser> observerSelectModel;

    @Property
    private String observerId;

    @Persist
    private SelectModel programSelectModel;

    @Persist
    private boolean observerSelect;

    @Persist
    private boolean programSelect;

    /**
     * Need to be call in setupRender of the page which inherits this abstract
     * class.
     */
    public void initCompanyFilter()  {
        // Very very important to do that at beginning
        // Evo #2227 : Guest user has no default company filter
        if (!user.isAdmin() && !user.isGuest()) {
            if (log.isDebugEnabled()) {
                log.debug("Company filtered : " + user.getCompany().getName());
            }
            getFilter().setCompany(user.getCompany());
        }
    }

    /**
     * Initialize all select for filters. By default, only facade, sector and
     * sampleRow selects will be initialized.
     *
     * @param companySelect if true, company select will be initialized
     * @param observerSelect if true, observer select will be initialized
     * @param programSelect
     * @throws WaoException
     */
    protected void initSelectFilters(boolean companySelect,
            boolean observerSelect, boolean programSelect) throws WaoException {
        this.observerSelect = observerSelect;
        this.programSelect = programSelect;

        if (getFilter().getSampleRow() != null) {
            if (log.isDebugEnabled()) {
                log.debug("SampleRow filtered : " + getFilter().getSampleRow());
            }
            sampleRowId = getFilter().getSampleRow().getTopiaId();
        }
        
        if (user.isAdmin() && companySelect) {
            resetCompanySelect();
            if (getFilter().getCompany() != null) {
                companyId = getFilter().getCompany().getTopiaId();
            }
        }
//        else if (!user.isAdmin()) {
//            if (log.isDebugEnabled()) {
//                log.debug("Company filtered : " + user.getCompany());
//            }
//            getFilter().setCompany(user.getCompany());
//        }

        if (observerSelect) {
            observerSelectModel = null;
            getObserverSelectModel();

            if (getFilter().getObserver() != null) {
                observerId = getFilter().getObserver().getTopiaId();
            }
        }

        if (programSelect) {
            programSelectModel = null;
            getProgramSelectModel();
        }

        resetFacadeSelect();
        resetSectorSelect();
        resetSampleRowSelect();
    }

    protected void resetCompanySelect() throws WaoException {
        companySelectModel = null;
        getCompanySelectModel();
    }

    protected void resetFacadeSelect() throws WaoException {
        facadeSelectModel = null;
        getFacadeSelectModel();
    }

    protected void resetSectorSelect() throws WaoException {
        sectorSelectModel = null;
        getSectorSelectModel();
    }

    protected void resetSampleRowSelect() throws WaoException {
        sampleRowSelectModel = null;
        getSampleRowSelectModel();
    }

    /**
     * Get select model for fishingZone facade. Depends on company set in
     * filter.
     *
     * @return the SelectModel for facade names
     * @throws WaoException
     */
    public SelectModel getFacadeSelectModel() throws WaoException {
        if (facadeSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String facadeName : serviceReferential.getFacades(getFilter())) {
                options.add(new OptionModelImpl(facadeName,facadeName));
            }
            facadeSelectModel = new SelectModelImpl(null, options);
        }
        return facadeSelectModel;
    }

    /**
     * Get select model for fishingZone sector. Depends on facadeName and
     * company set in filter.
     *
     * @return the SelectModel for sector names
     * @throws WaoException
     */
    public SelectModel getSectorSelectModel() throws WaoException {
        if (sectorSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String sectorName : serviceReferential.getSectors(getFilter())) {
                options.add(new OptionModelImpl(sectorName,sectorName));
            }
            sectorSelectModel = new SelectModelImpl(null, options);
        }
        return sectorSelectModel;
    }

    /**
     * Get select model for sampleRows. Depends on filter (Sampling part only).
     * The sampleRow set from filter is temporarly extracted to avoid returning
     * only one element in the select.
     *
     * @return the GenericSelectModel<SampleRow> for sampleRows
     * @throws WaoException
     */
    public GenericSelectModel<SampleRow> getSampleRowSelectModel() throws WaoException {
        if (sampleRowSelectModel == null) {
            // WARN : no use of sampleRow filter
            SampleRow rowFiltered = getFilter().getSampleRow();
            getFilter().setSampleRow(null);
            List<SampleRow> sampleRows = serviceSampling.getSampleRowsByFilter(getFilter());
            getFilter().setSampleRow(rowFiltered);
            
            sampleRowSelectModel = new GenericSelectModel<SampleRow>(sampleRows, SampleRow.class,
                "code", "topiaId", propertyAccess);
        }
        return sampleRowSelectModel;
    }

    /**
     * Get select model for companies. Depends on
     * {@link #isAvailableDataForFiltersOnly()} () } to return only activated
     * companies. This select model is only used by admin users.
     *
     * @return the GenericSelectModel<Company> for companies
     * @throws WaoException
     */
    public GenericSelectModel<Company> getCompanySelectModel() throws WaoException {
        if (companySelectModel == null) {
            List<Company> companies = serviceUser.getCompanies(isAvailableDataForFiltersOnly());
            companySelectModel = new GenericSelectModel<Company>(companies,Company.class,
                    "name","topiaId",propertyAccess);
        }
        return companySelectModel;
    }

    /**
     * Get the select model for observers. Depends on
     * {@link #isAvailableDataForFiltersOnly() } to return only activated
     * companies.
     *
     * @return the GenericSelectModel<WaoUser> for observers
     * @throws WaoException
     */
    public GenericSelectModel<WaoUser> getObserverSelectModel() throws WaoException {
        if (observerSelectModel == null) {
            List<WaoUser> observers =  
                    serviceUser.getObservers(getFilter().getCompany(),
                        isAvailableDataForFiltersOnly());
            observerSelectModel = new GenericSelectModel<WaoUser>(observers,WaoUser.class,
                    "fullName","topiaId",propertyAccess);
        }
        return observerSelectModel;
    }

    /**
     * Get select model for sampleRow program. Depends on company set in
     * filter.
     *
     * @return the SelectModel for program names
     * @throws WaoException
     */
    public SelectModel getProgramSelectModel() throws WaoException {
        if (programSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String programName : serviceSampling.getPrograms(getFilter().getCompany())) {
                options.add(new OptionModelImpl(programName, programName));
            }
            programSelectModel = new SelectModelImpl(null, options);
        }
        return programSelectModel;
    }

    protected abstract SamplingFilter getFilter() throws WaoException;

    /**
     * Used to show only active company and observers.
     *
     * @return true if only active users will be showed, false otherwise
     */
    protected abstract boolean isAvailableDataForFiltersOnly();

    public String[] onProvideCompletionsFromBoatName(String input) throws WaoException {
        List<String> results = serviceBoat.getBoatNamesStartWith(input);
        return results.toArray(new String[results.size()]);
    }

    /**
     * Called when search submit button is pressed. This method is useful to
     * refresh all filters data from selects.
     *
     * @throws WaoException
     */
    public void onSelectedFromSearch() throws WaoException {
        if (log.isDebugEnabled()) {
            log.debug("REFRESH FILTERS DATA");
            log.debug("observerSelect : " + observerSelect);
        }
        SampleRow row = getSampleRowSelectModel().findObject(sampleRowId);
        getFilter().setSampleRow(row);
        if (user.isAdmin()) {
            Company company = getCompanySelectModel().findObject(companyId);
            getFilter().setCompany(company);
        } else {
            getFilter().setCompany(user.getCompany());
        }

        if (observerSelect) {
            WaoUser observer = getObserverSelectModel().findObject(observerId);
            if (log.isDebugEnabled()) {
                log.debug("Observer selected : " +
                        (observer != null ? observer.getFullName() : "null"));
            }
            getFilter().setObserver(observer);
        }
    }

    private boolean edited;

    /**
     * EVENT :: selected on refreshByCompany submit button. Reset selects for
     * company selected : programs, facades, sectors, sampleRows and observers.
     * This method set the state to edited, you can use
     * {@link AbstractFilteredPage#isEdited() } method to test it.
     *
     * @throws WaoException
     * @see AbstractFilteredPage#onSelectedFromRefreshByProgram()
     */
    public void onSelectedFromRefreshByCompany() throws WaoException {
        // Reset programs (the behavior is needed even if programSelect is false)
        getFilter().setProgramName(null);
        onSelectedFromRefreshByProgram();
        if (observerSelect) {
            // Reset observers
            getFilter().setObserver(null);
            observerSelectModel = null;
            getObserverSelectModel();
        }
        if (programSelect) {
            programSelectModel = null;
            getProgramSelectModel();
        }
        edited = true;
    }

    /**
     * EVENT :: selected on refreshByProgram submit button. Reset selects for
     * program selected : facades, sectors and sampleRows.
     * This method set the state to edited, you can use
     * {@link AbstractFilteredPage#isEdited() } method to test it.
     *
     * @throws WaoException
     * @see AbstractFilteredPage#onSelectedFromRefreshByFacade()
     */
    public void onSelectedFromRefreshByProgram() throws WaoException {
        // Reset facades
        getFilter().setFacadeName(null);
        onSelectedFromRefreshByFacade();
        resetFacadeSelect();
    }

    /**
     * EVENT :: selected on refreshByFacade submit button. Reset selects for
     * facade selected : sectors and sampleRows.
     * This method set the state to edited, you can use
     * {@link AbstractFilteredPage#isEdited() } method to test it.
     *
     * @throws WaoException
     * @see AbstractFilteredPage#onSelectedFromRefreshBySector()
     */
    public void onSelectedFromRefreshByFacade() throws WaoException {
        // Reset sectors
        getFilter().setSectorName(null);
        onSelectedFromRefreshBySector();
        resetSectorSelect();
    }

    /**
     * EVENT :: selected on refreshBySector submit button. After refreshing 
     * data selected, reset selects for sector selected : sampleRows.
     * This method set the state to edited, you can use
     * {@link AbstractFilteredPage#isEdited() } method to test it.
     *
     * @throws WaoException
     * @see AbstractFilteredPage#onSelectedFromSearch()
     */
    public void onSelectedFromRefreshBySector() throws WaoException {
        // Update selected data
        onSelectedFromSearch();
        // Reset sampleRows;
        getFilter().setSampleRow(null);
        resetSampleRowSelect();
        edited = true;
    }

    /**
     * Used to indicate that a refresh as been triggered.
     *
     * @return true if the page filter is in edited state.
     */
    protected boolean isEdited() {
        return edited;
    }
   
}
