package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIContext.java 673 2013-03-24 13:03:42Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiUIContext.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.TuttiIOUtil;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.RessourceClassLoader;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.service.ClosedPersistenceService;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.TuttiServiceContext;
import fr.ifremer.tutti.service.protocol.TuttiProtocolImportExportService;
import fr.ifremer.tutti.service.pupitri.TuttiPupitriImportExportService;
import fr.ifremer.tutti.service.referential.TuttiReferentialImportExportService;
import fr.ifremer.tutti.service.referential.TuttiReferentialSynchronizeService;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.util.TuttiErrorHelper;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.UIMessageNotifier;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.help.JAXXHelpBroker;
import jaxx.runtime.swing.help.JAXXHelpUIHandler;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.UserI18nInitializer;
import org.nuiton.widget.SwingSession;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

/**
 * UI application context.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiUIContext extends AbstractBean implements Closeable, UIMessageNotifier, JAXXHelpUIHandler {

    public static final String VALIDATION_CONTEXT_EDIT = "edit";

    public static final String VALIDATION_CONTEXT_VALIDATE = "validate";

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIContext.class);

    public static final String PROPERTY_PROGRAM_ID = "programId";

    public static final String PROPERTY_CRUISE_ID = "cruiseId";

    public static final String PROPERTY_PROTOCOL_ID = "protocolId";

    public static final String PROPERTY_SCREEN = "screen";

    public static final String PROPERTY_PROGRAM_FILLED = "programFilled";

    public static final String PROPERTY_CRUISE__FILLED = "cruiseFilled";

    public static final String PROPERTY_PROTOCOL_FILLED = "protocolFilled";

    public static final String PROPERTY_VALIDATION_CONTEXT = "validationContext";

    public static final String PROPERTY_BUSY = "busy";

    public static final String PROPERTY_HIDE_BODY = "hideBody";

    public static final String PROPERTY_LOCALE = "locale";

    public static final Set<String> PROPERTIES_TO_SAVE = Sets.newHashSet(
            PROPERTY_PROGRAM_ID,
            PROPERTY_CRUISE_ID,
            PROPERTY_PROTOCOL_ID,
            PROPERTY_LOCALE);

    public static final String PROPERTY_DB_EXIST = "dbExist";

    public static final String PROPERTY_DB_LOADED = "dbLoaded";

    /**
     * Application context (only one for all the application).
     *
     * @since 0.1
     */
    private static TuttiUIContext applicationContext;

    /**
     * Application global configuration.
     *
     * @since 0.1
     */
    protected final TuttiApplicationConfig config;

    /**
     * ClassLoader ressource.
     *
     * @since 0.3
     */
    protected final RessourceClassLoader resourceLoader;

    /**
     * Service context used by any service.
     *
     * @since 0.1
     */
    protected final TuttiServiceContext serviceContext;

    /**
     * Swing session used to save ui states.
     *
     * @since 0.1
     */
    protected final SwingSession swingSession;

    /**
     * Erro helper.
     *
     * @since 1.0
     */
    protected final TuttiErrorHelper errorHelper;

    /**
     * Id of last selected program (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String programId;

    /**
     * Id of last selected cruise (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String cruiseId;

    /**
     * Id of last selected protocol (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String protocolId;

    /**
     * Shared data context.
     *
     * @since 1.0.2
     */
    protected TuttiDataContext dataContext;

    /**
     * Tutti help broker.
     *
     * @since 1.1
     */
    protected TuttiHelpBroker helpBroker;

    /**
     * Current screen displayed in ui.
     *
     * @since 0.1
     */
    protected TuttiScreen screen;

    /**
     * Current locale used in application.
     *
     * @since 1.0.3
     */
    protected Locale locale;

    /**
     * Busy state ({@code true} when a blocking action is running).
     *
     * @since 1.0.3
     */
    protected boolean busy;

    /**
     * Flag to hide (or not) the body of application.
     *
     * @since 1.1
     */
    protected boolean hideBody;

    /**
     * Message notifiers.
     *
     * @since 0.3
     */
    protected final Set<UIMessageNotifier> messageNotifiers;

    /**
     * Validation context (used by fishingOperation screens).
     *
     * @since 0.3
     */
    private String validationContext;

    private MainUI mainUI;

    private TuttiActionUI actionUI;

    /**
     * Flag to know if there is an exsiting db.
     *
     * @since 1.0
     */
    private boolean dbExist;

    /**
     * Flag to know if there is a loaded db.
     *
     * @since 1.0
     */
    private boolean dbLoaded;

    private Properties helpMapping;

    public static TuttiUIContext newContext(TuttiApplicationConfig config) {
        Preconditions.checkNotNull(config);
        Preconditions.checkState(applicationContext == null,
                                 "Application context was already opened!");
        applicationContext = new TuttiUIContext(config);
        return applicationContext;
    }

    public static TuttiUIContext getApplicationContext() {
        return applicationContext;
    }

    public static TuttiErrorHelper getErrorHelper() {
        return applicationContext.errorHelper;
    }

    protected TuttiUIContext(TuttiApplicationConfig config) {
        this.config = config;
        this.resourceLoader = new RessourceClassLoader(Thread.currentThread().getContextClassLoader());
        this.serviceContext = new TuttiServiceContext(resourceLoader, config.getServiceConfig());
        this.swingSession = new SwingSession(getConfig().getUIConfigFile(), false);
        this.errorHelper = new TuttiErrorHelper(this);
        this.dataContext = new TuttiDataContext(this);
        UIMessageNotifier logMessageNotifier = new UIMessageNotifier() {

            @Override
            public void showInformationMessage(String message) {
                if (log.isInfoEnabled()) {
                    log.info(message);
                }
            }
        };
        this.messageNotifiers = Sets.newHashSet();
        addMessageNotifier(logMessageNotifier);
    }

    public PersistenceService getPersistenceService() {

        PersistenceService service;

        if (useRealPersistenceService()) {
            service = dataContext.service;
            if (service == null) {

                // use real service
                service = serviceContext.getService(PersistenceService.class);

                dataContext.open(service);
            }
        } else {
            service = serviceContext.getService(ClosedPersistenceService.class);
        }
        return service;
    }

    public DecoratorService getDecoratorService() {
        return serviceContext.getService(DecoratorService.class);
    }

    public TuttiReferentialSynchronizeService getTuttiReferentialSynchronizeService() {
        return serviceContext.getService(TuttiReferentialSynchronizeService.class);
    }

    public TuttiProtocolImportExportService getTuttiProtocolImportExportService() {
        return serviceContext.getService(TuttiProtocolImportExportService.class);
    }

    public TuttiPupitriImportExportService getTuttiPupitriImportExportService() {
        return serviceContext.getService(TuttiPupitriImportExportService.class);
    }

    public TuttiReferentialImportExportService getTuttiReferentialImportExportService() {
        return serviceContext.getService(TuttiReferentialImportExportService.class);
    }

    public boolean useRealPersistenceService() {
        return isDbExist() && isDbLoaded();
    }

    public PersistenceService reloadPersistenceService() {

        try {
            serviceContext.close();
        } catch (IOException e) {
            throw new TuttiTechnicalException("Could not close services", e);
        }
        dataContext.close();

        return getPersistenceService();
    }

    public TuttiApplicationConfig getConfig() {
        return config;
    }

    public SwingSession getSwingSession() {
        return swingSession;
    }

    public String getProgramId() {
        return programId;
    }

    public String getCruiseId() {
        return cruiseId;
    }

    public String getProtocolId() {
        return protocolId;
    }

    public boolean isCruiseFilled() {
        return isProgramFilled() && StringUtils.isNotBlank(cruiseId);
    }

    public boolean isProtocolFilled() {
        return StringUtils.isNotBlank(protocolId);
    }

    public boolean isProgramFilled() {
        return StringUtils.isNotBlank(programId);
    }

    public TuttiScreen getScreen() {
        return screen;
    }

    public boolean isDbExist() {
        return dbExist;
    }

    public void setDbExist(boolean dbExist) {
        this.dbExist = dbExist;
        firePropertyChange(PROPERTY_DB_EXIST, null, dbExist);
    }

    public boolean isBusy() {
        return busy;
    }

    public void setBusy(boolean busy) {
        this.busy = busy;
        firePropertyChange(PROPERTY_BUSY, null, busy);
    }

    public boolean isHideBody() {
        return hideBody;
    }

    public void setHideBody(boolean hideBody) {
        this.hideBody = hideBody;
        firePropertyChange(PROPERTY_HIDE_BODY, null, hideBody);
    }

    public Locale getLocale() {
        return locale;
    }

    public void setLocale(Locale locale) {
        this.locale = locale;
        firePropertyChange(PROPERTY_LOCALE, null, locale);
    }

    public boolean isDbLoaded() {
        return dbLoaded;
    }

    public void setDbLoaded(boolean dbLoaded) {
        this.dbLoaded = dbLoaded;
        firePropertyChange(PROPERTY_DB_LOADED, null, dbLoaded);
    }

    public void setProgramId(String programId) {
        boolean oldProgramFilled = isProgramFilled();
        boolean oldCruiseFilled = isCruiseFilled();

        this.programId = programId;

        // always propagate the change
        firePropertyChange(PROPERTY_PROGRAM_ID, -1, programId);
        firePropertyChange(PROPERTY_PROGRAM_FILLED, oldProgramFilled, isProgramFilled());
        firePropertyChange(PROPERTY_CRUISE__FILLED, oldCruiseFilled, isCruiseFilled());
    }

    public void setCruiseId(String cruiseId) {
        boolean oldValue = isCruiseFilled();

        this.cruiseId = cruiseId;

        // always propagate the change
        firePropertyChange(PROPERTY_CRUISE_ID, -1, cruiseId);
        firePropertyChange(PROPERTY_CRUISE__FILLED,
                           oldValue, isCruiseFilled());
    }

    public void setProtocolId(String protocolId) {
        boolean oldValue = isProtocolFilled();
        this.protocolId = protocolId;

        // always propagate the change
        firePropertyChange(PROPERTY_PROTOCOL_ID, -1, protocolId);
        firePropertyChange(PROPERTY_PROTOCOL_FILLED,
                           oldValue, isProtocolFilled());
    }

    public void setValidationContext(String validationContext) {
        Object oldValue = getValidationContext();
        this.validationContext = validationContext;
        firePropertyChange(PROPERTY_VALIDATION_CONTEXT, oldValue, validationContext);
    }

    public String getValidationContext() {
        return validationContext;
    }

    public void setScreen(TuttiScreen screen) {
        Object oldValue = getScreen();
        this.screen = screen;
        firePropertyChange(PROPERTY_SCREEN, oldValue, screen);
    }

    public TuttiDataContext getDataContext() {
        return dataContext;
    }

    public void init() {

        config.getServiceConfig().prepareDirectories();

        // use our special classLoader (which will read some files from resources from a configuration directory)
        Thread.currentThread().setContextClassLoader(getResourceLoader());

        // Use shutdownHook to close context on System.exit
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {

            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("Close context on shutdown");
                }
                close();
            }
        }));

        config.getServiceConfig().getPersistenceConfig().initConfig(getResourceLoader());

        //--------------------------------------------------------------------//
        // init i18n
        //--------------------------------------------------------------------//
        File i18nDirectory = config.getI18nDirectory();
        if (!config.isFullLaunchMode()) {

            i18nDirectory = new File(config.getDataDirectory(), "i18n");

            if (i18nDirectory.exists()) {
                // clean i18n cache
                TuttiIOUtil.cleanDirectory(
                        i18nDirectory,
                        "Could not delete i18n cache at " + i18nDirectory);
            }
        }

        TuttiIOUtil.forceMkdir(i18nDirectory,
                               "Could not create i18n at " + i18nDirectory);

        if (log.isDebugEnabled()) {
            log.debug("I18N directory: " + i18nDirectory);
        }

        Locale i18nLocale = config.getI18nLocale();

        if (log.isInfoEnabled()) {
            log.info(String.format("Starts i18n with locale [%s] at [%s]",
                                   i18nLocale, i18nDirectory));
        }
        I18n.init(new UserI18nInitializer(
                i18nDirectory, new DefaultI18nInitializer("tutti-i18n")),
                  i18nLocale);

        //--------------------------------------------------------------------//
        // init help
        //--------------------------------------------------------------------//

        File helpDirectory = config.getHelpDirectory();

        if (!config.isFullLaunchMode()) {

            if (!helpDirectory.exists()) {
                helpDirectory = new File(config.getDataDirectory(), "help");
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Help directory: " + helpDirectory);
        }
        TuttiIOUtil.forceMkdir(
                helpDirectory,
                "Could not create helpDirectory at " + helpDirectory);

        // load help mapping
        String mappingProperties =
                "/tutti-help-" + i18nLocale.getLanguage() + ".properties";
        try {

            InputStream resourceAsStream =
                    getClass().getResourceAsStream(mappingProperties);
            helpMapping = new Properties();
            helpMapping.load(resourceAsStream);

        } catch (Exception eee) {
            log.error("Failed to load help mapping file at '" +
                      mappingProperties + "'", eee);
        }
        if (log.isInfoEnabled()) {
            log.info(String.format("Starts help with locale at [%s]",
                                   helpDirectory));
        }

        //--------------------------------------------------------------------//
        // init action UI
        //--------------------------------------------------------------------//
        setActionUI(new TuttiActionUI(null, this));
    }

    public void open() {

        setLocale(config.getI18nLocale());

        if (programId == null) {

            // load it from config
            setProgramId(config.getProgramId());
        }

        if (cruiseId == null) {

            // load it from config
            setCruiseId(config.getCruiseId());
        }

        if (protocolId == null) {

            // load it from config
            setProtocolId(config.getProtocolId());
        }

        boolean dbExists =
                config.getServiceConfig().getPersistenceConfig().isDbExists();

        setDbExist(dbExists);

        if (!dbExists) {

            setProtocolId(null);
            setProgramId(null);
            setCruiseId(null);
            setDbLoaded(false);
        }

        // save back to config
        saveContextToConfig();

        // list when programId or campaingId change to save the configuration
        addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                if (PROPERTIES_TO_SAVE.contains(evt.getPropertyName())) {
                    saveContextToConfig();
                }
            }
        });
    }

    @Override
    public void close() {

        // Clear data references
        messageNotifiers.clear();

        programId = null;
        cruiseId = null;
        protocolId = null;
        validationContext = null;
        IOUtils.closeQuietly(dataContext);

        setScreen(null);

        IOUtils.closeQuietly(serviceContext);

        // remove listeners
        PropertyChangeListener[] listeners = getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            if (log.isDebugEnabled()) {
                log.debug("Remove listener: " + listener);
            }
            removePropertyChangeListener(listener);
        }
        setMainUI(null);
        if (actionUI != null) {

            // close action ui
            actionUI.getModel().clear();
        }
        setActionUI(null);
    }

    protected void saveContextToConfig() {
        if (log.isInfoEnabled()) {
            log.info("Save config (programId: " + programId + ", cruiseId: " +
                     cruiseId + ", protocolId: " + protocolId + ", locale: " +
                     locale + ")");
        }
        config.setProgramId(programId);
        config.setCruiseId(cruiseId);
        config.setProtocolId(protocolId);
        config.setI18nLocale(locale);
        config.save();
    }

    public void addMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.add(messageNotifier);
    }

    public void removeMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.remove(messageNotifier);
    }

    @Override
    public void showInformationMessage(String message) {
        for (UIMessageNotifier messageNotifier : messageNotifiers) {
            messageNotifier.showInformationMessage(message);
        }
    }

    public RessourceClassLoader getResourceLoader() {
        return resourceLoader;
    }

    public void setMainUI(MainUI mainUI) {
        this.mainUI = mainUI;
    }

    public MainUI getMainUI() {
        return mainUI;
    }

    public void setActionUI(TuttiActionUI actionUI) {
        this.actionUI = actionUI;
    }

    public TuttiActionUI getActionUI() {
        return actionUI;
    }

    public void clearDbContext() {
        protocolId = null;
        programId = null;
        cruiseId = null;
        saveContextToConfig();
    }

    public void checkDbContext() {

        //check if programId is sane
        PersistenceService service = getPersistenceService();

        if (isProtocolFilled()) {

            if (!service.isProtocolExist(protocolId)) {

                // not found in this db

                if (log.isWarnEnabled()) {
                    log.warn("Remove invalid protocolId: " + protocolId);
                }

                setProtocolId(null);
            }
        }

        if (isProgramFilled()) {

            Program program = service.getProgram(programId);
            if (program == null) {

                // not found in this db

                if (log.isWarnEnabled()) {
                    log.warn("Remove invalid programId: " + programId);
                }

                setProgramId(null);
                setCruiseId(null);

            } else {

                if (log.isInfoEnabled()) {
                    log.info("ProgramId valid: " + programId);
                }

                setProgramId(programId);

                // test cruiseId
                if (isCruiseFilled()) {

                    Cruise cruise = service.getCruise(cruiseId);

                    if (cruise != null &&
                        !cruise.getProgram().getId().equals(programId)) {

                        // not matchin program, reset cruise id
                        cruise = null;
                    }

                    if (cruise == null) {

                        // not found in this db

                        if (log.isWarnEnabled()) {
                            log.warn("Remove invalid cruiseId: " + cruiseId);
                        }
                        setCruiseId(null);

                    } else {

                        if (log.isInfoEnabled()) {
                            log.info("CruiseId valid: " + cruiseId);
                        }
                    }
                }
            }
        }

        saveContextToConfig();
    }

    public TuttiHelpBroker getHelpBroker() {
        return helpBroker;
    }

    public void setHelpBroker(TuttiHelpBroker helpBroker) {
        this.helpBroker = helpBroker;
    }

    @Override
    public void showHelp(JAXXContext context,
                         JAXXHelpBroker broker,
                         String helpId) {
        if (helpId == null) {
            helpId = broker.getDefaultID();
        }

        if (log.isInfoEnabled()) {
            log.info("show help " + helpId);
        }

        String value = (String) helpMapping.get(helpId);

        if (value == null) {
            throw new TuttiTechnicalException(
                    "Could not find help page for " + helpId);
        }

        File helpDirectory = getConfig().getHelpDirectoryWithLocale();
        URI resolvedUri = helpDirectory.toURI().resolve(value);
        TuttiUIUtil.openLink(resolvedUri);
    }

    public void reloadDecoratorService() {
        serviceContext.reloadService(DecoratorService.class);
    }
}
