
package fr.ifremer.tutti.ui.swing.content.operation.catches;

/*
 * #%L
 * Tutti :: UI
 * $Id: ComputeWeightsAction.java 450 2013-02-20 13:03:56Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/ComputeWeightsAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SampleCategory;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import jaxx.runtime.JAXXUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JOptionPane;
import javax.swing.JTextField;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0
 */
public class ComputeWeightsAction extends AbstractTuttiAction<EditCatchesUIModel, EditCatchesUI, EditCatchesUIHandler> {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(ComputeWeightsAction.class);

    protected String errorMessage;

    protected String errorTitle;

    protected Component errorComponent;

    public ComputeWeightsAction(EditCatchesUIHandler handler) {
        super(handler,
              "computeWeights",
              "generate",
              _("tutti.action.computeWeights"),
              _("tutti.action.computeWeights.tip"),
              true);
    }

    @Override
    protected void doAction(ActionEvent event) throws Exception {
        EditCatchesUIModel model = getModel();
        
        computeSpeciesBatches();
        Float speciesTotalSortedWeight = model.getSpeciesTotalSortedComputedWeight();
        Float speciesTotalUnsortedWeight = model.getSpeciesTotalUnsortedComputedWeight();
        
        EditCatchesUI ui = getUI();

        model.setCatchTotalSortedComputedWeight(speciesTotalSortedWeight);
        model.setCatchTotalUnsortedComputedWeight(speciesTotalUnsortedWeight);
        
        Float totalWeight = model.getCatchTotalWeight();
        Float rejectedWeight = model.getCatchTotalRejectedWeight();

        if (rejectedWeight == null && totalWeight != null) {
            if (!totalWeight.equals(speciesTotalUnsortedWeight
                                    + speciesTotalSortedWeight)) {
                errorMessage = _("tutti.action.computeWeights.error.incoherentTotal");
                errorTitle = _("tutti.action.computeWeights.error.incoherentTotal.title");
                errorComponent = ui.getCatchTotalWeightField();

            } else {

                model.setCatchTotalRejectedComputedWeight(totalWeight
                                                          - speciesTotalUnsortedWeight
                                                          - speciesTotalSortedWeight);
            }

        } else if (totalWeight == null) {
            if (rejectedWeight == null) {
                rejectedWeight = 0f;
                model.setCatchTotalRejectedComputedWeight(0f);
            }
            model.setCatchTotalComputedWeight(speciesTotalUnsortedWeight
                                              + speciesTotalSortedWeight 
                                              + rejectedWeight);

        } else if (rejectedWeight != null
                   && !totalWeight.equals(speciesTotalUnsortedWeight
                                          + speciesTotalSortedWeight 
                                          + rejectedWeight)) {
            errorMessage = _("tutti.action.computeWeights.error.incoherentTotal");
            errorTitle = _("tutti.action.computeWeights.error.incoherentTotal.title");
            errorComponent = ui.getCatchTotalWeightField();
        }

    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSpeciesBatches() {
        PersistenceService persistenceService =
                getService(PersistenceService.class);
        EditCatchesUIModel model = getModel();

        Float totalSortedWeight = 0f;
        Float totalUnsortedWeight = 0f;
        SpeciesBatchUI speciesUI = getUI().getSpeciesTabContent();
        EditCatchesUI ui = getUI();

        List<SpeciesBatchRowModel> roots = getUI().getSpeciesTabContent().getModel().getRows();
        for (SpeciesBatchRowModel row : roots) {
            if (row.isBatchRoot()) {
                Float weight = computeSpeciesBatch(row);
                if (weight == null) {
                    JOptionPane.showMessageDialog(
                            speciesUI,
                            _("tutti.dialog.catches.species.computeWeight.error.message"),
                            _("tutti.dialog.catches.species.computeWeight.error.title"),
                            JOptionPane.ERROR_MESSAGE);
                    totalSortedWeight = null;
                    totalUnsortedWeight = null;
                    break;
                }
                if (persistenceService.isSortedQualitativeValue(row.getSortedUnsortedCategory().getCategoryValue())) {
                    totalSortedWeight += weight;
                } else {
                    totalUnsortedWeight += weight;
                }
            }
        }
        
        Number inertWeight = model.getSpeciesTotalInertWeight();
        if (inertWeight != null) {
            totalSortedWeight += inertWeight.floatValue();
        } else {
            model.setSpeciesTotalInertWeight(0f);
        }
        
        Number livingNotItemizedWeight = model.getSpeciesTotalLivingNotItemizedWeight();
        if (livingNotItemizedWeight != null) {
            totalSortedWeight += livingNotItemizedWeight.floatValue();
        } else {
            model.setSpeciesTotalLivingNotItemizedWeight(0f);
        }

        model.setSpeciesTotalSampleSortedComputedWeight(totalSortedWeight);

        Float speciesTotalSortedWeight = model.getSpeciesTotalSortedWeight();
        if (speciesTotalSortedWeight == null) {
            speciesTotalSortedWeight = totalSortedWeight;
            model.setSpeciesTotalSortedComputedWeight(totalSortedWeight);

        } else if (speciesTotalSortedWeight < totalSortedWeight) {
            errorMessage = _("tutti.action.computeWeights.error.incoherentSpeciesTotalSorted");
            errorTitle = _("tutti.action.computeWeights.error.incoherentSpeciesTotalSorted.title");
            errorComponent = ui.getSpeciesTotalSortedWeightField();

        } else if (speciesTotalSortedWeight < 1.05 * totalSortedWeight) {
            // TODO Si  le "Poids total VRAC" est saisi est que sa valeur 
            // est supérieure de moins de x% (x en configuration) 
            // du "Poids total Vrac trié", demander confirmation que 
            // le "Poids total VRAC" est bien une valeur observée 
            // sinon la remplacer par le "Poids total Vrac trié"
        }

        model.setSpeciesTotalUnsortedComputedWeight(totalUnsortedWeight);

        Float totalWeight = totalUnsortedWeight + speciesTotalSortedWeight;
        model.setSpeciesTotalComputedWeight(totalWeight);

        speciesUI.getTable().repaint();
    }

    protected Float computeSpeciesBatch(SpeciesBatchRowModel row) {
        SampleCategory finestCategory = row.getFinestCategory();

        Float result = null;
        Float categoryWeight = finestCategory.getCategoryWeight();
        Float rowWeight = row.getWeight();

        List<SpeciesBatchRowModel> children = row.getChildBatch();
        // if the row is not a leaf
        if (!row.isBatchLeaf()) {
            Float sum = 0f;
            // make the sum of the children weights
            for (SpeciesBatchRowModel child : children) {
                Float weight = computeSpeciesBatch(child);
                if (weight == null) {
                    sum = null;
                    break;
                }
                sum += weight;
            }

            if (sum != null) {
                if (categoryWeight == null) {
                    finestCategory.setComputedWeight(sum);
                    for (SpeciesBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(false);
                    }

                } else if (categoryWeight < sum) {
                    errorMessage = _("tutti.action.computeWeights.error.incoherentParentCategoryWeight");
                    errorTitle = _("tutti.action.computeWeights.error.incoherentParentCategoryWeight.title");
                    errorComponent = null;

                } else {
                    boolean subSample = categoryWeight > sum;
                    for (SpeciesBatchRowModel child : children) {
                        child.getFinestCategory().setSubSample(subSample);
                    }
                }
                result = sum;
            }

        } else {// the row is a leaf
            row.setComputedWeight(null);

            List<SpeciesFrequencyRowModel> frequency = row.getFrequency();

            if (CollectionUtils.isNotEmpty(frequency)) {
                // if there are frequencies, then compute their weight
                Float frequencyWeight = 0f;
                for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                    Float w = frequencyModel.getWeight();
                    if (w == null) {

                        // can't sum when a null value appears
                        frequencyWeight = null;
                        break;

                    } else if (frequencyWeight != null) {

                        // still can sum weights
                        frequencyWeight += w;
                    }
                }

                if (categoryWeight == null && rowWeight != null) {
                    errorMessage = _("tutti.action.computeWeights.error.incoherentRowWeightCategory");
                    errorTitle = _("tutti.action.computeWeights.error.incoherentRowWeightCategory.title");
                    errorComponent = null;

                } else if (categoryWeight == null && frequencyWeight != null) {
                    // if the category weight is null and the frequencies have a weight, 
                    // then this weight is the result
                    finestCategory.setComputedWeight(frequencyWeight);
                    result = frequencyWeight;

                } else if (frequencyWeight != null
                           && !frequencyWeight.equals(categoryWeight)) {

                    // if the weight of the frequencies is different from the category
                    // weight, then set the weight of the sample
                    if (categoryWeight != null && frequencyWeight > categoryWeight) {
                        errorMessage = _("tutti.action.computeWeights.error.incoherentCategoryWeight");
                        errorTitle = _("tutti.action.computeWeights.error.incoherentCategoryWeight.title");
                        errorComponent = null;

                    } else if (rowWeight == null) {
                        row.setComputedWeight(frequencyWeight);

                    } else if (!rowWeight.equals(frequencyWeight)) {
                        errorMessage = _("tutti.action.computeWeights.error.incoherentRowWeightFrequency");
                        errorTitle = _("tutti.action.computeWeights.error.incoherentRowWeightFrequency.title");
                        errorComponent = null;
                    }
                    result = categoryWeight;

                } else {
                    result = categoryWeight;
                }

            } else {
                result = categoryWeight;
            }
        }

        return result;
    }

    @Override
    protected void releaseAction(ActionEvent event) {
        if (errorMessage != null) {
            JOptionPane.showMessageDialog(
                    getUI(),
                    errorMessage,
                    errorTitle,
                    JOptionPane.ERROR_MESSAGE);
        }
        if (errorComponent != null) {
            errorComponent.requestFocus();
        }
        errorMessage = null;
        errorComponent = null;
    }

}
