/*
 * #%L
 * SGQ :: Business
 * $Id: ProductionService.java 442 2013-11-13 10:21:18Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.4/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/ProductionService.java $
 * %%
 * Copyright (C) 2012, 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;

import au.com.bytecode.opencsv.CSVReader;
import au.com.bytecode.opencsv.CSVWriter;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.business.model.ImportLog;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.Expedition;
import com.herbocailleau.sgq.entities.ExpeditionDAO;
import com.herbocailleau.sgq.entities.LabelError;
import com.herbocailleau.sgq.entities.LabelErrorDAO;
import com.herbocailleau.sgq.entities.Presentation;
import com.herbocailleau.sgq.entities.PresentationCode;
import com.herbocailleau.sgq.entities.PresentationDAO;
import com.herbocailleau.sgq.entities.Production;
import com.herbocailleau.sgq.entities.ProductionDAO;
import com.herbocailleau.sgq.entities.Zone;

/**
 * Service gerant principalement:
 * 
 * <ul>
 * <li>Les ventes et transformations des présentations des lots
 * <li>les gestions des erreurs qui se produise lors des imports
 * </ul>
 * 
 * @author echatellier
 */
public class ProductionService extends SgqService {

    private static final Log log = LogFactory.getLog(ProductionService.class);

    /** Séparateur CSV pour la sauvegarde interne (replay). */
    protected static final char ERROR_REPLAY_SEPARATOR = ',';

    /** Date format du fichier d'entrée étiquette. */
    protected final DateFormat LABEL_DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");

    /**
     * Retourne la configuration pour obtenir les dates de dernière
     * mise à jour (import expedition/fabrication).
     * 
     * @return la configuration (can be null on first import)
     */
    public Date getLastImportDate(Zone zone) {
        Date result = null;
        try {
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();
            result = expeditionDAO.findMaxDateForZone(zone);

            Date result2 = productionDAO.findMaxDateForZone(zone);
            if (result2 != null && (result == null || result.before(result2))) {
                result = result2;
            }

        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get configuration", ex);
        }
        return result;
    }

    /**
     * Import label file.
     * Use file name to import production file or expedition file.
     * 
     * Le fichier est structuré ainsi:
     * D,20110701,B,147136,C,11520,P,PFEN21,L,20036,Q,40.000
     * 
     * D : Date 
     * B : N° de Bon de commande interne
     * C : Code client
     * P : Code présentation + Code plante
     * L : N° de lot
     * Q : Quantité
     * R : Remplacement lot
     * 
     * La methode fait un grand taille, mais elle n'est pas evident à diviser.
     * 
     * Champs a renseigner :
     * <ul>
     * <li>sur le lot: DMESD = Date de Mise En Service Détaillant : date de la première vente
     *    du lot quelques soit sa présentation mais au niveau de l'atelier des expéditions.
     * <li>sur la presentation : une plante est présente en ZC s'il y a un
     *   mouvement pour le client 99997 : valorisation de packaging à 'true'
     * </ul>
     * 
     * @param fileFilename file name (FIF_HIST.txt or FIC_HIST.txt allowed)
     * @param file file
     */
    public List<ImportLog> importLabelsFile(String fileFilename, File file) {

        if (log.isInfoEnabled()) {
            log.info("Importing labels file : " + file);
        }

        // test filename to get correct last import date
        // if file in not FIF_HIST.txt or FIC_HIST.txt
        // an exception is thrown
        String productionFileName = config.getLabelProductionFilename();
        Zone sourceZone = null;
        Date lastImport = null;
        if (productionFileName.equalsIgnoreCase(fileFilename)) {
            sourceZone = Zone.ZE;
            lastImport = getLastImportDate(Zone.ZE);
        } else {
            String expeditionFileName = config.getLabelExpeditionFilename();
            if (expeditionFileName.equalsIgnoreCase(fileFilename)) {
                sourceZone = Zone.ZP;
                lastImport = getLastImportDate(Zone.ZP);
            } else {
                throw new SgqBusinessException(_("Nom de fichier '%s' invalide ! (Autorisé %s ou %s)",
                        fileFilename, productionFileName, expeditionFileName));
            }
        }

        // cas du premier import
        Date importFrom = lastImport;
        if (importFrom == null) {
            importFrom = new Date(0); // pour que l'algo fonctionne
        }
        // date du dernier import + 1 à minuit
        importFrom = DateUtils.addDays(importFrom, 1);
        importFrom = DateUtils.truncate(importFrom, Calendar.DAY_OF_MONTH);

        // date du jour a minuit
        Date today = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH);

        // performing import
        List<ImportLog> importLogs = new ArrayList<ImportLog>();
        CSVReader reader = null;
        try {

            reader = new CSVReader(new FileReader(file));

            // get date format
            DateFormat dfOut = new SimpleDateFormat("dd/MM/yyyy");

            // get dao
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();

            // pour le premier import, on charge toutes les présentations
            // de tous les lots de toutes la base sinon c'est vraiment trop long
            Map<Batch, List<Presentation>> batchPresentations = new HashMap<Batch, List<Presentation>>();
            Map<Integer, Batch> batchCache = new HashMap<Integer, Batch>();
            if (lastImport == null) { // si c'est null, c'est le premier import
                if (log.isDebugEnabled()) {
                    log.debug("Populating cache for first import");
                }
                List<Presentation> allPresentations = presentationDAO.findAll();
                for (Presentation presentation : allPresentations) {
                    Batch batch = presentation.getBatch();
                    batchCache.put(batch.getNumber(), batch);
                    List<Presentation> batchPresentation = batchPresentations.get(batch);
                    if (batchPresentation == null) {
                        batchPresentation = new ArrayList<Presentation>();
                        batchPresentations.put(batch, batchPresentation);
                    }
                    batchPresentation.add(presentation);
                }
            }

            int line = 0;
            String[] csvdata = null;
            while ((csvdata = reader.readNext()) != null) {
                if (csvdata.length <= 1) {
                    continue;
                }
                line++;

                // ajout d'un check pour verifier que le fichier a la forme
                // attendue, notement au niveau des prefix de colonnes
                String columns = "";
                for (int i = 0; i < csvdata.length ; i+=2) {
                    columns += csvdata[i];
                }
                if (!"DBCPLQ".equals(columns) && !"DBCPLQR".equals(columns)) {
                    throw new SgqBusinessException(_("Nom de colonne invalides; trouvé : %s, attendu : %s ou %s",
                            columns, "DBCPLQ", "DBCPLQR"));
                }

                String dateyyyyMMdd = csvdata[1];
                String clientId = csvdata[5];
                String productId = csvdata[7];
                String batchId = csvdata[9];
                String quantityStr = csvdata[11];
                String batchCorrectionId = null;
                if (csvdata.length >= 14) {
                    batchCorrectionId = csvdata[13];
                }

                Date lineDate = LABEL_DATE_FORMAT.parse(dateyyyyMMdd);

                // controle des dates, import entre le last import
                // et la veille de la date du jour
                if (lineDate.before(importFrom)) {
                    if (log.isDebugEnabled()) {
                        log.debug(_("Skipping date %s, before %s", lineDate.toString(),
                                importFrom.toString()));
                    }
                    continue;
                }

                // les lignes suivantes donne lieu a une entrée de log
                ImportLog importLog = new ImportLog();
                importLog.setLine(line);
                importLog.setCode(dfOut.format(lineDate));
                importLogs.add(importLog);

                // check de la date du jour
                if (!lineDate.before(today)) { // pas le meme test que date.after !!!
                    if (log.isDebugEnabled()) {
                        log.debug(_("Skipping date of current day %s", lineDate.toString()));
                    }
                    importLog.setError(true);
                    importLog.setMessage(_("Ligne ignorée (date du jour)"));
                    continue;
                }

                // get source batch (quantity is deducted from that batch)
                int sourceBatchNumber = Integer.parseInt(batchId);
                Batch sourceBatch = batchCache.get(sourceBatchNumber);
                if (sourceBatch == null) {
                    sourceBatch = batchDAO.findByNumber(sourceBatchNumber);
                }
                if (sourceBatch == null) {
                    importLog.setError(true);
                    importLog.setMessage(_("Impossible de trouver le lot %s", batchId));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }
                batchCache.put(sourceBatchNumber, sourceBatch);

                // get correction batch (if batchCorrectionId is not empty)
                Batch correctionBatch = null;
                if (StringUtils.isNotBlank(batchCorrectionId)) {
                    int destBatchNumber = Integer.parseInt(batchCorrectionId);
                    correctionBatch = batchCache.get(destBatchNumber);
                    if (correctionBatch == null) {
                        correctionBatch = batchDAO.findByNumber(destBatchNumber);
                    }
                    if (correctionBatch == null) {
                        importLog.setError(true);
                        importLog.setMessage(_("Impossible de trouver le lot %s", batchCorrectionId));
                        labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                                LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                                LabelError.PROPERTY_SOURCE, sourceZone);
                        continue;
                    }
                    batchCache.put(destBatchNumber, correctionBatch);

                    // lot corrigé = lot source
                    if (sourceBatch.equals(correctionBatch)) {
                        importLog.setError(true);
                        importLog.setMessage(_("Correction sur un lot identique au lot corrigé :%d", batchCorrectionId));
                        labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                                LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                                LabelError.PROPERTY_SOURCE, sourceZone);
                        continue;
                    }
                }

                // lot ignoré
                if (sourceBatch.isInvalid()) {
                    importLog.setError(true);
                    importLog.setMessage(_("Le lot %d a été marqué comme ignoré. Impossible d'enregistrer une étiquette sur ce lot.", sourceBatch.getNumber()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }
                // lot ignoré
                if (correctionBatch != null && correctionBatch.isInvalid()) {
                    importLog.setError(true);
                    importLog.setMessage(_("Le lot %d a été marqué comme ignoré. Impossible d'enregistrer une étiquette sur ce lot.", correctionBatch.getNumber()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }

                // etiquette entrée après épuisement du lot
                if (sourceBatch.getExpiredDate() != null && lineDate.after(sourceBatch.getExpiredDate())) {
                    importLog.setError(true);
                    importLog.setMessage(_("La date de l'étiquette (%1$td/%1$tm/%1$tY) porte après la date d'expiration du lot %2$d (%3$td/%3$tm/%3$tY).",
                            lineDate, sourceBatch.getNumber(), sourceBatch.getExpiredDate()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone,
                            LabelError.PROPERTY_AFTER_BATCH_EXPIRATION, sourceBatch);
                    continue;
                }
                // etiquette entrée après épuisement du lot
                if (correctionBatch != null && correctionBatch.getExpiredDate() != null && lineDate.after(correctionBatch.getExpiredDate())) {
                    importLog.setError(true);
                    importLog.setMessage(_("La date de l'étiquette (%1$td/%1$tm/%1$tY) porte après la date d'expiration du lot %2$d (%3$td/%3$tm/%3$tY).",
                            lineDate, correctionBatch.getNumber(), correctionBatch.getExpiredDate()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone,
                            LabelError.PROPERTY_AFTER_BATCH_EXPIRATION, correctionBatch);
                    continue;
                }
                
                // test presentation
                PresentationCode presentationCode = getPresentationCode(productId);
                if (presentationCode == null) {
                    importLog.setError(true);
                    importLog.setMessage(_("Lot %d : Code de présentation inconnu %s",
                            sourceBatch.getNumber(), productId));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }

                // gestion de l'equivalence pour les code de presentation utilisé par la facturation
                if (presentationCode.getEquivalence() != null) {
                    presentationCode = presentationCode.getEquivalence();
                }

                double quantity = Double.parseDouble(quantityStr);
                int client = Integer.parseInt(clientId);

                // Les lignes ayant un code ayant une structure du type F0XXXX devront être importées sous la forme FXXXX
                // Les lignes ayant un code ayant une structure du type T0XXXX devront être importées sous la forme TXXXX
                if ((presentationCode == PresentationCode.F || presentationCode == PresentationCode.T)
                        && productId.charAt(1) == '0') {
                    // suppression du premier 0
                    // mais le code pres doit rester dans le code produit dans ce cas
                    productId = presentationCode.getCode() + productId.substring(2);
                } else {
                    productId = productId.substring(1);
                }

                // check du code produit qui doit correspondre au code
                // produit sur lequel le lot porte
                if (!checkBatchProduct(sourceBatch, productId, lineDate)) {
                    if (log.isErrorEnabled()) {
                        log.error(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                                sourceBatch.getNumber(), productId, sourceBatch.getProduct().getCode()));
                    }
                    importLog.setError(true);
                    importLog.setMessage(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                            sourceBatch.getNumber(), productId, sourceBatch.getProduct().getCode()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }
                // check du code produit qui doit correspondre au code
                // produit sur lequel le lot porte
                if (correctionBatch != null && !checkBatchProduct(correctionBatch, productId, lineDate)) {
                    if (log.isErrorEnabled()) {
                        log.error(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                                correctionBatch.getNumber(), productId, correctionBatch.getProduct().getCode()));
                    }
                    importLog.setError(true);
                    importLog.setMessage(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                            correctionBatch.getNumber(), productId, correctionBatch.getProduct().getCode()));
                    labelErrorDAO.create(LabelError.PROPERTY_LINE, StringUtils.join(csvdata, ERROR_REPLAY_SEPARATOR),
                            LabelError.PROPERTY_MESSAGE, importLog.getMessage(),
                            LabelError.PROPERTY_SOURCE, sourceZone);
                    continue;
                }

                // recherche de la presentation du lot source affectée par la production
                // (presentation d'origine) et/ou la vente (presentation dest)
                List<Presentation> sourcePresentations = batchPresentations.get(sourceBatch);
                if (sourcePresentations == null) {
                    sourcePresentations = presentationDAO.findAllByBatch(sourceBatch);
                    batchPresentations.put(sourceBatch, sourcePresentations);
                }
                Presentation srcPresentation = null;
                Presentation srcOriginPresentation = null;
                for (Presentation presentation : sourcePresentations) {
                    // la quantite produite est toujours déduite de la
                    // presentation .
                    if (presentation.getPresentationCode() == presentationCode) {
                        srcPresentation = presentation;
                    }
                    if (presentation.isOriginal()) {
                        srcOriginPresentation = presentation;
                    }
                }

                // recherche de la présentation du lot de correction
                Presentation corrPresentation = null;
                Presentation corrOriginPresentation = null;
                if (correctionBatch != null) {
                    List<Presentation> destinationPresentations = batchPresentations.get(correctionBatch);
                    if (destinationPresentations == null) {
                        destinationPresentations = presentationDAO.findAllByBatch(correctionBatch);
                        batchPresentations.put(correctionBatch, destinationPresentations);
                    }
                    
                    for (Presentation presentation : destinationPresentations) {
                        // la quantite produite est ajoutée a la presentation
                        // ayant le même code que celui de l'etiquette
                        if (presentation.getPresentationCode() == presentationCode) {
                            corrPresentation = presentation;
                        }
                        if (presentation.isOriginal()) {
                            corrOriginPresentation = presentation;
                        }
                    }
                }

                // 99998 et 99999 = production
                if (client == Zone.ZE.getClient() || client == Zone.ZP.getClient()) {

                    // dans le cas ou la presentation produite n'existe pas,
                    // on doit en creer une nouvelle
                    if (srcPresentation == null) {
                        srcPresentation = presentationDAO.createByNaturalId(sourceBatch, presentationCode);

                        importLog.setMessage(_("Nouvelle presentation %c pour le lot %s",
                                presentationCode.getCode(), batchId));

                        // cache update
                        batchPresentations.get(sourceBatch).add(srcPresentation);
                    } else {
                        importLog.setMessage(_("Mise à jour de la presentation %c du lot %s",
                                presentationCode.getCode(), batchId));
                    }

                    // pour FIF_HIST (de ZE vers ZE c'est une production)
                    // dans tous les autres cas, il n'y a production que
                    // si la quantité de destination n'est pas suffisante
                    double productionQuantity;
                    if (sourceZone == Zone.ZE && client == Zone.ZE.getClient() || srcPresentation.getQuantity() <= 0) {
                        productionQuantity = quantity;
                    } else {
                        if (srcPresentation.getQuantity() < quantity) {
                            productionQuantity = quantity - srcPresentation.getQuantity();
                        } else {
                            productionQuantity = 0;  // transfert
                        }
                    }

                    // si production, reduction du stock (original)
                    if (srcOriginPresentation == null) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find original presentation for row " + Arrays.toString(csvdata));
                        }
                    } else {

                        // Il se peut même que la transformation ne s'applique
                        // pas a la presentation d'origine, mais on ne peut
                        // pas le savoir et on diminue quand meme le stock de
                        // la presentation d'origine. Il peut donc en resulter un
                        // stock négatif
                        double origPresQuantity = srcOriginPresentation.getQuantity();
                        origPresQuantity -= productionQuantity;
                        if (sourceBatch.getExpiredDate() == null) {
                            srcOriginPresentation.setQuantity(origPresQuantity);
                        }
                        presentationDAO.update(srcOriginPresentation);
                    }

                    // si production, augmentation du stock (produit)
                    double presQuantity = srcPresentation.getQuantity();
                    presQuantity += productionQuantity;
                    if (sourceBatch.getExpiredDate() == null) {
                        srcPresentation.setQuantity(presQuantity);
                    }
                    presentationDAO.update(srcPresentation);

                    // eventuelle correction
                    if (correctionBatch != null && correctionBatch.getExpiredDate() == null) {
                        // gestion des cas incoherents
                        if (corrOriginPresentation == null) {
                            corrOriginPresentation = presentationDAO.createByNaturalId(correctionBatch, PresentationCode._);
                        }
                        if (corrPresentation == null) {
                            corrPresentation = presentationDAO.createByNaturalId(correctionBatch, presentationCode);
                        }

                        // détermination de la quantité réelle suivant les cas
                        double oldProductionQuantity;
                        if (sourceZone == Zone.ZE && client == Zone.ZE.getClient()) {
                            oldProductionQuantity = quantity;
                        } else {
                            // pour les autres cas, si la quantite produite est
                            // exactement la quantité existante, on annule la
                            // production, sinon on suppose que la presentation
                            // existait deja avant et on ne fait rien
                            if (corrPresentation.getQuantity() == quantity) {
                                oldProductionQuantity = quantity;
                            } else {
                                oldProductionQuantity = 0;  // transfert
                            }
                        }

                        // reajout du stock sur la presentation d'origine
                        double corrOriginQuantity = corrOriginPresentation.getQuantity();
                        corrOriginQuantity += oldProductionQuantity;
                        corrOriginPresentation.setQuantity(corrOriginQuantity);
                        presentationDAO.update(corrOriginPresentation);

                        // retrait du stock precedement ajouté sur la presentation
                        double corrQuantity = corrPresentation.getQuantity();
                        corrQuantity -= oldProductionQuantity;
                        corrPresentation.setQuantity(corrQuantity);
                        presentationDAO.update(corrPresentation);
                    }

                    // sauvegarde des productions
                    Production production = productionDAO.create();
                    production.setDate(lineDate);
                    production.setQuantity(quantity);
                    production.setPresentation(srcPresentation);
                    production.setCorrection(corrPresentation);
                    production.setSource(sourceZone);
                    if (client == Zone.ZE.getClient()) {
                        // Toutes étiquettes ayant un code client interne 99999 alimentent la zone ZE
                        production.setDestination(Zone.ZE);
                    } else {
                        // Toutes étiquettes ayant un code client interne 99998 alimentent la zone ZP
                        production.setDestination(Zone.ZP);
                    }
                    productionDAO.update(production);

                } else {

                    // il peut se produire des cas où une presentation
                    // est expédié directement à un client sans qu'elle
                    // ai été enregistrée en production (manque de tracabilité)
                    // on ne peut donc pas diminuer le stock dans ce cas
                    // mais l'expedition peut-être enregistrer
                    if (srcPresentation == null) {
                        if (log.isWarnEnabled()) {
                            log.warn(dateyyyyMMdd + " : Can't find expedition presentation for batch " + sourceBatch.getNumber() + ". Create it !");
                        }
                        srcPresentation = presentationDAO.createByNaturalId(sourceBatch, presentationCode);

                        // cache update
                        batchPresentations.get(sourceBatch).add(srcPresentation);
                    }

                    double expeditionQuantity = 0;
                    if (srcPresentation.getQuantity() < quantity) {
                        if (srcPresentation.getQuantity() > 0) {
                            expeditionQuantity = srcPresentation.getQuantity();
                        }

                        // cas ou une quantite supplémentaire
                        double productionQuantity = quantity - expeditionQuantity;
                        double presQuantity = srcOriginPresentation.getQuantity();
                        presQuantity -= productionQuantity;
                        if (sourceBatch.getExpiredDate() == null) {
                            srcOriginPresentation.setQuantity(presQuantity);
                        }
                        presentationDAO.update(srcOriginPresentation);
                    } else {
                        expeditionQuantity = quantity;
                    }

                    // reduction de la quantité de la presentation
                    double presQuantity = srcPresentation.getQuantity();
                    presQuantity -= expeditionQuantity;
                    if (sourceBatch.getExpiredDate() == null) {
                        srcPresentation.setQuantity(presQuantity);
                    }
                    presentationDAO.update(srcPresentation);

                    importLog.setMessage(_("Nouvelle expedition pour le lot %s au client %s",
                            batchId, client));

                    // correction eventuelle
                    if (correctionBatch != null && correctionBatch.getExpiredDate() == null) {
                        // ajout du stock precedement retirée sur la presentation
                        if (corrPresentation == null) {
                            corrPresentation = presentationDAO.createByNaturalId(correctionBatch, presentationCode);
                        }
                        double corrQuantity = corrPresentation.getQuantity();
                        corrQuantity += quantity;
                        corrPresentation.setQuantity(corrQuantity);
                        presentationDAO.update(corrPresentation);
                    }

                    // ajout d'une expedition
                    Expedition expedition = expeditionDAO.create();
                    expedition.setDate(lineDate);
                    expedition.setPresentation(srcPresentation);
                    expedition.setCorrection(corrPresentation);
                    expedition.setQuantity(quantity);
                    // les expeditions ne sont pas liées aux entités client
                    // car ils ne sont pas en base. Les clients en base
                    // sont ceux dediés à certains lots
                    expedition.setClient(clientId);
                    expedition.setSource(sourceZone);

                    // Toutes étiquettes ayant un code client interne 99997 alimentent la zone ZC
                    if (client == Zone.ZC.getClient()) {
                        expedition.setDestination(Zone.ZC);
                    } else {
                        // "Toutes les étiquettes ayant un code client externe alimentent la zone ZP"
                        // En fait, une etiquette pour un client externe ne change par la zone
                        expedition.setDestination(sourceZone);
                    }
                    expeditionDAO.update(expedition);

                    // DMESD : date de la première vente du lot quelques soit
                    // sa présentation mais au niveau de l'atelier des expéditions.
                    // (seulement pour les clients externes)
                    if (sourceZone == Zone.ZP && client != Zone.ZC.getClient()) {
                        Date previousDate = sourceBatch.getDmesd();
                        if (previousDate == null || lineDate.before(previousDate)) {
                            sourceBatch.setDmesd(lineDate);
                            if (log.isDebugEnabled()) {
                                log.debug("Update dmesd for batch to " + lineDate);
                            }
                        }
                    }
                }
            }

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (ParseException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import label file", ex);
        } finally {
            IOUtils.closeQuietly(reader);
        }

        return importLogs;
    }

    /**
     * Extraction du code de control et d'enregistrement dans un methode
     * séparée de la routine d'import pour être aussi utilisé lors du replay.
     * 
     * C'est une duplication du code dans la bouble de importLabelsFile,
     * mais il n'est pas perfomant (cache) pour être utilisé en standalone.
     */
    protected ImportLog importSingleLine(String[] csvdata, Zone sourceZone) {

        ImportLog importLog = new ImportLog();

        try {
            // get date format
            DateFormat dfIn = new SimpleDateFormat("yyyyMMdd");

            // get dao
            PresentationDAO presentationDAO = daoHelper.getPresentationDAO();
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ExpeditionDAO expeditionDAO = daoHelper.getExpeditionDAO();
            ProductionDAO productionDAO = daoHelper.getProductionDAO();

            String dateyyyyMMdd = csvdata[1];
            String clientId = csvdata[5];
            String productId = csvdata[7];
            String batchId = csvdata[9];
            String quantityStr = csvdata[11];
            String batchCorrectionId = null;
            if (csvdata.length >= 14) {
                batchCorrectionId = csvdata[13];
            }

            Date lineDate = dfIn.parse(dateyyyyMMdd);

            // get source batch
            int srcBatchNumber = Integer.parseInt(batchId);
            Batch sourceBatch = batchDAO.findByNumber(srcBatchNumber);
            if (sourceBatch == null) {
                importLog.setError(true);
                importLog.setMessage(_("Impossible de trouver le lot %s", batchId));
                return importLog;
            }

            // get source batch
            Batch correctionBatch = null;
            if (StringUtils.isNotBlank(batchCorrectionId)) {
                int destBatchNumber = Integer.parseInt(batchCorrectionId);
                correctionBatch = batchDAO.findByNumber(destBatchNumber);
                if (correctionBatch == null) {
                    importLog.setError(true);
                    importLog.setMessage(_("Impossible de trouver le lot %s", batchCorrectionId));
                    return importLog;
                }
                
                // lot corrigé = lot source
                if (sourceBatch.equals(correctionBatch)) {
                    importLog.setError(true);
                    importLog.setMessage(_("Correction sur un lot identique au lot corrigé :%d", batchCorrectionId));
                    return importLog;
                }
            }

            // lot ignoré
            if (sourceBatch.isInvalid()) {
                importLog.setError(true);
                importLog.setMessage(_("Le lot %d a été marqué comme ignoré. Impossible d'enregistrer une étiquette sur ce lot.", sourceBatch.getNumber()));
                return importLog;
            }
            // lot ignoré
            if (correctionBatch != null && correctionBatch.isInvalid()) {
                importLog.setError(true);
                importLog.setMessage(_("Le lot %d a été marqué comme ignoré. Impossible d'enregistrer une étiquette sur ce lot.", correctionBatch.getNumber()));
                return importLog;
            }
            
            // etiquette entrée après épuisement du lot
            if (sourceBatch.getExpiredDate() != null && lineDate.after(sourceBatch.getExpiredDate())) {
                importLog.setError(true);
                importLog.setMessage(_("La date de l'étiquette (%1$td/%1$tm/%1$tY) porte après la date d'expiration du lot %2$d (%3$td/%3$tm/%3$tY).",
                        lineDate, sourceBatch.getNumber(), sourceBatch.getExpiredDate()));
                return importLog;
            }
            // etiquette entrée après épuisement du lot
            if (correctionBatch != null && correctionBatch.getExpiredDate() != null && lineDate.after(correctionBatch.getExpiredDate())) {
                importLog.setError(true);
                importLog.setMessage(_("La date de l'étiquette (%1$td/%1$tm/%1$tY) porte après la date d'expiration du lot %2$d (%3$td/%3$tm/%3$tY).",
                        lineDate, correctionBatch.getNumber(), correctionBatch.getExpiredDate()));
                return importLog;
            }

            // test presentation
            PresentationCode presentationCode = getPresentationCode(productId);
            if (presentationCode == null) {
                importLog.setError(true);
                importLog.setMessage(_("Lot %d : Code de présentation inconnu %s",
                        sourceBatch.getNumber(), productId));
                return importLog;
            }

            // gestion de l'equivalence pour les code de presentation utilisé par la facturation
            if (presentationCode.getEquivalence() != null) {
                presentationCode = presentationCode.getEquivalence();
            }

            double quantity = Double.parseDouble(quantityStr);
            int client = Integer.parseInt(clientId);

            // Les lignes ayant un code ayant une structure du type F0XXXX devront être importées sous la forme FXXXX
            // Les lignes ayant un code ayant une structure du type T0XXXX devront être importées sous la forme TXXXX
            if ((presentationCode == PresentationCode.F || presentationCode == PresentationCode.T)
                    && productId.charAt(1) == '0') {
                // suppression du premier 0
                // mais le code pres doit rester dans le code produit dans ce cas
                productId = presentationCode.getCode() + productId.substring(2);
            } else {
                productId = productId.substring(1);
            }

            // check du code produit qui doit correspondre au code
            // produit sur lequel le lot porte
            if (correctionBatch != null && !checkBatchProduct(correctionBatch, productId, lineDate)) {
                if (log.isErrorEnabled()) {
                    log.error(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                            correctionBatch.getNumber(), productId, correctionBatch.getProduct().getCode()));
                }
                importLog.setError(true);
                importLog.setMessage(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                        correctionBatch.getNumber(), productId, correctionBatch.getProduct().getCode()));
                return importLog;
            }

            // check du code produit qui doit correspondre au code
            // produit sur lequel le lot porte
            if (!checkBatchProduct(sourceBatch, productId, lineDate)) {
                if (log.isErrorEnabled()) {
                    log.error(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                            sourceBatch.getNumber(), productId, sourceBatch.getProduct().getCode()));
                }
                importLog.setError(true);
                importLog.setMessage(_("Lot %d : Le code produit %s ne correspond pas au code produit attendu pour ce lot %s",
                        sourceBatch.getNumber(), productId, sourceBatch.getProduct().getCode()));
                return importLog;
            }

            // recherche de la presentation affectée par la production
            // (presentation d'origine) et/ou la vente (presentation dest)
            List<Presentation> srcPresentations = presentationDAO.findAllByBatch(sourceBatch);
            Presentation srcPresentation = null;
            Presentation srcOriginPresentation = null;
            for (Presentation presentation : srcPresentations) {
                if (presentation.getPresentationCode() == presentationCode) {
                    srcPresentation = presentation;
                }
                if (presentation.isOriginal()) {
                    srcOriginPresentation = presentation;
                }
            }

            // recherche de la présentation du lot de correction
            Presentation corrPresentation = null;
            Presentation corrOriginPresentation = null;
            if (correctionBatch != null) {
                List<Presentation> destPresentations = presentationDAO.findAllByBatch(correctionBatch);
                for (Presentation presentation : destPresentations) {
                    if (presentation.getPresentationCode() == presentationCode) {
                        corrPresentation = presentation;
                    }
                    if (presentation.isOriginal()) {
                        corrOriginPresentation = presentation;
                    }
                }
            }

            // 99998 et 99999 = production
            if (client == Zone.ZE.getClient() || client == Zone.ZP.getClient()) {

                // dans le cas ou la presentation produite n'existe pas,
                // on doit en creer une nouvelle
                if (srcPresentation == null) {
                    srcPresentation = presentationDAO.createByNaturalId(sourceBatch, presentationCode);

                    importLog.setMessage(_("Nouvelle presentation %c pour le lot %s",
                            presentationCode.getCode(), batchId));
                } else {
                    importLog.setMessage(_("Mise à jour de la presentation %c du lot %s",
                            presentationCode.getCode(), batchId));
                }

                // pour FIF_HIST (de ZE vers ZE c'est une production)
                // dans tous les autres cas, il n'y a production que
                // si la quantité de destination n'est pas suffisante
                double productionQuantity;
                if (sourceZone == Zone.ZE && client == Zone.ZE.getClient() || srcPresentation.getQuantity() <= 0) {
                    productionQuantity = quantity;
                } else {
                    if (srcPresentation.getQuantity() < quantity) {
                        productionQuantity = quantity - srcPresentation.getQuantity();
                    } else {
                        productionQuantity = 0;  // transfert
                    }
                }

                // si production, reduction du stock (original)
                if (srcOriginPresentation == null) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't find original presentation for row " + Arrays.toString(csvdata));
                    }
                } else {

                    // Il se peut même que la transformation ne s'applique
                    // pas a la presentation d'origine, mais on ne peut
                    // pas le savoir et on diminue quand meme le stock de
                    // la presentation d'origine. Il peut donc en resulter un
                    // stock négatif
                    double origPresQuantity = srcOriginPresentation.getQuantity();
                    origPresQuantity -= productionQuantity;
                    if (sourceBatch.getExpiredDate() == null) {
                        srcOriginPresentation.setQuantity(origPresQuantity);
                    }
                    presentationDAO.update(srcOriginPresentation);
                }

                // si production, augmentation du stock (produit)
                double presQuantity = srcPresentation.getQuantity();
                presQuantity += productionQuantity;
                if (sourceBatch.getExpiredDate() == null) {
                    srcPresentation.setQuantity(presQuantity);
                }
                presentationDAO.update(srcPresentation);

                // eventuelle correction
                if (correctionBatch != null && correctionBatch.getExpiredDate() == null) {
                    // gestion des cas incoherents
                    if (corrOriginPresentation == null) {
                        corrOriginPresentation = presentationDAO.createByNaturalId(correctionBatch, PresentationCode._);
                    }
                    if (corrPresentation == null) {
                        corrPresentation = presentationDAO.createByNaturalId(correctionBatch, presentationCode);
                    }

                    // détermination de la quantité réelle suivant les cas
                    double oldProductionQuantity;
                    if (sourceZone == Zone.ZE && client == Zone.ZE.getClient()) {
                        oldProductionQuantity = quantity;
                    } else {
                        // pour les autres cas, si la quantite produite est
                        // exactement la quantité existante, on annule la
                        // production, sinon on suppose que la presentation
                        // existait deja avant et on ne fait rien
                        if (corrPresentation.getQuantity() == quantity) {
                            oldProductionQuantity = quantity;
                        } else {
                            oldProductionQuantity = 0;  // transfert
                        }
                    }

                    // reajout du stock sur la presentation d'origine
                    double corrOriginQuantity = corrOriginPresentation.getQuantity();
                    corrOriginQuantity += oldProductionQuantity;
                    corrOriginPresentation.setQuantity(corrOriginQuantity);
                    presentationDAO.update(corrOriginPresentation);

                    // retrait du stock precedement ajouté sur la presentation
                    double corrQuantity = corrPresentation.getQuantity();
                    corrQuantity -= oldProductionQuantity;
                    corrPresentation.setQuantity(corrQuantity);
                    presentationDAO.update(corrPresentation);
                }

                // sauvegarde des productions
                Production production = productionDAO.create();
                production.setDate(lineDate);
                production.setQuantity(quantity);
                production.setPresentation(srcPresentation);
                production.setCorrection(corrPresentation);
                production.setSource(sourceZone);
                if (client == Zone.ZE.getClient()) {
                    // Toutes étiquettes ayant un code client interne 99999 alimentent la zone ZE
                    production.setDestination(Zone.ZE);
                } else {
                    // Toutes étiquettes ayant un code client interne 99998 alimentent la zone ZP
                    production.setDestination(Zone.ZP);
                }
                productionDAO.update(production);

            } else {

                // il peut se produire des cas où une presentation
                // est expédié directement à un client sans qu'elle
                // ai été enregistrée en production (manque de tracabilité)
                // on ne peut donc pas diminuer le stock dans ce cas
                // mais l'expedition peut-être enregistrer
                if (srcPresentation == null) {
                    if (log.isWarnEnabled()) {
                        log.warn(dateyyyyMMdd + " : Can't find expedition presentation for batch " + sourceBatch.getNumber() + ". Create it !");
                    }
                    srcPresentation = presentationDAO.createByNaturalId(sourceBatch, presentationCode);
                }

                double expeditionQuantity = 0;
                if (srcPresentation.getQuantity() < quantity) {
                    if (srcPresentation.getQuantity() > 0) {
                        expeditionQuantity = srcPresentation.getQuantity();
                    }

                    // cas ou une quantite supplémentaire
                    double productionQuantity = quantity - expeditionQuantity;
                    double presQuantity = srcOriginPresentation.getQuantity();
                    presQuantity -= productionQuantity;
                    if (sourceBatch.getExpiredDate() == null) {
                        srcOriginPresentation.setQuantity(presQuantity);
                    }
                    presentationDAO.update(srcOriginPresentation);
                } else {
                    expeditionQuantity = quantity;
                }

                // reduction de la quantité de la presentation
                double presQuantity = srcPresentation.getQuantity();
                presQuantity -= expeditionQuantity;
                if (sourceBatch.getExpiredDate() == null) {
                    srcPresentation.setQuantity(presQuantity);
                }
                presentationDAO.update(srcPresentation);

                importLog.setMessage(_("Nouvelle expedition pour le lot %s au client %s",
                        batchId, client));

                // correction eventuelle
                if (correctionBatch != null && correctionBatch.getExpiredDate() == null) {
                    // ajout du stock precedement retirée sur la presentation
                    if (corrPresentation == null) {
                        corrPresentation = presentationDAO.createByNaturalId(correctionBatch, presentationCode);
                    }
                    double corrQuantity = corrPresentation.getQuantity();
                    corrQuantity += quantity;
                    corrPresentation.setQuantity(corrQuantity);
                    presentationDAO.update(corrPresentation);
                }

                // ajout d'une expedition
                Expedition expedition = expeditionDAO.create();
                expedition.setDate(lineDate);
                expedition.setPresentation(srcPresentation);
                expedition.setCorrection(corrPresentation);
                expedition.setQuantity(quantity);
                // les expeditions ne sont pas liées aux entités client
                // car ils ne sont pas en base. Les clients en base
                // sont ceux dediés à certains lots
                expedition.setClient(clientId);
                expedition.setSource(sourceZone);

                // Toutes étiquettes ayant un code client interne 99997 alimentent la zone ZC
                if (client == Zone.ZC.getClient()) {
                    expedition.setDestination(Zone.ZC);
                } else {
                    // "Toutes les étiquettes ayant un code client externe alimentent la zone ZP"
                    // En fait, une etiquette pour un client externe ne change par la zone
                    expedition.setDestination(sourceZone);
                }
                expeditionDAO.update(expedition);

                // DMESD : date de la première vente du lot quelques soit
                // sa présentation mais au niveau de l'atelier des expéditions.
                // (seulement pour les clients externes)
                if (sourceZone == Zone.ZP && client != Zone.ZC.getClient()) {
                    Date previousDate = sourceBatch.getDmesd();
                    if (previousDate == null || lineDate.before(previousDate)) {
                        sourceBatch.setDmesd(lineDate);
                        if (log.isDebugEnabled()) {
                            log.debug("Update dmesd for batch to " + lineDate);
                        }
                    }
                }
            }
        } catch (ParseException ex) {
            throw new SgqBusinessException("Can't import line", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import line", ex);
        }
        
        return importLog;
    }

    /**
     * Extrait le code presentation pour une étiquette produit.
     * 
     * Certains cas particuliers peuvent entrer en compte pour déterminer
     * le code presentation de certains produits.
     * 
     * @param productId product id to parse
     * @return
     */
    protected PresentationCode getPresentationCode(String productId) {

        char presentationChar = productId.charAt(0);
        PresentationCode result = PresentationCode.getPresentationCodeFor(presentationChar);
        
        if (result == null) {
            // Cas du SAFRAN vendu au détail
            if ("ZSAF00".equals(productId) || "ZSAF01".equals(productId)) {
                result = PresentationCode._;
            }
        }

        return result;
    }

    /**
     * Test que le code produit de la ligne est celui attendu (c'est à dire
     * celui défini pour le lot).
     * 
     * Cependant, cette methode défini certaines regles de correspondance
     * pour les cas ou le produit ne correspond pas mais la ligne devant
     * quand même être acceptée.
     * 
     * @param batch le lot
     * @param productCode le code produit à tester
     * @param date la date de la ligne (car certaines regles s'appliquent plus à partir d'un certain moment)
     */
    protected boolean checkBatchProduct(Batch batch, String productCode, Date date) {

        boolean result = batch.getProduct().getCode().equals(productCode);

        if (!result) {
            if (log.isDebugEnabled()) {
                log.debug(String.format("Testing special case for batch %d and product %s",
                    batch.getNumber(), productCode));
            }

            // Cas du SAFRAN vendu au détail
            if (productCode.equals("SAF00")) { productCode = "SAF11"; }
            else if (productCode.equals("SAF01")) { productCode = "SAF12"; }

            // Cas des plantes dites "coupé gros"
            else if (productCode.equals("BOL13")) { productCode = "BOL12"; }
            else if (productCode.equals("BOU23")) { productCode = "BOU21"; }
            else if (productCode.equals("CAS34")) { productCode = "CAS32"; }
            else if (productCode.equals("CER34")) { productCode = "CER31"; }
            else if (productCode.equals("CIT32")) { productCode = "CIT31"; }
            else if (productCode.equals("NOY12")) { productCode = "NOY11"; }
            else if (productCode.equals("OLI12")) { productCode = "OLI11"; }
            else if (productCode.equals("SAU14")) { productCode = "SAU11"; }
            else if (productCode.equals("SEN24")) { productCode = "SEN22"; }
            else if (productCode.equals("VIG14")) { productCode = "VIG11"; }
            else if (productCode.equals("ZW199")) { productCode = "ZW198"; }

            // Cas des « transformations internes »
            else if (productCode.equals("BOL11")) { productCode = "BOL12"; }
            else if (productCode.equals("EUC12")) { productCode = "EUC11"; }
            else if (productCode.equals("HAR22")) { productCode = "HAR21"; }
            else if (productCode.equals("ORG11")) { productCode = "ORG12"; }
            else if (productCode.equals("SOU11")) { productCode = "SOU12"; }
            
            // Cas particuliers de fabrications exceptionnelles
            else if (productCode.equals("BLE21")) { productCode = "BLE22"; }
            else if (productCode.equals("GIN23")) { productCode = "GIN21"; }
            else if (productCode.equals("GIN28")) { productCode = "GIN26"; }
            else if (productCode.equals("MEL21")) { productCode = "MEL23"; }
            else if (productCode.equals("MEL22")) { productCode = "MEL21"; }
            else if (productCode.equals("MEN32")) { productCode = "MEL31"; }

            // Cas des anciens codes encore actif (par erreur) dans DBASE
            else if (productCode.equals("TAN12")) { productCode = "TAN11"; }
            else if (productCode.equals("AIR12")) { productCode = "MYR32"; }
            else if (productCode.equals("BAD12")) { productCode = "BAD11"; }
            else if (productCode.equals("CAM12")) { productCode = "CAM11"; }
            else if (productCode.equals("MAT12")) { productCode = "MAT11"; }
            else if (productCode.equals("TRE11")) { productCode = "TRE12"; }

            // regles en plus (code lutin)
            //else if (productCode.equals("ECH12")) { productCode = "ECH21"; }

            result = batch.getProduct().getCode().equals(productCode);
        }

        return result;
    }

    /**
     * Return label errors count.
     * 
     * @return label errors count
     */
    public long getLabelErrorCount() {
        long result;
        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            result = labelErrorDAO.findLabelErrorsCount();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label error count", ex);
        }
        return result;
    }

    /**
     * Recupere toutes les erreurs d'import non traitées pour une zone.
     * 
     * @param zone zone (ZE or ZP)
     * @return errors by zone
     */
    public List<LabelError> getLabelErrorsForZone(Zone source) {

        List<LabelError> errors = null;
        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            errors = labelErrorDAO.findLabelErrorsBySource(source);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label errors", ex);
        }

        return errors;
    }

    /**
     * Get error history.
     * 
     * @param beginDate begin date
     * @param endDate end date
     * @param offset offset
     * @param count count
     * @return count history
     */
    public Pair<List<LabelError>, Long> getHistory(Date beginDate, Date endDate, int offset, int count) {
        Pair<List<LabelError>, Long>  result = null;
        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            List<LabelError> elements = labelErrorDAO.findLabelErrors(beginDate, endDate, offset, count);
            long totalCount = labelErrorDAO.findLabelErrorsCount(beginDate, endDate);
            result = Pair.of(elements, totalCount);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label errors", ex);
        }
        return result;
    }

    /**
     * Get history export as csv.
     * 
     * @param beginDate begin date
     * @param endDate end date
     * @return csv export stream
     */
    public InputStream getHistoryAsCsv(Date beginDate, Date endDate) {
        InputStream result = null;
        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            List<LabelError> elements = labelErrorDAO.findAllLabelErrors(beginDate, endDate);

            File file = File.createTempFile("sgq-search", ".csv");
            file.deleteOnExit();

            CSVWriter csvWriter = new CSVWriter(new FileWriter(file), ';');

            // write headers
            List<String> headers = new ArrayList<String>();
            headers.add("Source");
            headers.add("Date");
            headers.add("Données");
            headers.add("État");
            headers.add("Commentaire");
            csvWriter.writeNext(headers.toArray(new String[headers.size()]));

            for (LabelError labelError : elements) {
                List<String> data = new ArrayList<String>(headers.size());
                
                if (labelError.getSource() == Zone.ZE) {
                    data.add("FIF_HIST.txt");
                } else {
                    data.add("FIC_HIST.txt");
                }

                // warning, date can be null (rare)
                if (labelError.getLabelDate() == null) {
                    data.add("");
                } else {
                    data.add(SgqUtils.formatSgqDate(labelError.getLabelDate()));
                }
                
                data.add(labelError.getLine());
                if (labelError.isDeleted()) {
                    data.add("Supprimé");
                } else if (labelError.isModified()) {
                    data.add("Corrigé");
                } else {
                    data.add("Error");
                }
                data.add(labelError.getComment());
                
                csvWriter.writeNext(data.toArray(new String[data.size()]));
            }

            csvWriter.close();

            result = new FileInputStream(file);

        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label errors", ex);
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't export label errors", ex);
        }

        return result;
    }

    /**
     * Replay selected error with corrected replay line.
     * 
     * @param labelErrorId error id to replay
     * @param replayLine corrected data to replay
     * @param comment comment
     */
    public ImportLog replayError(String labelErrorId, String replayLine, String comment) {

        if (StringUtils.isBlank(comment)) {
            throw new SgqBusinessException("Comment is mandatory");
        }

        ImportLog importLog = null;

        try {
            // delete previous
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            LabelError labelError = labelErrorDAO.findByTopiaId(labelErrorId);
            
            // concurrency check (if multiple user try to correct a line at same moment)
            if (labelError.isModified()) {
                throw new SgqBusinessException("Line already modified");
            }
            if (labelError.isDeleted()) {
                throw new SgqBusinessException("Line already deleted");
            }

            labelError.setLine(replayLine);
            labelError.setComment(comment);

            // replay current
            String[] csvdata = replayLine.split(String.valueOf(ERROR_REPLAY_SEPARATOR));

            // ajout d'un check pour verifier que la ligne a le format
            // attendue, notement au niveau des prefix de colonnes
            String columns = "";
            for (int i = 0; i < csvdata.length ; i+=2) {
                columns += csvdata[i];
            }
            if (!"DBCPLQ".equals(columns) && !"DBCPLQR".equals(columns)) {
                throw new SgqBusinessException(_("Nom de colonne invalides; trouvé : %s, attendu : %s ou %s",
                        columns, "DBCPLQ", "DBCPLQR"));
            }

            // perform replay (label error will be recreated if necessary)
            importLog = importSingleLine(csvdata, labelError.getSource());
            if (importLog.isError()) {
                labelError.setMessage(importLog.getMessage());
            } else {
                labelError.setModified(true);
                labelError.setLabelDate(getLabelErrorDate(labelError));
            }

            labelErrorDAO.update(labelError);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label errors", ex);
        }

        return importLog;
    }

    /**
     * Extract label date in error line data.
     * 
     * @return label date or {@code null} is date can't be parsed but should not append
     */
    protected Date getLabelErrorDate(LabelError labelError) {
        String replayLine = labelError.getLine();
        String[] csvdata = replayLine.split(String.valueOf(ERROR_REPLAY_SEPARATOR));
        Date result = null;
        if (csvdata.length > 2) {
            String dateyyyyMMdd = csvdata[1];
            try {
                result = LABEL_DATE_FORMAT.parse(dateyyyyMMdd);
            } catch (ParseException ex) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't parse " + dateyyyyMMdd + " as yyyyMMdd date", ex);
                }
            }
        }
        return result; 
    }

    /**
     * Delete selected label error.
     * 
     * @param labelErrorId label error to delete
     */
    public void deleteError(String labelErrorId, String comment) {
        if (StringUtils.isBlank(comment)) {
            throw new SgqBusinessException("Comment is mandatory");
        }

        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            LabelError labelError = labelErrorDAO.findByTopiaId(labelErrorId);
            labelError.setDeleted(true);
            labelError.setComment(comment);
            labelError.setLabelDate(getLabelErrorDate(labelError));
            labelErrorDAO.update(labelError);
            daoHelper.commit();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get label errors", ex);
        }
    }

    /**
     * Lots probablement expirés et pour lesquels une étiquette a été émise.
     * 
     * @return batch with error
     */
    public List<Batch> getBatchWithLabelAfterExpiration() {
        List<Batch> result;
        try {
            LabelErrorDAO labelErrorDAO = daoHelper.getLabelErrorDAO();
            result = labelErrorDAO.getBatchWithLabelAfterExpiration();
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't get error batch", ex);
        }
        return result;
    }
}
