/*
 * #%L
 * SGQ :: Business
 * $Id: SearchColumn.java 345 2013-04-30 09:25:05Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-1.1.3/sgq-business/src/main/java/com/herbocailleau/sgq/business/model/SearchColumn.java $
 * %%
 * Copyright (C) 2012, 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.model;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.collections.CollectionUtils;

import com.herbocailleau.sgq.business.SgqUtils;
import com.herbocailleau.sgq.entities.AnalyzeType;
import com.herbocailleau.sgq.entities.BatchAnalyze;
import com.herbocailleau.sgq.entities.SynthesisMention;

/**
 * Liste des colonnes selectionnables et affichable dans les resultats de
 * recherche (aussi bien web que export csv).
 * 
 * Elle est dynamique pour la génération du tableau ou du fichier csv,
 * l'enumeration doit donc être capable d'afficher toute seule les données
 * d'un {@link BatchModel}.
 * 
 * @author echatellier
 */
public enum SearchColumn {
    PRODUCT_CODE(_("Code Produit"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            return batchModel.getBatch().getProduct().getCode();
        }
    }, "search:column:common", true),
    PRODUCT_NAME(_("Nom Produit"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            return batchModel.getBatch().getProduct().getName();
        }
    }, "search:column:common", true),
    PRODUCT_LATIN_NAME(_("Nom latin"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            return batchModel.getBatch().getProduct().getLatinName();
        }
    }, "search:column:common", false),
    PRODUCT_CATEGORY(_("Cat\u00E9gorie"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            return batchModel.getBatch().getProduct().getCategory();
        }
    }, "search:column:common", false),
    BATCH_NUMBER(_("Lot"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            return String.valueOf(batchModel.getBatch().getNumber());
        }
    }, "search:column:common", true),
    BATCH_SUPPLIER(_("Fournisseur"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getSupplier() != null) {
                result = batchModel.getBatch().getSupplier().getName();
            }
            return result;
        }
    }, "search:column:supplier", false),
    BATCH_CLIENT(_("Client"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getDedicatedClient() != null) {
                result = batchModel.getBatch().getDedicatedClient().getName();
            }
            return result;
        }
    }, "search:column:client", false),
    BATCH_ORIGIN(_("Origine"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getOrigin() != null) {
                result = batchModel.getBatch().getOrigin().getCountry();
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_TOTAL_PRESENTATION(_("Quantité disponible"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            //return String.valueOf(batchModel.getTotalPresentation());
            return SgqUtils.formatNumber(batchModel.getTotalPresentation());
        }
    }, "search:column:common", true),
    BATCH_TOTAL_SALE(_("Cumul vente"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            //return String.valueOf(batchModel.getTotalSale());
            return SgqUtils.formatNumber(batchModel.getTotalSale());
        }
    }, "search:column:common", false),
    BATCH_ENTRY_DATE(_("Date d'entrée"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getEntryDate() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getBatch().getEntryDate());
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_DPMES(_("DPMES"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getDpmes() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getBatch().getDpmes());
            }
            return result;
        }
    }, "search:column:common", false),
    BATCHDMES(_("DMES"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getDmes() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getDmes());
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_DMESD(_("DMESD"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getDmesd() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getBatch().getDmesd());
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_DLUO(_("DLUO"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getDluo() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getBatch().getDluo());
            }
            return result;
        }
    }, "search:column:common", true),
    ANALYZE_CONFORMITY(_("Conformité Analyse"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {

            StringBuilder strResult = new StringBuilder();
            String separator = "";
            for (BatchAnalyze batchAnalyze : batchModel.getBatchAnalyzes()) {
                AnalyzeType type = batchAnalyze.getAnalyzeType();
                if ((CollectionUtils.isEmpty(searchModel.getAnalyzeTypes()) ||
                        searchModel.getAnalyzeTypes().contains(type)) && 
                        (searchModel.getSynthesisMention() == null ||
                        searchModel.getSynthesisMention() == batchAnalyze.getSynthesisMention())) {
                    strResult.append(separator);
                    strResult.append(type.getAnalyzeCategory().getName());
                    strResult.append(" - ").append(type.getName());
                    strResult.append(" : ").append(getCurrentResult(batchAnalyze));
                    separator = ", ";
                }
            }

            return strResult.toString();
        }

        protected int compare(BatchAnalyze o1, BatchAnalyze o2) {
            int result = 0;
            
            if (o1 == null) {
                if (o2 != null) {
                    result = -1;
                }
            } else if (o2 == null) {
                result = 1;
            } else if (o1.getSynthesisMention() != null) {
                if (o2.getSynthesisMention() != null) {
                    result = o2.getSynthesisMention().ordinal() - o1.getSynthesisMention().ordinal();
                } else {
                    result = 1;
                }
            } else if (o2.getSynthesisMention() != null) {
                    result = -1;
            } else if (o1.getSentDate() != null) {
                if (o2.getSentDate() != null) {
                    result = 1;
                }
            } else if (o2.getSentDate() != null) {
                result = -1;
            }
            return result;
        }
        protected String getCurrentResult(BatchAnalyze batchAnalyze) {
            String result = null;
            if (SynthesisMention.CONFORM == batchAnalyze.getSynthesisMention()) {
                result = "Conforme";
            } else if (SynthesisMention.ACCEPTABLE == batchAnalyze.getSynthesisMention()) {
                result = "Acceptable";
            } else if (SynthesisMention.NOT_ACCEPTABLE == batchAnalyze.getSynthesisMention()) {
                result = "Non acceptable";
            } else if (batchAnalyze.getSentDate() != null) {
                result = "Échantillon envoyé";
            } else {
                result = "À faire";
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_EXPIRED_DATE(_("Date d'épuisement"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = null;
            if (batchModel.getBatch().getExpiredDate() != null) {
                result = SgqUtils.formatSgqDate(batchModel.getBatch().getExpiredDate());
            }
            return result;
        }
    }, "search:column:common", false),
    BATCH_COMMENT(_("Commentaire"), new BatchModelFunction() {
        @Override
        public String apply(SearchModel searchModel, BatchModel batchModel) {
            String result = batchModel.getBatch().getComment();
            return result;
        }
    }, "search:column:comment", false);

    protected String description;

    protected boolean defaultSelected;

    protected BatchModelFunction function;

    /** Permission shiro necessaire à l'exploitation de la colonne. */
    protected String permission;

    private SearchColumn(String description, BatchModelFunction function, String permission, boolean defaultSelected) {
        this.description = description;
        this.function = function;
        this.permission = permission;
        this.defaultSelected = defaultSelected;
    }

    public String getDescription() {
        return description;
    }

    public boolean isDefaultSelected() {
        return defaultSelected;
    }

    public String getPermission() {
        return permission;
    }

    public String getValueFor(SearchModel searchModel, BatchModel batchModel) {
        return function.apply(searchModel, batchModel);
    }
}

interface BatchModelFunction {
    public String apply(SearchModel searchModel, BatchModel batchModel);
}