/*
 * #%L
 * SGQ :: Business
 * $Id: DataRecoveryService.java 51 2012-09-11 13:56:40Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/sgq-ch/tags/sgq-ch-0.2/sgq-business/src/main/java/com/herbocailleau/sgq/business/services/DataRecoveryService.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business.services;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Iterator;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.Row;
import org.nuiton.topia.TopiaException;

import com.herbocailleau.sgq.business.SgqBusinessException;
import com.herbocailleau.sgq.business.SgqService;
import com.herbocailleau.sgq.entities.Batch;
import com.herbocailleau.sgq.entities.BatchDAO;
import com.herbocailleau.sgq.entities.BatchImpl;
import com.herbocailleau.sgq.entities.Product;
import com.herbocailleau.sgq.entities.ProductDAO;

public class DataRecoveryService extends SgqService {

    private static final Log log = LogFactory.getLog(DataRecoveryService.class);

    /**
     * Import du fichier numéro de lot.
     * 
     * @param batchFile fichier numero lot
     */
    public void importBatchFile(File batchFile) {

        FileInputStream batchStream = null;
        try {
            
            BatchDAO batchDAO = daoHelper.getBatchDAO();
            ProductDAO productDAO = daoHelper.getProductDAO();

            batchStream = new FileInputStream(batchFile);

            POIFSFileSystem myFileSystem = new POIFSFileSystem(batchStream);
            HSSFWorkbook myWorkBook = new HSSFWorkbook(myFileSystem);

            HSSFSheet mySheet = myWorkBook.getSheet("Entrées pdts");
            Iterator<Row> rowIter = mySheet.rowIterator();

            // Code Pdt,N°Lot, DMES, Produit, Date d'entrée, Colis, Poids,
            // Unité, Fournisseur, DLUO
            rowIter.next(); // skip header

            while (rowIter.hasNext()) {
                HSSFRow myRow = (HSSFRow) rowIter.next();
                //HSSFComment comment = myRow.getCell(10).getCellComment();

                // get value for row
                String codeProduit = myRow.getCell(0).getStringCellValue();
                double numeroLot = myRow.getCell(1).getNumericCellValue();
                Date dmes = null;
                if (myRow.getCell(2).getCellType() == HSSFCell.CELL_TYPE_NUMERIC) {
                    dmes = myRow.getCell(2).getDateCellValue();
                }
                // produit depend du code produit
                Date dateEntree = myRow.getCell(4).getDateCellValue();
                //String colis = myRow.getCell(0).getStringCellValue();
                double poids = myRow.getCell(6).getNumericCellValue();
                String unite = myRow.getCell(7).getStringCellValue();
                String fournisseur = myRow.getCell(8).getStringCellValue();
                Date dluo = myRow.getCell(9).getDateCellValue();

                // transform value according to database
                int indexOfSlash = codeProduit.indexOf('/');
                if (indexOfSlash >= 0) {
                    codeProduit = codeProduit.substring(0, indexOfSlash);
                }
                //Date dateEntreeDate = df.parse(dateEntree);
                //Date dluoDate = df.parse(dluo);

                Batch batch = new BatchImpl();
                batch.setNumber((int)numeroLot);
                batch.setEntryDate(dateEntree);
                batch.setDluo(dluo);
                batch.setProduct(productDAO.findByCode(codeProduit));
                batchDAO.create(batch);
            }

            batchStream.close();

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import numero lot file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import numero lot file", ex);
        } finally {
            IOUtils.closeQuietly(batchStream);
        }
    }

    public void importProductFile(File productsFile) {
        FileInputStream batchStream = null;
        try {

            ProductDAO productDAO = daoHelper.getProductDAO();

            batchStream = new FileInputStream(productsFile);

            POIFSFileSystem myFileSystem = new POIFSFileSystem(batchStream);
            HSSFWorkbook myWorkBook = new HSSFWorkbook(myFileSystem);

            HSSFSheet mySheet = myWorkBook.getSheetAt(0);
            Iterator<Row> rowIter = mySheet.rowIterator();

            // Code Produit, Nom, Produit, Catégorie, Désignation Latine
            // Famille, Liste A Pharmacopée, Alimentaire Pharmacopée,
            // Toxique Pharmacopée, Liste B Pharmacopée, Commentaire Pharmacopée
            // Monopole Pharmacopée,
            // Décret n°2008-841 - Plantes médicinales libérées pour les compléments alimentaires
            // Norme AFNOR - NF EN ISO 676, EHIA
            // Arrêté Belge, Commentaires Arrêté Belge
            // Physico Chimique, Lieu d'analyse Physico Chimique
            // Bactério Catégorie 4A, Bactério Catégorie C, Bactério EHIA
            // Pesticides,  Métaux lourds, Botanique, Identification Commentaires

            rowIter.next(); // skip header
            rowIter.next(); // skip second header

            while (rowIter.hasNext()) {
                HSSFRow myRow = (HSSFRow) rowIter.next();

                // get value for row
                HSSFCell cell = myRow.getCell(0);
                if (cell == null) {
                    break; // end of sheet
                }
                String codeProduit = cell.getStringCellValue();
                if (log.isDebugEnabled()) {
                    log.debug("Importing product " + codeProduit);
                }
                String nom = myRow.getCell(1).toString();
                String category = myRow.getCell(2).getStringCellValue();
                String latinName = myRow.getCell(3).getStringCellValue();
                String family = myRow.getCell(4).getStringCellValue();
                String status = "";
                if ("Oui".equals(myRow.getCell(5).getStringCellValue())) {
                    status += "Liste A Pharmacopée,";
                }
                if ("Oui".equals(myRow.getCell(6).getStringCellValue())) {
                    status += "Alimentaire Pharmacopée,";
                }
                if ("Oui".equals(myRow.getCell(7).getStringCellValue())) {
                    status += "Toxique Pharmacopée,";
                }
                if ("Oui".equals(myRow.getCell(8).getStringCellValue())) {
                    status += "Liste B Pharmacopée,";
                }
                if (StringUtils.isNotBlank(myRow.getCell(9).getStringCellValue())) {
                    status += myRow.getCell(9).getStringCellValue() + ",";
                }
                if ("Oui".equals(myRow.getCell(10).getStringCellValue())) {
                    status += "Monopole Pharmacopée,";
                }
                if ("Oui".equals(myRow.getCell(11).getStringCellValue())) {
                    status += "Décret n°2008-841 - Plantes médicinales libérées pour les compléments alimentaires,";
                }
                if ("Oui".equals(myRow.getCell(12).getStringCellValue())) {
                    status += "Norme AFNOR - NF EN ISO 676,";
                }
                if ("Oui".equals(myRow.getCell(13).getStringCellValue())) {
                    status += "EHIA,";
                }
                if ("Oui".equals(myRow.getCell(14).getStringCellValue())) {
                    status += "Arrêté Belge,";
                }
                if (StringUtils.isNotBlank(myRow.getCell(15).getStringCellValue())) {
                    status += myRow.getCell(15).getStringCellValue() + ",";
                }
                
                String analyze = "";
                if (StringUtils.isNotBlank(myRow.getCell(16).getStringCellValue())) {
                    analyze += myRow.getCell(16).getStringCellValue() + ",";
                }
                // skip 17 for now
                if ("Oui".equals(myRow.getCell(18).getStringCellValue())) {
                    analyze += "Bactério Catégorie 4A,";
                }
                if ("Oui".equals(myRow.getCell(18).getStringCellValue())) {
                    analyze += "Bactério Catégorie C,";
                }
                if ("Oui".equals(myRow.getCell(18).getStringCellValue())) {
                    analyze += "Bactério EHIA,";
                }
                if ("Oui".equals(myRow.getCell(18).getStringCellValue())) {
                    analyze += "Analyse,";
                }
                if ("Oui".equals(myRow.getCell(18).getStringCellValue())) {
                    analyze += "Métaux lourds,";
                }
                boolean botanicControl = false;
                if ("Oui".equals(myRow.getCell(19).getStringCellValue())) {
                    botanicControl = true;
                }
                boolean identificationControl = false;
                if ("Oui".equals(myRow.getCell(20).getStringCellValue())) {
                    identificationControl = true;
                }

                Product product = productDAO.createByNaturalId(codeProduit);
                product.setName(nom);
                product.setCategory(category);
                product.setLatinName(latinName);
                product.setFamily(family);
                product.setStatus(status);
                product.setAnalyzes(analyze);
                product.setBotanicControl(botanicControl);
                product.setIdentificationControl(identificationControl);
                productDAO.update(product);
            }

            batchStream.close();

            daoHelper.commit();
        } catch (IOException ex) {
            throw new SgqBusinessException("Can't import product file", ex);
        } catch (TopiaException ex) {
            throw new SgqBusinessException("Can't import product file", ex);
        } finally {
            IOUtils.closeQuietly(batchStream);
        }
    }

    /**
     * Export le fichier d'inventaire.
     * 
     * @param inventoryFile inventory file
     */
    public void exportInventory(File inventoryFile) {

        FileOutputStream fileOut = null;

        try {
            HSSFWorkbook wb = new HSSFWorkbook();

            HSSFSheet sheet = wb.createSheet();

            // header
            HSSFRow headerRow = sheet.createRow(0);
            headerRow.createCell(0).setCellValue("Lot");

            // data
            HSSFRow row = sheet.createRow(1);
            HSSFCell cell = row.createCell(0);
            cell.setCellValue(1);
            row = sheet.createRow(2);
            cell = row.createCell(0);
            cell.setCellValue(2);

            // write it
            fileOut = new FileOutputStream(inventoryFile);
            wb.write(fileOut);
            fileOut.close();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } finally {
            IOUtils.closeQuietly(fileOut);
        }
        
    }
}
