/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty.solr;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.embedded.EmbeddedSolrServer;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.core.CoreContainer;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FacetTopic;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.FieldType.TYPE;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.TreeNode;
import org.nuiton.wikitty.UpdateResponse;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyExtensionStorage;
import org.nuiton.wikitty.WikittySearchEngin;
import org.nuiton.wikitty.WikittyTransaction;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.arjuna.ats.arjuna.coordinator.BasicAction;
import com.arjuna.ats.arjuna.coordinator.OnePhaseResource;
import com.arjuna.ats.arjuna.coordinator.TwoPhaseOutcome;
import com.arjuna.ats.arjuna.state.InputObjectState;
import com.arjuna.ats.arjuna.state.OutputObjectState;
import com.arjuna.ats.internal.arjuna.abstractrecords.LastResourceRecord;

/**
 *
 * @author poussin
 * @version $Revision: 351 $
 *
 * Last update: $Date: 2010-09-28 12:05:55 +0200 (mar., 28 sept. 2010) $
 * by : $Author: bleny $
 */
public class WikittySearchEnginSolr implements WikittySearchEngin {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittySearchEnginSolr.class);

    /** Solr data dir config option name. */
    static final protected String SOLR_DATA_DIR_CONFIG = "solr.data.dir";

    /** id field in solr */
    static final protected String SOLR_ID = "id";

    /** extensions field name in solr */
    static final public String SOLR_EXTENSIONS = "extensions";

    /** group all fields is not null */
    static final public String SOLR_NOT_NULL_FIELDS = "not_null_fields";

    /** extension use to store field without extension to search on all extesnion */
    static final public String SOLR_ALL_EXTENSIONS = "all";

    /** Precise the query parser to use, is allow leading wildcard */
    static final public String SOLR_QUERY_PARSER = "{!wikitty}";

    // Use for indexation tree node
    static final public String TREENODE_PREFIX = TreeNode.EXT_TREENODE + ".";
    static final public String TREENODE_EMPTY = TREENODE_PREFIX + "empty";
    static final public String TREENODE_ROOT = TREENODE_PREFIX + "root";
    static final public String TREENODE_PATH = TREENODE_PREFIX + "path";

    /** use to permit client to modify fieldname during query generation */
    static public interface FieldModifier {
        public String convertToSolr(WikittyTransaction transaction, String fieldname);
        public String convertToField(WikittyTransaction transaction, String solrName);
    }

    static protected class TypeFieldModifer implements FieldModifier {
        protected WikittyExtensionStorage extensionStorage;
        public TypeFieldModifer(WikittyExtensionStorage extensionStorage) {
            this.extensionStorage = extensionStorage;
        }

        @Override
        public String convertToSolr(WikittyTransaction transaction, String fqfieldname) {
            String result = fqfieldname;
            String[] searchField = fqfieldname.split("\\.");

            if (Element.ELT_EXTENSION.equals(fqfieldname)) {
                result = SOLR_EXTENSIONS;

            } else if (Element.ELT_ID.equals(fqfieldname)) {
                result = SOLR_ID;
                
            } else if (searchField.length >= 2) {
                String extName = searchField[0];
                String fieldName = searchField[1];

                if (Criteria.ALL_EXTENSIONS.equals(extName)) {
                    fqfieldname = SOLR_ALL_EXTENSIONS + "." + fieldName;
                }

                if (searchField.length >= 3) {
                    String fieldNameType = searchField[2];
                    TYPE type = FieldType.TYPE.valueOf(fieldNameType);
                    result = WikittySearchEnginSolr.getSolrFieldName(fqfieldname, type);
                    return result;
                }

                // Search type of field in extension
                String version = extensionStorage.getLastVersion(transaction, extName);
                if (version != null) { // not valid extension is version == null
                    WikittyExtension ext = extensionStorage.restore(transaction, extName, version);
                    FieldType fieldType = ext.getFieldType(fieldName);
                    log.debug(ext.toDefinition() + " for " + fieldName);
                    if (fieldType != null) { // type can be null if extension version differ
                        TYPE type = fieldType.getType();
                        result = WikittySearchEnginSolr.getSolrFieldName(fqfieldname, type);
                        return result;
                    }
                }
            }

            return result;
        }

        @Override
        public String convertToField(WikittyTransaction transaction, String solrName) {
            String fieldName = solrName.replaceAll("(_b$)|(_dt$)|(_s$)|(_d$)", "");
            if(SOLR_EXTENSIONS.equals(fieldName)) {
                fieldName = Element.ELT_EXTENSION;
            }
            return fieldName;
        }
    }

    /**
     * Helper to get information nodes and elements for reindexation.
     */
    static protected class ReindexChildTreeNode {

        protected SolrResource solrResource;
        protected SolrServer solrServer;
        
        protected Map<String, Collection<String>> includedNodeIds;
        protected Map<String, Collection<String>> excludedNodeIds;
        protected Map<String, String> parents;

        public ReindexChildTreeNode(SolrServer solrServer, SolrResource solrResource) {
            this.solrServer = solrServer;
            this.solrResource = solrResource;
            includedNodeIds = new HashMap<String, Collection<String>>();
            excludedNodeIds = new HashMap<String, Collection<String>>();
            parents = new HashMap<String, String>();
        }

        public void putIncludedAttachments(String nodeId, Collection<String> attchmentIds) {
            putAttachements(includedNodeIds, nodeId, attchmentIds);
        }

        public void putExcludedAttachments(String nodeId, Collection<String> attachmentIds) {
            putAttachements(excludedNodeIds, nodeId, attachmentIds);
        }

        public void putIncludedAttachment(String nodeId, String attachmentId) {
            putAttachment(includedNodeIds, nodeId, attachmentId);
        }

        public void putExcludedAttachment(String nodeId, String attachmentId) {
            putAttachment(excludedNodeIds, nodeId, attachmentId);
        }

        public Collection<String> getExcludedNodeIds(String attachmentId) {
            Collection<String> result = excludedNodeIds.get(attachmentId);
            if (result == null) {
                result = new HashSet<String>();
            }
            return result;
        }

        public Collection<String> getIncludedNodeIds(String attachmentId) {
            Collection<String> result = includedNodeIds.get(attachmentId);
            if (result == null) {
                result = new HashSet<String>();
            }
            return result;
        }

        protected void putAttachements(Map<String, Collection<String>> map, String nodeId, Collection<String> attachmentIds) {
            if (attachmentIds != null) {
                for (String attachmentId : attachmentIds) {
                    putAttachment(map, nodeId, attachmentId);
                }
            }
        }

        protected void putAttachment(Map<String, Collection<String>> map, String nodeId, String attachmentId) {
            Collection<String> values = map.get(attachmentId);
            if(values == null) {
                values = new HashSet<String>();
                map.put(attachmentId, values);
            }
            values.add(nodeId);
        }

        public void putParent(String nodeId, String parentId) {
            parents.put(nodeId, parentId);
        }

        public String getParent(String nodeId) {
            String parentId = parents.get(nodeId);

            // If not found in map, search in index
            if(parentId == null) {
                SolrDocument doc = findById(solrServer, nodeId);
                if(doc == null) {
                    // is root
                    return null;
                }
                parentId = (String) doc.getFieldValue(TreeNode.FQ_FIELD_TREENODE_PARENT);
                parents.put(nodeId, parentId);
            }

            Collection<String> deletedDocIds = solrResource.getDeletedDocs();
            if(deletedDocIds.contains(parentId)) {
                return null;
            }
            return parentId;
        }

        public Collection<String> getReindexIds() {
            Collection<String> result = new HashSet<String>();
            result.addAll(includedNodeIds.keySet());
            result.addAll(excludedNodeIds.keySet());
            result.addAll(solrResource.getAddedDocIds());
            return result;
        }

        /**
         * Add in doc fields on association between nodes.
         *
         * For example if you have a element in node with parent, like this
         * A -> B -> C => element, the method add field in document solr :
         * TreeNode.root : A
         * TreeNode.A : B
         * TreeNode.B : C
         * TreeNode.C : TreeNode.empty
         * 
         * @throws SolrServerException 
         */
        public void reindex() throws SolrServerException {
            for (String id : getReindexIds()) {

                // Get documents
                SolrInputDocument doc = solrResource.getAddedDoc(id);
                if(doc == null) {
                    doc = new SolrInputDocument();

                    // Copy old field value
                    SolrDocument found = findById(solrServer, id);
                    if (found != null) {
                	    Collection<String> fieldNames = found.getFieldNames();
	                    for (String fieldName : fieldNames) {
	                        Collection<Object> fieldValues = found.getFieldValues(fieldName);
	
	                        if(!fieldName.startsWith(TREENODE_PREFIX)) {
	                            for (Object fieldValue : fieldValues) {
	                                doc.addField(fieldName, fieldValue);
	                            }
	                        }
	                    }

    	                solrResource.addDoc(id, doc);
    	            } else {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find wikitty id '" + id + "' in index. Skip this wikitty.");
                        }
                    }
                }

                // Add tree node fields
                Collection<String> includedChildNodeIds = getIncludedNodeIds(id);
                Collection<String> excludedChildNodeIds = getExcludedNodeIds(id);

                // Find all node contain child
                SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TreeNode.FQ_FIELD_TREENODE_ATTACHMENT + ":" + id);
                QueryResponse response = solrServer.query(query);
                SolrDocumentList updateDocs = response.getResults();

                for (Iterator<SolrDocument> iterator = updateDocs.iterator();
                        iterator.hasNext();) {
                    SolrDocument solrDocument = iterator.next();

                    String nodeId = (String) solrDocument.getFieldValue(SOLR_ID);
                    includedChildNodeIds.add(nodeId);
                }

                // Excluded nodes
                includedChildNodeIds.removeAll(excludedChildNodeIds);
                includedChildNodeIds.removeAll(solrResource.getDeletedDocs());

                // Add paths in doc
                Map<String, String> paths = new HashMap<String, String>();
                for (String nodeId : includedChildNodeIds) {
                    doc.addField(TREENODE_PREFIX + nodeId, TREENODE_EMPTY);

                    // Add path
                    String childParent = nodeId;
                    String parent = getParent(childParent);
                    while (parent != null) {
                        String parentPath = paths.get(childParent);
                        if(parentPath == null) {
                            doc.addField(TREENODE_PREFIX + parent, childParent);
                            paths.put(childParent, parent);
                        }

                        childParent = parent;
                        parent = getParent(childParent);
                    }

                    String parentPath = paths.get(childParent);
                    if(parentPath == null) {
                        doc.addField(TREENODE_ROOT, childParent);
                        paths.put(childParent, TREENODE_ROOT);
                    }
                }
            }
        }
    }

    /**
     * Use to plug solr indexation in JTA transaction.
     */
    static protected class SolrResource implements OnePhaseResource {

        protected SolrServer solrServer;
        protected ThreadLocal<Map<String, SolrInputDocument>> addedDocs;
        protected ThreadLocal<List<String>> deletedDocs;

        public SolrResource(SolrServer solrServer) {
            this.solrServer = solrServer;
            addedDocs = new ThreadLocal<Map<String, SolrInputDocument>>();
            deletedDocs = new ThreadLocal<List<String>>();

            clear();
        }

        protected void init() {
            // Add resource on phase in current transaction
            LastResourceRecord lastResourceRecord = new LastResourceRecord(this);
            BasicAction.Current().add(lastResourceRecord);
        }

        public Map<String, SolrInputDocument> getAddedDocs() {
            Map<String, SolrInputDocument> result = addedDocs.get();
            if(result == null) {
                result = new HashMap<String, SolrInputDocument>();
                addedDocs.set(result);
            }
            return result;
        }

        public List<String> getDeletedDocs() {
            List<String> result = deletedDocs.get();
            if(result == null) {
                result = new ArrayList<String>();
                deletedDocs.set(result);
            }
            return result;
        }

        public void clear() {
            addedDocs.set(new HashMap<String, SolrInputDocument>());
            deletedDocs.set(new ArrayList<String>());
        }

        public void addDoc(String id, SolrInputDocument doc) {
            getAddedDocs().put(id, doc);
        }

        public SolrInputDocument getAddedDoc(String id) {
            SolrInputDocument result = getAddedDocs().get(id);
            return result;
        }

        public Collection<String> getAddedDocIds() {
            Collection<String> result = getAddedDocs().keySet();
            return result;
        }

        public void deleteDoc(String docId) {
            getDeletedDocs().add(docId);
        }

        @Override
        public int commit() {
            try {
                synchronized(this) {
                    Collection<SolrInputDocument> docs = getAddedDocs().values();
                    if(!docs.isEmpty()) {
                        solrServer.add(docs);
                    }
                    List<String> ids = getDeletedDocs();
                    if(!ids.isEmpty()) {
                        solrServer.deleteById(ids);
                    }
                    solrServer.commit();
                }
                clear();
                return TwoPhaseOutcome.FINISH_OK;
            } catch (Exception eee) {
                log.error("Error commit solr", eee);
                return TwoPhaseOutcome.FINISH_ERROR;
            }
        }

        @Override
        public int rollback() {
            clear();
            return TwoPhaseOutcome.FINISH_OK;
        }

        @Override
        public void pack(OutputObjectState arg0) throws IOException {
        }

        @Override
        public void unpack(InputObjectState arg0) throws IOException {
        }
    }

    /** solr server */
    protected SolrServer solrServer;

    /** Field modifier use to transform to solr format */
    protected TypeFieldModifer fieldModifier;

    /** JTA resource */
    protected SolrResource solrResource;

    /**
     * Init wikitty search engin on solr embedded server.
     * 
     * @param extensionStorage extension storage
     */
    public WikittySearchEnginSolr(WikittyExtensionStorage extensionStorage) {
        this(extensionStorage, null);
    }

    /**
     * Init wikitty search engin on solr embedded server.
     * 
     * @param extensionStorage extension storage
     * @param properties properties (can be null)
     */
    public WikittySearchEnginSolr(WikittyExtensionStorage extensionStorage, Properties properties) {

        // init system env solr.data.dir
        if (properties != null) {
            String solrDataDir = properties.getProperty(SOLR_DATA_DIR_CONFIG);
            if (solrDataDir != null) {
                System.setProperty(SOLR_DATA_DIR_CONFIG, solrDataDir);
            }
        }

        try {
            CoreContainer.Initializer initializer = new CoreContainer.Initializer();
            CoreContainer coreContainer = initializer.initialize();
            solrServer = new EmbeddedSolrServer(coreContainer, "");

            fieldModifier = new TypeFieldModifer(extensionStorage);
            solrResource = new SolrResource(solrServer);
            
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        try {
            solrResource.init();
            solrServer.deleteByQuery("*:*");
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction, Collection<Wikitty> wikitties) {
        try {
            solrResource.init();
            ReindexChildTreeNode reindexChildTreeNode =
                new ReindexChildTreeNode(solrServer, solrResource);
            for (Wikitty w : wikitties) {
                String id = w.getId();

                if (w.hasExtension(TreeNode.EXT_TREENODE)) {

                    Set<String> attachments = w.getFieldAsSet(TreeNode.EXT_TREENODE, TreeNode.FIELD_TREENODE_ATTACHMENT, String.class);
                    reindexChildTreeNode.putIncludedAttachments(id, attachments);

                    // Search deleted children
                    SolrDocument treeNodeDoc = findById(solrServer, id);
                    if (treeNodeDoc != null) {
                        Collection oldAttachments = treeNodeDoc.getFieldValues(TreeNode.FQ_FIELD_TREENODE_ATTACHMENT);
                        if (oldAttachments != null) {
                            // if no more children, remove all old children
                            if(attachments == null) {
                                reindexChildTreeNode.putExcludedAttachments(id, oldAttachments);
                            } else {
                                // exclude only the removed children
                                for (Object oldAttachment : oldAttachments) {
                                    if(!attachments.contains(oldAttachment)) {
                                        reindexChildTreeNode.putExcludedAttachment(id,(String) oldAttachment);
                                    }
                                }
                            }
                        }
                    }

                    // Get new parent id (may be the same old parent)
                    String parentId = w.getFieldAsString(TreeNode.EXT_TREENODE, TreeNode.FIELD_TREENODE_PARENT);
                    reindexChildTreeNode.putParent(id, parentId);
                }

                // Index
                SolrInputDocument doc = createIndexDocument(w);
                solrResource.addDoc(id, doc);
            }

            // Reindex child in tree node
            reindexChildTreeNode.reindex();

            // no specific result needed
            UpdateResponse result = new UpdateResponse();
            return result;
            
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            solrResource.init();
            ReindexChildTreeNode reindexChildTreeNode =
                new ReindexChildTreeNode(solrServer, solrResource);
            for (String id : ids) {

                // Find child in node id
                SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TREENODE_PREFIX + id + ":*");
                QueryResponse response = solrServer.query(query);
                SolrDocumentList updateDocs = response.getResults();

                for (Iterator<SolrDocument> iterator = updateDocs.iterator(); iterator.hasNext();) {
                    SolrDocument solrDocument = iterator.next();
                    String childId = (String) solrDocument.getFieldValue(SOLR_ID);
                    reindexChildTreeNode.putExcludedAttachment(id, childId);
                }

                solrResource.deleteDoc(id);
            }

            // Reindex child in tree node
            reindexChildTreeNode.reindex();

            // No specific result needed
            UpdateResponse result = new UpdateResponse();
            return result;
            
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    @Override
    public void delete(Collection<String> idList) throws WikittyException {
        try {
            for (String id : idList) {
                solrServer.deleteById(id);
            }
            solrServer.commit();
        } catch (Exception e) {
            throw new WikittyException(e);
        }
    }
    
    @Override
    public PagedResult<String> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        try {
            // Create query with restriction
            Restriction2Solr restriction2Solr = new Restriction2Solr(transaction, fieldModifier);
            String queryString = restriction2Solr.toSolr(criteria.getRestriction(), solrServer);
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + queryString);

            // Add paged
            int firstIndex = criteria.getFirstIndex();
            int endIndex = criteria.getEndIndex();

            query.setStart(firstIndex);
            int nbRows;
            if (endIndex == -1) {
                // WARNING It is necessary to substract 'start' otherwise, there is a capacity overlow in solR
                nbRows = Integer.MAX_VALUE - firstIndex;
            } else {
                nbRows = endIndex - firstIndex + 1;
            }
            query.setRows(nbRows);

            // Add sorting
            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                for (String sort : sortAscending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.asc);
                }
            }
            
            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                for (String sort : sortDescending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.desc);
                }
            }

            // Add faceting
            List<String> facetField = criteria.getFacetField();
            log.debug("facetField : " + facetField);
            List<Criteria> facetCriteria = criteria.getFacetCriteria();

            // use to map query string to criteria facet name
            Map<String, String> facetQueryToName = new HashMap<String, String>();

            if ((facetField != null && !facetField.isEmpty())
                    || (facetCriteria != null && !facetCriteria.isEmpty())) {
                query.setFacet(true);
                query.setFacetMinCount(1);
                // query.setFacetLimit(8); // no limit actualy

                // field facetisation
                if (facetField != null) {
                    for (String fqfieldName : facetField) {
                        String tranform = fieldModifier.convertToSolr(transaction, fqfieldName);
                        query.addFacetField(tranform);
                    }
                }

                // query facetisation
                if (facetCriteria != null) {
                    for (Criteria facet : facetCriteria) {
                        String queryFacet =
                                restriction2Solr.toSolr(facet.getRestriction());
                        facetQueryToName.put(queryFacet, facet.getName());
                        query.addFacetQuery(queryFacet);
                    }
                }
            }

            QueryResponse resp = solrServer.query(query);
            SolrDocumentList solrResults = resp.getResults();

            Map<String, List<FacetTopic>> facets = new HashMap<String, List<FacetTopic>>();
            if (facetField != null && !facetField.isEmpty()) {
                for (FacetField facet : resp.getFacetFields()) {
                    String facetName = fieldModifier.convertToField(transaction, facet.getName());
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    if (facet.getValues() != null) {
                        for (FacetField.Count value : facet.getValues()) {
                            String topicName = value.getName();
                            if(!topicName.endsWith(TREENODE_EMPTY)) {
                                int topicCount = (int) value.getCount();
                                FacetTopic topic = new FacetTopic(facetName, topicName, topicCount);
                                topics.add(topic);
                            }
                        }
                    }
                    facets.put(facetName, topics);
                }
            }
            if (facetCriteria != null && !facetCriteria.isEmpty()) {
                for (Map.Entry<String, Integer> facet : resp.getFacetQuery().entrySet()) {
                    String facetName = facet.getKey();
                    // don't use contains because, map can have key with null value
                    if (null != facetQueryToName.get(facetName)) {
                        facetName = facetQueryToName.get(facetName);
                    }
                    Integer count = facet.getValue();
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    FacetTopic topic = new FacetTopic(facetName, facetName, count);
                    topics.add(topic);
                    facets.put(facetName, topics);
                }
            }

            List<String> ids = new ArrayList<String>(solrResults.size());
            for (SolrDocument doc : solrResults) {
                String id = (String) doc.getFieldValue(SOLR_ID);
                ids.add(id);
            }

            int numFound = (int)resp.getResults().getNumFound();
            PagedResult<String> result = new PagedResult<String>(
                    firstIndex, numFound, queryString, facets, ids);

            return result;
        } catch (SolrServerException e) {
            throw new WikittyException(e);
        }
    }

    @Override
    public Integer findNodeCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        String wikittyId = w.getId();

        String parent = w.getFieldAsWikitty(TreeNode.EXT_TREENODE, TreeNode.FIELD_TREENODE_PARENT);
        if(parent == null) {
            parent = TREENODE_ROOT;
        } else {
            parent = TREENODE_PREFIX + parent;
        }

        Criteria criteria = Search.query(filter)
                .eq(parent, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(0);
        PagedResult<String> search = findAllByCriteria(transaction, criteria);

        int numFound = search.getNumFound();
        return numFound;
    }


    @Override
    public Map<String, Integer> findAllChildrenCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        String wikittyId = w.getId();
        
        String parent = w.getFieldAsWikitty(TreeNode.EXT_TREENODE, TreeNode.FIELD_TREENODE_PARENT);
        if(parent == null) {
            parent = TREENODE_ROOT;
        } else {
            parent = TREENODE_PREFIX + parent;
        }
        
        // Find count with facet, if the node not contain recurcively content,
        // the node not found with facet
        Criteria criteria = Search.query(filter).eq(parent, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(0)
                .addFacetField(TREENODE_PREFIX + wikittyId);
        PagedResult<String> search = findAllByCriteria(transaction, criteria);

        Map<String, Integer> counts = new HashMap<String, Integer>();
        List<FacetTopic> topics = search.getTopic(TREENODE_PREFIX + wikittyId);
        if(topics != null) {
            for (FacetTopic topic : topics) {
                String topicName = topic.getTopicName();
                int topicCount = topic.getCount();
                counts.put(topicName, topicCount);
            }
        }

        log.debug("Facet result " + counts);

        // Find all children, add the other node not found with facet
        criteria = Search.query().eq(TreeNode.FQ_FIELD_TREENODE_PARENT, wikittyId).criteria()
                .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS);
        search = findAllByCriteria(transaction, criteria);

        List<String> children = search.getAll();
        for (String child : children) {
            if(!counts.containsKey(child)) {
                counts.put(child, 0);
            }
        }

        return counts;
    }

    /**
     * Create all index document to used to modify indexation.
     * this method don't modify index.
     * 
     * The document looks like :
     * SolrId : wikittyId
     * extensions : extensionNames
     * fieldName : fieldValue
     *
     * @param w all wikitties object to index
     * @return solrInputDocument used to modify index
     */
    protected SolrInputDocument createIndexDocument(Wikitty w) {
        log.debug("index wikitty " + w.getId());

        SolrInputDocument doc = new SolrInputDocument();
        String id = w.getId();
        doc.addField(SOLR_ID, id);

        for (String name : w.getExtensionNames()) {
            doc.addField(SOLR_EXTENSIONS, name);
        }

        for (String fqfieldName : w.fieldNames()) {
            FieldType fieldType = w.getFieldType(fqfieldName);
            TYPE type = fieldType.getType();
            String solrFqFieldName = getSolrFieldName(fqfieldName, type);
            
            String[] solrFieldName = solrFqFieldName.split("\\.");
            String solrAllFieldName = SOLR_ALL_EXTENSIONS + "." + solrFieldName[1];
            
            Object objectValue = w.getFqField(fqfieldName);
            if(objectValue != null) {
                if (fieldType.isCollection()) {
                    Collection collectionValue = (Collection) objectValue;
                    for (Object itemValue : collectionValue) {
                        if (itemValue != null) {
                            doc.addField(solrFqFieldName, itemValue);
                            doc.addField(solrAllFieldName, itemValue);

                            // Store string field in differents styles
                            if(type == TYPE.STRING) {
                                doc.addField(solrFqFieldName + "_t", itemValue);
                                doc.addField(solrAllFieldName + "_t", itemValue);
                                String itemValueLowerCase = itemValue.toString().toLowerCase();
                                doc.addField(solrFqFieldName + "_c", itemValueLowerCase);
                                doc.addField(solrAllFieldName + "_c", itemValueLowerCase);
                            }
                            
                            doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                            log.debug("index field " + solrFqFieldName + " with value '" + itemValue + "'");
                        }
                    }
                } else {
                    doc.addField(solrFqFieldName, objectValue);
                    doc.addField(solrAllFieldName, objectValue);

                    // Store string field in differents styles
                    if(type == TYPE.STRING) {
                        doc.addField(solrFqFieldName + "_t", objectValue);
                        doc.addField(solrAllFieldName + "_t", objectValue);
                        String objectValueLowerCase = objectValue.toString().toLowerCase();
                        doc.addField(solrFqFieldName + "_c", objectValueLowerCase);
                        doc.addField(solrAllFieldName + "_c", objectValueLowerCase);
                    }

                    doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                    log.debug("index field " + solrFqFieldName + " with value '" + objectValue + "'");
                }
            }
        }
        return doc;
    }

    /**
     * Find solr document by id
     */
    protected static SolrDocument findById(SolrServer solrServer, String id) {
        SolrQuery query = new SolrQuery(SOLR_ID + ":" + id);
        QueryResponse response;
        try {
            response = solrServer.query(query);
        } catch (SolrServerException eee) {
            throw new WikittyException(eee);
        }

        SolrDocumentList results = response.getResults();
        long numFound = results.getNumFound();
        if(numFound == 1) {
            return results.get(0);
        }

        return null;
    }

    public static String getSolrFieldName(String fqfieldName, TYPE type) {
        switch (type) {
            case BOOLEAN:
                return fqfieldName + "_b";
            case DATE:
                return fqfieldName + "_dt";
            case STRING:
                return fqfieldName + "_s";
            case NUMERIC:
                return fqfieldName + "_d";
            default:
                return fqfieldName;
        }
    }
}
