package org.nuiton.wikitty.generator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.Transformer;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;





/**
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.wikitty.generator.WikittyContractGenerator"
 */
public class WikittyContractGenerator extends ObjectModelTransformerToJava {

    private static final Log log = LogFactory.getLog(WikittyContractGenerator.class);

    @Override
    protected Transformer<ObjectModel, ObjectModel> initPreviousTransformer() {
        return new WikittyPurifierTransformer();
    }

    /** will permet to get the result of a processed class
     * it is useful for dealing with inheritence : when an extension depend
     * of another. In this case the child can find the processed class of the parent
     * and then look into it to find methods and data to copy 
     */
    protected Map<ObjectModelClass, ObjectModelInterface> processedClasses =
            new HashMap<ObjectModelClass, ObjectModelInterface>();

    protected Set<ObjectModelClass> processedEntities = new HashSet<ObjectModelClass>();
    
    @Override
    public void transformFromModel(ObjectModel model) {

        log.info(model.getClasses().size() + " classes to process");
        
        for (ObjectModelClass clazz : model.getClasses()) {
            if (WikittyTransformerUtil.isBusinessEntity(clazz)) {
                processEntity(clazz);
            }
        }

        for (ObjectModelClass clazz : model.getClasses()) {
            if (WikittyTransformerUtil.isMetaExtension(clazz)) {
                processMetaExtension(clazz);
            }
        }
        
        processedClasses.clear();
        processedEntities.clear();
    }
    
    /** contains code commons to entities and meta-extensions */
    protected ObjectModelInterface prepareOutputClass(ObjectModelClass businessEntity) {

        ObjectModelInterface contract = processedClasses.get(businessEntity);

        if (contract == null) {

            log.debug("preparing " + businessEntity.getPackageName() +
                                  "." + businessEntity.getName());

            contract = createInterface(WikittyTransformerUtil.businessEntityToContractName(businessEntity),
                                                            businessEntity.getPackageName());
            addInterface(contract, WikittyTransformerUtil.BUSINESS_ENTITY_CLASS_FQN);

            // TODO 20100811 bleny remove unused imports
            addImport(contract, WikittyTransformerUtil.BUSINESS_ENTITY_CLASS_FQN);
            addImport(contract, WikittyTransformerUtil.BUSINESS_ENTITY_WIKITTY_CLASS_FQN);
            addImport(contract, WikittyTransformerUtil.WIKITTY_CLASS_FQN);
            addImport(contract, "org.nuiton.wikitty.WikittyExtension");
            addImport(contract, "org.nuiton.wikitty.WikittyUtil");
            addImport(contract, "org.nuiton.wikitty.WikittyUser");
            addImport(contract, "org.nuiton.wikitty.WikittyUserAbstract");
            addImport(contract, "org.nuiton.wikitty.WikittyUserImpl");
            addImport(contract, "org.nuiton.wikitty.TreeNode");
            addImport(contract, "org.nuiton.wikitty.TreeNodeAbstract");
            addImport(contract, "org.nuiton.wikitty.TreeNodeImpl");
            addImport(contract, java.util.List.class);
            addImport(contract, java.util.ArrayList.class);
            addImport(contract, java.util.Collection.class);
            addImport(contract, java.util.Collections.class);
            addImport(contract, java.util.Set.class);
            addImport(contract, java.util.Date.class);
            addImport(contract, java.util.LinkedHashSet.class);

            String documentation = businessEntity.getDocumentation();
            if (businessEntity.hasTagValue(WikittyTransformerUtil.TAG_DOCUMENTATION)) {
                documentation += "\n\n";
                documentation += businessEntity.getTagValue(WikittyTransformerUtil.TAG_DOCUMENTATION);
            }

            setDocumentation(contract, documentation);

            processedClasses.put(businessEntity, contract);

        } else {
            log.debug("already prepared : " + businessEntity.getPackageName() +
                                        "." + businessEntity.getName());
        }

        return contract;
    }

    protected void processEntity(ObjectModelClass businessEntity) {
        
        if ( processedEntities.contains(businessEntity) ) {
            log.debug("entity already processed : " + businessEntity.getPackageName() +
                    "." + businessEntity.getName());
            // already processed
            return ;
        }

        log.debug("processing entity : " + businessEntity.getPackageName() +
                "." + businessEntity.getName());

        ObjectModelInterface contract = prepareOutputClass(businessEntity);

        
        // adding public static final String EXT_CLIENT = "Client";
        addConstant(contract,
                "EXT_" + businessEntity.getName().toUpperCase(),
                "String",
                "\"" + businessEntity.getName() + "\"",
                ObjectModelModifier.PUBLIC);

        String extensionVariableName = WikittyTransformerUtil.classToExtensionVariableName(businessEntity, false);

        for(ObjectModelAttribute attribute : businessEntity.getAttributes()) {
            if (attribute.isNavigable()) {
                // two variables needed below
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, false);
                
                // adding constants to contract
                addConstant(contract,
                        fieldVariableName,
                        "String",
                        "\"" + attribute.getName() + "\"",
                        ObjectModelModifier.PUBLIC);
                // adding public static final String FQ_FIELD_CLIENT_NAME = EXT_CLIENT + ".name";
                addConstant(contract,
                        "FQ_" + fieldVariableName,
                        "String",
                        extensionVariableName + " + \"." + attribute.getName() + "\"",
                        ObjectModelModifier.PUBLIC);                
            }
        }
        
        for (ObjectModelAttribute attribute : businessEntity.getAttributes()) {
            if (attribute.isNavigable()) {
                // needed below, in templates
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);
                String attributeType = WikittyTransformerUtil.generateResultType(attribute, false);
    
                String attributeName = attribute.getName();
                if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                    // there is a conflict, purifier transformer give as the right name to use
                    attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                }

                ObjectModelOperation getter;

                if (attribute.getMaxMultiplicity() > 1 || attribute.getMaxMultiplicity() == -1) {
                    // attributed is a collection, we will generate operations get, add, remove and clear
    
                    String attributeTypeSimpleNameInSet = WikittyTransformerUtil.generateResultType(attribute, true);
            
                    // now, for this attribute, we will generate add, remove and clear methods
                    // adding operations to contract
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    getter = addOperation(contract, getterName, attributeTypeSimpleNameInSet);

                    String addName = "add" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation adder = addOperation(contract, addName, "void");
                    addParameter(adder, "String", "element");

                    String removeName = "remove" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation remover = addOperation(contract, removeName, "void");
                    addParameter(remover, "String", "element");

                    String clearName = "clear" + StringUtils.capitalize(attributeName);
                    addOperation(contract, clearName, "void");

                } else {
                    // attribute is not a collection, we generate a getter and a setter
                    String getterName = "get" + StringUtils.capitalize(attributeName);
                    getter = addOperation(contract, getterName, attributeType);
                    
                    String setterName = "set" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation setter = addOperation(contract, setterName, "void");
                    addParameter(setter, attributeType, attributeName);
                }

                setDocumentation(getter, attribute.getDocumentation());
            }
        }

        // now, add to this contract all operation due to inheritence from
        // other business entities
        Collection<ObjectModelClass> superClasses = businessEntity.getSuperclasses();
        for (ObjectModelClass superClass : superClasses) {
            addInterface(contract, superClass.getQualifiedName()); // extends ?
            if (WikittyTransformerUtil.isBusinessEntity(superClass)) {
                // superclass must have been processed first to have its operations set
                if ( ! processedEntities.contains(superClass)) {
                    log.debug(businessEntity.getName() + " require to process " +
                              superClass.getName() + " first");
                    processEntity(superClass);
                }

                // getting the signatures of those operations 
                for (ObjectModelOperation operation : processedClasses.get(superClass).getOperations()) {
                    cloneOperationSignature(operation, contract, true);
                }
            }
        }

        processedEntities.add(businessEntity);
    }
    
    /** add stuff if input model element is stereotyped as "meta" */
    protected void processMetaExtension(ObjectModelClass metaExtension) {
        log.debug("processing meta-extension : " + metaExtension.getPackageName() +
                "." + metaExtension.getName());

        ObjectModelInterface contract = prepareOutputClass(metaExtension);

        ObjectModelOperation addMetaExtension = addOperation(contract, "addMetaExtension", "void");
        addParameter(addMetaExtension, WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN, "extension");
        setDocumentation(addMetaExtension, String.format(
                "add %s meta-extension on given extension to this entity",
                metaExtension.getName()));

    }
}
