/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: TreeNodeResult.java 779 2011-04-11 15:48:19Z jcouteau $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.1.1/wikitty-api/src/main/java/org/nuiton/wikitty/search/TreeNodeResult.java $
 * %%
 * Copyright (C) 2009 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.search;


import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;
import javax.swing.tree.DefaultMutableTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * L'iteration se fait en profondeur
 *
 * @author poussin
 * @version $Revision: 779 $
 * @since 3.1
 * 
 * Last update: $Date: 2011-04-11 17:48:19 +0200 (lun., 11 avril 2011) $
 * by : $Author: jcouteau $
 */
public class TreeNodeResult<T> extends DefaultMutableTreeNode implements Iterable<TreeNodeResult<T>> {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(TreeNodeResult.class);

    private static final long serialVersionUID = 31L;

    /**
     * Visitor for TreeNodeResult
     * @param <T>
     */
    static public interface Visitor<T> {
        /**
         *
         * @param node node to visit
         * @return if true visit this element
         */
        public boolean visitEnter(TreeNodeResult<T> node);
        /**
         *
         * @param node node visited
         * @return if true visit next element
         */
        public boolean visitLeave(TreeNodeResult<T> node);
    }

    protected int attCount;

    /**
     *
     * @param object L'id, le wikitty ou le BusinessEntity suivant le type de T
     * @param attCount le nombre d'attachment pour ce noeud (avec les sous noeud)
     */
    public TreeNodeResult(T object, int attCount) {
        super(object);
        this.attCount = attCount;
    }

    /**
     * Visite en profondeur de l'arbre, il est possible d'arreter la visite
     * soit en entrant dans le noeud soit en sortant du noeud, si respectivement
     * visitEnter ou visitLeave retourne false.
     * 
     * @param visitor
     */
    public boolean acceptVisitor(Visitor<T> visitor) {
        if (visitor.visitEnter(this)) {
            for (Enumeration e = children(); e.hasMoreElements();) {
                TreeNodeResult<T> child = (TreeNodeResult<T>) e.nextElement();
                if (!child.acceptVisitor(visitor)) {
                    break;
                }
            }
        }
        boolean result = visitor.visitLeave(this);
        return result;
    }

    /**
     * Get direct children of this node
     *
     * @return
     */
    public List<TreeNodeResult<T>> getChildren() {
        List<TreeNodeResult<T>> result;
        if (children == null) {
            result = Collections.emptyList();
        } else {
            result = Collections.unmodifiableList(
                    new ArrayList<TreeNodeResult<T>>(children));
        }
        return result;
    }

    /**
     * Iterate on all children or sub-children, in depth first
     * @return
     */
    @Override
    public Iterator<TreeNodeResult<T>> iterator() {
        Iterator<TreeNodeResult<T>> result = new Iterator<TreeNodeResult<T>>() {

            protected Enumeration enumDepth = TreeNodeResult.this.depthFirstEnumeration();
            
            @Override
            public boolean hasNext() {
                return enumDepth.hasMoreElements();
            }

            @Override
            public TreeNodeResult<T> next() {
                TreeNodeResult<T> result = (TreeNodeResult<T>)enumDepth.nextElement();
                return result;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("Not supported yet.");
            }
        };
        return result;
    }

    /**
     * Retourne l'objet associe avec ce noeud (id, wikitty ou BusinessEntity)
     * @return l'objet associe avec ce noeud (id, wikitty ou BusinessEntity)
     */
    public T getObject() {
        return (T)getUserObject();
    }

    /**
     * Return TreeNodeResult where object in TreeNodeResult equals child in
     * parameter
     * @param child
     * @return
     */
    public TreeNodeResult<T> getChild(T child) {
        TreeNodeResult<T> result = null;
        if (child != null) {
            for (Enumeration e = children(); e.hasMoreElements();) {
                TreeNodeResult<T> r = (TreeNodeResult<T>) e.nextElement();
                if (child.equals(r.getObject())) {
                    result = r;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Retourn le nombre d'attachment pour ce noeud (avec les sous noeud)
     * @return
     */
    public int getAttCount() {
        return attCount;
    }

}
