/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import javax.swing.JTree;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;

/**
 * To help getting and setting navigation tree objects from a {@link JAXXContext}.
 * <p/>
 * There is six types of data which can be hold in a context :
 * <ul>
 * <li>tree : the tree </li>
 * <li>tree model : the navigation tree model</li>
 * <li>tree handler : the navigation tree handler</li>
 * <li>selected path : the navigation path of the selected node</li>
 * <li>selected node : the selected node</li>
 * <li>selected bean : the selected bean</li>
 * <li>selected ui : the selected content ui</li>
 * </ul>
 * <p/>
 * To make possible the use of more than one navigation tree system in a same
 * context, we <b>MUST</b> distinguish the context entries definition. For this
 * purpose, entries definition are normalized and prefixed by a unique {@link #prefix}.
 * <p/>
 * Here is the keys mapping :
 * <ul>
 * <li>tree : {@code prefix + "-tree"}</li>
 * <li>tree model : {@code prefix + "-tree-model"}</li>
 * <li>tree handler : {@code prefix + "-tree-handler"}</li>
 * <li>selected path : {@code prefix + "-selected-path"}</li>
 * <li>selected node : {@code prefix + "-selected-node"}</li>
 * <li>selected bean : {@code prefix + "-selected-bean"}</li>
 * <li>selected ui: {@code prefix + "-selected-ui"}</li>
 * </ul>
 *
 * @author chemit
 * @since 1.7.2
 */
public class NavigationTreeContextHelper {

    protected final String prefix;
    protected JAXXContextEntryDef<String> selectedPathContextEntry;
    protected JAXXContextEntryDef<Object> selectedBeanContextEntry;
    protected JAXXContextEntryDef<NavigationTreeNode> selectedNodeContextEntry;
    protected JAXXContextEntryDef<NavigationTreeModel> treeModelContextEntry;
    protected JAXXContextEntryDef<NavigationTreeHandler> treeHandlerContextEntry;
    protected JAXXContextEntryDef<JTree> treeContextEntry;
//    protected JAXXContextEntryDef<Component> selectedUIContextEntry;

    public NavigationTreeContextHelper(String prefix) {
        this.prefix = prefix;
        treeContextEntry = JAXXUtil.newContextEntryDef(prefix + "-tree", JTree.class);
        treeModelContextEntry = JAXXUtil.newContextEntryDef(prefix + "-tree-model", NavigationTreeModel.class);
        treeHandlerContextEntry = JAXXUtil.newContextEntryDef(prefix + "-tree-handler", NavigationTreeHandler.class);
        selectedBeanContextEntry = JAXXUtil.newContextEntryDef(prefix + "-selected-bean", Object.class);
        selectedNodeContextEntry = JAXXUtil.newContextEntryDef(prefix + "-selected-node", NavigationTreeNode.class);
        selectedPathContextEntry = JAXXUtil.newContextEntryDef(prefix + "-selected-path", String.class);
//        selectedUIContextEntry = JAXXUtil.newContextEntryDef(prefix + "-selected-ui", Component.class);
    }

    public String getPrefix() {
        return prefix;
    }

    public JTree getTree(JAXXContext context) {
        JTree r = getTreeContextEntry().getContextValue(context);
        return r;
    }

    public NavigationTreeModel getTreeModel(JAXXContext context) {
        NavigationTreeModel r = getTreeModelContextEntry().getContextValue(context);
        return r;
    }

    public NavigationTreeHandler getTreeHandler(JAXXContext context) {
        NavigationTreeHandler r = getTreeHandlerContextEntry().getContextValue(context);
        return r;
    }

    public String getSelectedPath(JAXXContext context) {
        String r = getSelectedPathContextEntry().getContextValue(context);
        return r;
    }

    public NavigationTreeNode getSelectedNode(JAXXContext context) {
        NavigationTreeNode r = getSelectedNodeContextEntry().getContextValue(context);
        return r;
    }

    public Object getSelectedBean(JAXXContext context) {
        Object r = getSelectedBeanContextEntry().getContextValue(context);
        return r;
    }

//    public Component getSelectedUI(JAXXContext context) {
//        Component r = getSelectedUIContextEntry().getContextValue(context);
//        return r;
//    }

    public void setTreeModel(JAXXContext context, NavigationTreeModel model) {
        getTreeModelContextEntry().setContextValue(context, model);
    }

    public void setTree(JAXXContext context, JTree tree) {
        getTreeContextEntry().setContextValue(context, tree);
    }

    public void setTreeHandler(JAXXContext context, NavigationTreeHandler handler) {
        getTreeHandlerContextEntry().setContextValue(context, handler);
    }

    public void setSelectedPath(JAXXContext context, String path) {
        if (path == null) {
            getSelectedPathContextEntry().removeContextValue(context);
        } else {
            getSelectedPathContextEntry().setContextValue(context, path);
        }
    }

    public void setSelectedNode(JAXXContext context, NavigationTreeNode node) {
        if (node == null) {
            getSelectedNodeContextEntry().removeContextValue(context);
        } else {
            getSelectedNodeContextEntry().setContextValue(context, node);
        }
    }

    public void setSelectedBean(JAXXContext context, Object bean) {
        if (bean == null) {
            getSelectedBeanContextEntry().removeContextValue(context);
        } else {
            getSelectedBeanContextEntry().setContextValue(context, bean);
        }
    }

//    public void setSelectedUI(JAXXContext context, Component ui) {
//        if (ui == null) {
//            getSelectedUIContextEntry().removeContextValue(context);
//        } else {
//            getSelectedUIContextEntry().setContextValue(context, ui);
//        }
//    }

    protected JAXXContextEntryDef<NavigationTreeModel> getTreeModelContextEntry() {
        return treeModelContextEntry;
    }

    protected JAXXContextEntryDef<NavigationTreeHandler> getTreeHandlerContextEntry() {
        return treeHandlerContextEntry;
    }

    protected JAXXContextEntryDef<Object> getSelectedBeanContextEntry() {
        return selectedBeanContextEntry;
    }

    protected JAXXContextEntryDef<NavigationTreeNode> getSelectedNodeContextEntry() {
        return selectedNodeContextEntry;
    }

    protected JAXXContextEntryDef<String> getSelectedPathContextEntry() {
        return selectedPathContextEntry;
    }

    protected JAXXContextEntryDef<JTree> getTreeContextEntry() {
        return treeContextEntry;
    }

//    public JAXXContextEntryDef<Component> getSelectedUIContextEntry() {
//        return selectedUIContextEntry;
//    }
}
