/**
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.i18n.plugin;

import org.nuiton.util.SourceEntry;
import org.apache.maven.plugin.logging.SystemStreamLog;

import java.beans.Introspector;
import java.io.File;
import org.nuiton.util.PluginHelper;
/**
 * Le logger utilisé par les mojo.
 *
 * @author chemit
 * @since 0.9
 */
public class I18nLogger extends SystemStreamLog {

    /** l'entrée en cours de traitement (pour les parseurs) */
    private SourceEntry entry;

    /** le fichier en cours de traitement (pour les parseurs) */
    protected File file;

    /** le prefix du mojo courant a ajouter dans les logs. */
    protected String parser;

    public I18nLogger(AbstractI18nPlugin parser) {
        this.parser = "i18n:" + Introspector.decapitalize(parser.getClass().getSimpleName()) + " on " + parser.getArtifactId();
    }

    @Override
    public void info(CharSequence content) {
        print(0, "INFO", null, content.toString());
    }

    @Override
    public void debug(CharSequence content) {
        print(0, "DEBUG", null, content.toString());
    }

    public void infoEntry(String action, CharSequence content) {
        print(0, "INFO", action, entry.toString() + (content == null ? "" : " - " + content.toString()));
    }

    public void infoFile(String action, String content) {
        print(2, "INFO", action, file.toString() + (content == null ? "" : " - " + content));
    }

    public void infoAction(String action, String content) {
        print(2, "INFO", action, (content == null ? "" : " - " + content));
    }

    private void print(int start, String prefix, String context, String content) {
        StringBuilder sb = new StringBuilder();
        sb.append("[").append(prefix).append("] [").append(parser).append("] ");

        for (int i = 0; i < start; i++) {
            sb.append(' ');
        }
        if (context != null) {
            sb.append("<").append(context).append("> ");
        }
        sb.append(content);
        System.out.println(sb.toString());
    }

    public void setEntry(SourceEntry entry) {
        this.entry = entry;
    }

    /**
     * Construit une chaine de log formatée.
     *
     * @param msg     le prefix du message
     * @param nbFiles le nombre de fichiers actuellement traités
     * @param time    le time de traitement de ce fichier
     * @param all     le temps de traitement de tous les fichiers
     * @return la chaine de log formatée
     */
    public String getLogEntry(String msg, int nbFiles, long time, long all) {
        long now = System.nanoTime();
        long delta = now - time;
        String s = msg;
        if (time > 0) {
            s += " (" + PluginHelper.convertTime(delta) + ")";
        }
        if (all > 0) {
            s += "(total time:" + PluginHelper.convertTime(now - all) + ")";
        }
        if (nbFiles > 0) {
            s += " ( ~ " + PluginHelper.convertTime(((now - all) / (nbFiles))) + " / file)";
        }
        return s;
    }

    public void setFile(File file) {
        this.file = file;
    }

}
