/**
 * *##% guix-compiler-swing
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.generator;

//~--- non-JDK imports --------------------------------------------------------
import java.beans.IntrospectionException;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.io.Reader;
import org.nuiton.guix.tags.swing.MenuBarHandler;
import org.nuiton.guix.tags.swing.SplitPanelHandler;
import org.nuiton.guix.tags.swing.TreeHandler;
import org.nuiton.guix.model.GuixModelObject;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.nuiton.guix.BindingUtils;
import org.nuiton.guix.SwingEventHandler;
import org.nuiton.guix.model.AttributeDescriptor;
import org.nuiton.guix.model.Rule;
import org.nuiton.guix.model.Selector;
import org.nuiton.guix.model.StyleSheet;
import org.nuiton.guix.tags.ScriptHandler;
import org.nuiton.guix.tags.ScriptHandler.ScriptPart;
import org.nuiton.guix.tags.TagHandler;
import org.nuiton.guix.tags.TagManager;
import org.nuiton.guix.tags.swing.CellHandler;
import org.nuiton.guix.tags.swing.RowHandler;
import org.nuiton.guix.tags.swing.TableHandler;

/**
 * Generates a swing abstract class
 *
 * @author morin
 */
public class SwingAbstractClassGenerator extends SwingJavaFileGenerator {

    private static final String CONSTRAINT_ATTRIBUTE = "constraints";
    private static final String SETTER_PATTERN = "%1$s oldValue = this.%2$s;\n%3$s\nfirePropertyChange(\"%2$s\", oldValue, %4$s);";
    private static final String SWING_DEFAULT_PACKAGE = "javax.swing";

    /** Stylesheets applied to the objects of the class */
    List<StyleSheet> styleSheets = new ArrayList<StyleSheet>();
    /** map of the different parts of the script associated with gmo */
    Map<ScriptPart, Object> script;
    /** Bindings to generate */
    Map<String,Map<String, String>> bindings2Generate = new HashMap<String,Map<String, String>>();

    SwingGenerator swingGenerator;

    /**
     * Constructor
     *
     * @param gmo GuixModelObject representing the class to generate
     * @param classes List of the classes to generate
     */
    public SwingAbstractClassGenerator(GuixModelObject gmo, Map<String, JavaFile> classes) {
        super(gmo, classes);
        jf = new JavaFile(Modifier.PUBLIC | Modifier.ABSTRACT,
                JavaFile.ABSTRACT_CLASS,
                gmo.getClassDescriptor().getPackageName(),
                gmo.getClassDescriptor().getName() + "Abstract",
                null, gmo.getJavadoc());
        
        script = ScriptHandler.decomposeScript(gmo.getClassDescriptor().getScript());
    }

    public void setSwingGenerator(SwingGenerator swingGenerator) {
        this.swingGenerator = swingGenerator;
    }

    @Override
    public JavaFile generate() {
        super.addImports();

        //add script imports
        for (String s : (List<String>) script.get(ScriptPart.IMPORTS)) {
            jf.addImport(s);
        }

        //add script fields
        for(JavaField field : (List<JavaField>) script.get(ScriptPart.FIELDS)) {
            jf.addField(field, false);           
        }

        //add script bodycode
        jf.addBodyCode((String) script.get(ScriptPart.BODYCODE));

        try {
            //get creation, setting and displying methods content
            Map<Method, String> methodBodies = browseModelObjects(gmo);
            // get the script initializers
            String scriptInitializers = ScriptHandler.postProcessInitializers((String)script.get(ScriptPart.INITIALIZERS), jf.getAllFields());

            //add constructor
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, null, gmo.getClassDescriptor().getName() + "Abstract",
                    null, null, "super(" + (gmo.getConstructor() != null ? gmo.getConstructor() : "") + ");" + (!jf.isSuperclassIsGuixObject() ? "\ninitialize();" : ""), "Constructor"));

            //add initialization method
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "initialize", null, null,
                    getMethodName(Method.BEFORE_CREATION) + "();\n" + getMethodName(Method.COMPONENTS_CREATION) + "();\n" +
                    (scriptInitializers != null ? scriptInitializers + "\n" : "") +
                    getMethodName(Method.BEFORE_SETTINGS) + "();\n" + getMethodName(Method.COMPONENTS_SETTINGS) + "();\n" +
                    getMethodName(Method.BEFORE_TREE) + "();\n" + getMethodName(Method.COMPONENTS_TREE) + "();\n" +
                    getMethodName(Method.BEFORE_BINDING) + "();\n"+ getMethodName(Method.DATABINDING_INIT) + "();\n" + getMethodName(Method.IN_THE_END) + "();",
                    "initialization"));

            //add overridable methods
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_CREATION), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_CREATION) + "();\n" : "", "Method called before the creation of the components"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_TREE), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_TREE) + "();\n" : "", "Method called before the creation of the component tree"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_SETTINGS), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_SETTINGS) + "();\n" : "", "Method called before the settings of the components"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_BINDING), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_BINDING) + "();\n" : "", "Method called before the binding"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.IN_THE_END), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.IN_THE_END) + "();\n" : "", "Method called in the end of the initialization"));

            //add creation and initialization methods
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_CREATION), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_CREATION) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_CREATION), "Components creation"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_SETTINGS), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_SETTINGS) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_SETTINGS), "Components settings"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_TREE), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_TREE) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_TREE), "components layout"));

            //add databinding method
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.DATABINDING_INIT), null, null,
                     jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.DATABINDING_INIT) + "();\n" :
                         "", "initilization of databinding"));
            
            //add script methods
            for (JavaMethod m : (List<JavaMethod>) script.get(ScriptPart.METHODS)) {
                ScriptHandler.postProcessMethodBody(m, jf.getAllFields());
                //checks if the method override a generated method
                if ((m.getName().equals(getMethodName(Method.BEFORE_BINDING)) || m.getName().equals(getMethodName(Method.BEFORE_CREATION)) || m.getName().equals(getMethodName(Method.BEFORE_SETTINGS)) || m.getName().equals(getMethodName(Method.BEFORE_TREE)) || m.getName().equals(getMethodName(Method.IN_THE_END))) && m.getReturnType() != null && m.getReturnType().equals("void") && m.getArguments() == null) {
                    int i = 0;
                    boolean methodFound = false;
                    while (i < jf.getMethods().length && !methodFound) {
                        methodFound = jf.getMethods()[i].getName().equals(m.getName()) && jf.getMethods()[i].getReturnType() != null && jf.getMethods()[i].getReturnType().equals("void") && jf.getMethods()[i].getArguments() == null;
                        i++;
                    }
                    //if the method already exists, add the code to the code already defined
                    if (methodFound) {
                        jf.getMethods()[i - 1].appendBodyCode(m.getBodyCode(), "\n");
                    }
                    //else create the method
                    else {
                        jf.addMethod(m);
                    }
                }
                else {
                    jf.addMethod(m);
                }
            }

            //Modify the setters by adding the firePropertyChange call
            for(JavaMethod method : jf.getMethods()) {
                if(method.getName().startsWith("set") && method.getName().length() > 3
                        && Character.isUpperCase(method.getName().charAt(3)) && method.getArguments() != null) {
                    method.setBodyCode(String.format(SETTER_PATTERN, method.getArguments()[0].getType(),
                            Character.toLowerCase(method.getName().charAt(3)) + method.getName().substring(4),
                            method.getBodyCode(), method.getArguments()[0].getName()));
                }
            }
        }
        catch (ClassNotFoundException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
        catch (ClassCastException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
        catch (NullPointerException eee) {
            eee.printStackTrace();
        }
        return jf;
    }

    /**
     * Browse the GuixModelObjects to add the creation, initialization and bindings methods to the file to gnerate
     *
     * @param gmo the GuixModelObject to analyze
     * @return A map containing the creation, initialization and bindings methods
     */
    private Map<Method, String> browseModelObjects(GuixModelObject gmo) throws ClassNotFoundException, ClassCastException {
        StringBuffer componentsCreation = new StringBuffer("");
        StringBuffer componentsSettings = new StringBuffer("");
        StringBuffer componentsTree = new StringBuffer("");
        SwingEventHandler seh = new SwingEventHandler();
        //the class represented by gmo
        Class clazz = null;
        //the layout constraint to place gmo
        String constraint = null;
        //do we need to overcharge the field
        boolean createField = true;
        //id with the first letter capitalized
        String capitalizedId = (gmo.getId().length() > 0) ? Character.toUpperCase(gmo.getId().charAt(0)) + gmo.getId().substring(1) : gmo.getId();
            
        //if gmo represents a table tag
        if ((gmo.getClassDescriptor().getSuperClass() != null
                && gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("Table"))
                || (gmo.getClassDescriptor().getPackageName() == null
                && gmo.getClassDescriptor().getName().equalsIgnoreCase("Table"))) {
            //matrix of the gridbaglayout : the free zones are represented by null or false
            List<List<Boolean>> layout = new ArrayList<List<Boolean>>();
            StringBuffer settingsMethod = new StringBuffer();
            //Checks in the table tag if any attribute defines constraints for the children
            TableHandler tableHandler = new TableHandler(gmo);
            clazz = tableHandler.getClassToGenerate();
            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't hav the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = Class.forName(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    throw new ClassCastException();
                }
            }
            if(createField) {
                jf.addField(new JavaField(Modifier.PROTECTED,
                                clazz.getName(),
                                gmo.getId(), gmo.getJavadoc(), tableHandler),
                                gmo.isJavaBean());
            
                //the layout of the table is the GridBagLayout
                componentsCreation.append(gmo.getId()).append(" = new ")
                        .append(clazz.getName()).append("(new java.awt.GridBagLayout());\n");
                if(jf.getInheritedField(gmo.getId()) != null) {
                    componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
            }
            else {
                settingsMethod.append(gmo.getId()).append(".setLayout(new java.awt.GridBagLayout());\n");
            }
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }

            constraint = processAttributes(clazz, gmo, settingsMethod, seh, tableHandler);

            //add the component to its parent
            if(gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null) {
                componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
            }
            else if(!ignoreTag(gmo)) {
                if((gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("ScrollPanel"))
                        || gmo.getParent().getClassDescriptor().toString().equals("javax.swing.JScrollPane")) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".setViewportView(").append(gmo.getId()).append(");\n");
                }
                else {
                    componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".add(").append(gmo.getId());
                    //if the constraints exists
                    if (constraint != null) {
                        componentsTree.append(",").append(constraint);
                    }
                    componentsTree.append(");\n");
                }
            }

            for (int r = 0; r < gmo.getChildren().size(); r++) {
                GuixModelObject row = gmo.getChildren().get(r);
                //the children of a table must be rows
                if (!row.getClassDescriptor().getName().equalsIgnoreCase("Row")) {
                    if (log.isErrorEnabled()) {
                        log.error("Table children must be Rows !");
                    }
                }
                else {
                    //Checks in the row tag if any attribute defines constraints for the children
                    RowHandler rowHandler = new RowHandler(row, tableHandler);
                    int iC = 0;
                    for (int c = 0; c < row.getChildren().size(); c++) {
                        GuixModelObject cell = row.getChildren().get(c);
                        //the children of a row must be cells
                        if (!cell.getClassDescriptor().getName().equalsIgnoreCase("Cell")) {
                            if (log.isErrorEnabled()) {
                                log.error("Row children must be Cells !");
                            }
                        }
                        //the cell can contain only one component
                        else if (cell.getChildren().size() != 1) {
                            if (log.isErrorEnabled()) {
                                log.error("Cells must contain only one tag !");
                            }
                        }
                        else {
                            CellHandler cellHandler = new CellHandler(cell, rowHandler);
                            componentsTree.append(TableHandler.createCell(gmo.getId(), cellHandler, layout, iC, r));

                            //browse the children of the child of the cell
                            Map<Method, String> methodBodies = browseModelObjects(cell.getChildren().get(0));
                            componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                            componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                            componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
                        }
                    }
                }
            }
            //create the table
            componentsSettings.append(gmo.getId()).append("Settings();\n");
                
        }
        else if ((gmo.getClassDescriptor().getSuperClass() != null && gmo.getClassDescriptor().getSuperClass().getPackageName() == null && gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("Tab"))
                || (gmo.getClassDescriptor().getPackageName() == null && gmo.getClassDescriptor().getName().equalsIgnoreCase("Tab"))) {
            //if the parent has a superclass and this superclass is a TabPanel or if the parent is null or if the parent is a TabPanel
            if(((gmo.getParent().getClassDescriptor().getSuperClass() != null
                    && ((gmo.getParent().getClassDescriptor().getSuperClass().getPackageName() == null
                    && (gmo.getParent().getClassDescriptor().getSuperClass().getName().equals("TabPanel") || gmo.getParent().getClassDescriptor().getSuperClass().getName().equals("JTabbedPane")))
                    || (gmo.getParent().getClassDescriptor().getSuperClass().toString().equals("javax.swing.JTabbedPane"))))
                    ||(gmo.getParent() == null
                    || (gmo.getParent().getClassDescriptor().getPackageName() == null
                    && (gmo.getParent().getClassDescriptor().getName().equals("TabPanel") || gmo.getParent().getClassDescriptor().getName().equals("JTabbedPane")))
                    || (gmo.getParent().getClassDescriptor().toString().equals("javax.swing.JTabbedPane"))))
                    && gmo.getChildren().size() == 1) {
                String title = null;
                for(AttributeDescriptor attribute : gmo.getAttributeDescriptors()) {
                    if(attribute.getName().equals("title")) {
                        title = attribute.getValue();
                        break;
                    }
                }

                //add the component to its parent
                componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".addTab(")
                            .append(title != null ? title : "\"" + gmo.getId() + "\"").append(",").append(gmo.getChildren().get(0).getId()).append(");\n");
                
                Map<Method, String> methodBodies = browseModelObjects(gmo.getChildren().get(0));
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            else if(log.isErrorEnabled()) {
                log.error("Tab tags can only be children of TabPanel or JTabbedPane tags, and can have only one child.");
            }
        }
         //if gmo represents a menubar tag
        else if ((gmo.getClassDescriptor().getSuperClass() != null
                && ((gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && (gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("MenuBar") || gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("JMenuBar")))
                || gmo.getClassDescriptor().getSuperClass().toString().equals("javax.swing.JMenuBar")))
                || ((gmo.getClassDescriptor().getPackageName() == null
                && (gmo.getClassDescriptor().getName().equalsIgnoreCase("MenuBar") || gmo.getClassDescriptor().getName().equalsIgnoreCase("JMenuBar")))
                || gmo.getClassDescriptor().toString().equals("javax.swing.JMenuBar"))) {
            StringBuffer settingsMethod = new StringBuffer();
            MenuBarHandler menuBarHandler = new MenuBarHandler();
            clazz = menuBarHandler.getClassToGenerate();
            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't hav the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = Class.forName(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    throw new ClassCastException();
                }
            }

            if(createField) {
                jf.addField(new JavaField(Modifier.PROTECTED,
                                clazz.getName(),
                                gmo.getId(), gmo.getJavadoc(), menuBarHandler),
                                gmo.isJavaBean());
            
                componentsCreation.append(gmo.getId()).append(" = new ")
                        .append(clazz.getName()).append("();\n");
                if(jf.getInheritedField(gmo.getId()) != null) {
                    componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
            }
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }
            constraint = processAttributes(clazz, gmo, settingsMethod, seh, menuBarHandler);

            //add the component to its parent
            componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".setJMenuBar(").append(gmo.getId()).append(");\n");
           
            for (GuixModelObject child : gmo.getChildren()) {
                //browse the children of the child of the menubar
                Map<Method, String> methodBodies = browseModelObjects(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            //create the menu
            componentsSettings.append(gmo.getId()).append("Settings();\n");
        }
        //if the tag is a SplitPanel (Vertical or Horizontal)
        else if((gmo.getClassDescriptor().getSuperClass() != null
                && ((gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && (gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("VerticalSplitPanel")
                || gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("HorizontalSplitPanel")
                || gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("JSplitPane")))
                || gmo.getClassDescriptor().getSuperClass().toString().equals("javax.swing.JSplitPane")))
                || ((gmo.getClassDescriptor().getPackageName() == null
                && (gmo.getClassDescriptor().getName().equalsIgnoreCase("VerticalSplitPanel")
                || gmo.getClassDescriptor().getName().equalsIgnoreCase("HorizontalSplitPanel")
                || gmo.getClassDescriptor().getName().equalsIgnoreCase("JSplitPane")))
                || gmo.getClassDescriptor().toString().equals("javax.swing.JSplitPane"))) {

            StringBuffer settingsMethod = new StringBuffer();
            SplitPanelHandler splitPanelHandler = new SplitPanelHandler();
            clazz = splitPanelHandler.getClassToGenerate();

            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't hav the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = Class.forName(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    throw new ClassCastException();
                }
            }

            if(createField) {
                jf.addField(new JavaField(Modifier.PROTECTED,
                                clazz.getName(),
                                gmo.getId(), gmo.getJavadoc(), splitPanelHandler),
                                gmo.isJavaBean());

                componentsCreation.append(gmo.getId()).append(" = new ")
                        .append(clazz.getName()).append("(").append(clazz.getName())
                        .append(gmo.getClassDescriptor().getName().equalsIgnoreCase("VerticalSplitPanel") ? ".VERTICAL_SPLIT" : ".HORIZONTAL_SPLIT").append(");\n");
            }
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }
            constraint = processAttributes(clazz, gmo, settingsMethod, seh, splitPanelHandler);

            //add the component to its parent
            if(gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null) {
                componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
            }
            else if(!ignoreTag(gmo)) {
                if((gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("ScrollPanel"))
                        || gmo.getParent().getClassDescriptor().toString().equals("javax.swing.JScrollPane")) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".setViewportView(").append(gmo.getId()).append(");\n");
                }
                else {
                    componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".add(").append(gmo.getId());
                    //if the constraints exists
                    if (constraint != null) {
                        componentsTree.append(",").append(constraint);
                    }
                    componentsTree.append(");\n");
                }
            }

            for (GuixModelObject child : gmo.getChildren()) {
                Map<Method, String> methodBodies = browseModelObjects(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            //create the splitpanel
            componentsSettings.append(gmo.getId()).append("Settings();\n");
        }
        //if the tag represents a tree
        else if((gmo.getClassDescriptor().getSuperClass() != null
                && ((gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && (gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("Tree") || gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("JTree")))
                || gmo.getClassDescriptor().getSuperClass().toString().equals("javax.swing.JTree")))
                ||((gmo.getClassDescriptor().getPackageName() == null
                && (gmo.getClassDescriptor().getName().equalsIgnoreCase("Tree") || gmo.getClassDescriptor().getName().equalsIgnoreCase("JTree")))
                ||gmo.getClassDescriptor().toString().equals("javax.swing.JTree"))) {

            StringBuffer settingsMethod = new StringBuffer();
            TreeHandler treeHandler = new TreeHandler();
            clazz = treeHandler.getClassToGenerate();

            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't have the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = Class.forName(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    throw new ClassCastException();
                }
            }
            //browse the children first to be able to ad them to the list after their creation
            for (GuixModelObject child : gmo.getChildren()) {
                Map<Method, String> methodBodies = browseModelObjects(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }

            if(createField) {
                jf.addField(new JavaField(Modifier.PRIVATE,
                                clazz.getName(),
                                gmo.getId(), gmo.getJavadoc(), treeHandler),
                                gmo.isJavaBean());

                if(gmo.getChildren().size() == 1) {
                    componentsCreation.append(gmo.getId()).append(" = new ")
                            .append(clazz.getName()).append("(").append(gmo.getChildren().get(0).getId()).append(");\n");
                }
                else {
                    componentsCreation.append("java.util.Vector<javax.swing.tree.TreeNode> ")
                            .append(gmo.getId()).append("Nodes = new java.util.Vector<javax.swing.tree.TreeNode>();\n");
                    for(GuixModelObject child : gmo.getChildren()) {
                        componentsCreation.append(gmo.getId()).append("Nodes.add(")
                                .append(child.getId()).append(");\n");
                    }
                    componentsCreation.append(gmo.getId()).append(" = new ")
                            .append(clazz.getName()).append("(").append(gmo.getId()).append("Nodes);\n");
                }
            }
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }
            constraint = processAttributes(clazz, gmo, settingsMethod, seh, treeHandler);

            //add the component to its parent
            if(gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null) {
                componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
            }
            else if(!ignoreTag(gmo)) {
                if((gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("ScrollPanel"))
                        || gmo.getParent().getClassDescriptor().toString().equals("javax.swing.JScrollPane")) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".setViewportView(").append(gmo.getId()).append(");\n");
                }
                else {
                    componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".add(").append(gmo.getId());
                    //if the constraints exists
                    if (constraint != null) {
                        componentsTree.append(",").append(constraint);
                    }
                    componentsTree.append(");\n");
                }
            }

            //create the splitpanel
            componentsSettings.append(gmo.getId()).append("Settings();\n");
        }
        else {
            //TagHandler of gmo's class
            TagHandler th = null;
            //TagHandler of gmo's first not generated superclass
            TagHandler superTh = null;
            //JavaFile representing the class of the field (if not a real class)
            JavaFile jFile = null;
            //gmo's first not generated superclass
            Class superClazz = null;
            StringBuffer settingsMethod = new StringBuffer();
            //List of gmo's generated superclasses
            List<String> generatedSuperClasses = new ArrayList<String>();

            //check if gmo's class has a TagHandler and get the class to generate instead
            if(TagManager.getGuixClassHandler(gmo.getClassDescriptor().getName()) != null) {
                th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().getName());
                if(th != null) {
                    clazz = th.getClassToGenerate();
                }
            }
            //check if gmo's class has a TagHandler and get the class to generate instead
            else {
                 if(gmo.getClassDescriptor().getPackageName() == null) {
                    gmo.getClassDescriptor().setPackageName(SWING_DEFAULT_PACKAGE);
                }
                if(TagManager.getGuixClassHandler(gmo.getClassDescriptor().toString()) != null) {
                    th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().toString());
                    if(th != null) {
                        clazz = th.getClassToGenerate();
                    }
                }
                //check if gmo's superclass has a TagHandler and get the class to generate instead
                else if(gmo.getParent() == null && gmo.getClassDescriptor().getSuperClass() != null && TagManager.getGuixClassHandler(gmo.getClassDescriptor().getSuperClass().getName()) != null) {
                    th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().getSuperClass().getName());
                    if(th != null) {
                        clazz = th.getClassToGenerate();
                    }
                }
                //check if gmo's type is a generated class
                else if(gmo.getClassDescriptor() != null && classes.get(gmo.getClassDescriptor().toString()) != null) {
                    jFile = classes.get(gmo.getClassDescriptor().toString());
                }
                //check if gmo is the first tag and its superclass is a generated class
                else if(gmo.getParent() == null && gmo.getClassDescriptor().getSuperClass() != null && classes.get(gmo.getClassDescriptor().getSuperClass().toString()) != null) {
                    jFile = classes.get(gmo.getClassDescriptor().getSuperClass().toString());
                }
                //gmo's class is a basic class or an uncompiled class from the source folder
                else {
                    try{
                        //get the class represented by cd
                        clazz = Class.forName(gmo.getClassDescriptor().toString());
                    }
                    catch (ClassNotFoundException eee) {
                        File f = new File(swingGenerator.getSrcDir(), gmo.getClassDescriptor().toString().replace('.', File.separatorChar) + ".java");
                        try {
                            Reader isr = new InputStreamReader(new FileInputStream(f));
                            jFile = JavaFileParser.parseJavaFile(gmo.getClassDescriptor().toString(), isr);
                            swingGenerator.getPropertyChangeListenerDependencies().add(jFile);
                        }
                        catch (FileNotFoundException eeee) {
                            log.error(eeee);
                            throw new ClassNotFoundException();
                        }
                    }
                }
            }

            if(clazz != null) {
                superClazz = clazz.getSuperclass();
            }
            else {
                //get the first not generated superclass of gmo, the corresponding taghandler and record all the generated superclasses of gmo
                JavaFile jFile2 = jFile;
                while(superClazz == null && jFile2 != null) {
                    try {
                        superClazz = Class.forName(jFile2.getSuperClass());
                        superTh = TagManager.getGuixClassHandler(superClazz.getName());
                    }
                    catch(ClassNotFoundException eee) {
                        generatedSuperClasses.add(jFile2.getSuperClass());
                        jFile2 = classes.get(jFile2.getSuperClass().endsWith("Abstract") ? jFile2.getSuperClass().substring(0, jFile2.getSuperClass().length() - 8)
                                : jFile2.getSuperClass());
                    }
                }
            }

            //check if we need to generate the field, and check if its type is compatible with an eventual inherited field
            if(gmo.getParent() == null) {
                createField = false;
            }
            else if(jf.getInheritedField(gmo.getId()) != null) {
                if((clazz != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))
                        || (jFile != null && jf.getInheritedField(gmo.getId()).getType().equals(jFile.getPackageName() + "." + jFile.getClassName()))) {
                    createField = false;
                }
                else {
                    Class inheritedClazz = null;
                    try {
                        inheritedClazz = Class.forName(jf.getInheritedField(gmo.getId()).getType());
                        if(clazz != null) {
                            if(!inheritedClazz.isAssignableFrom(clazz)) {
                                throw new ClassCastException();
                            }
                        }
                        else if(jFile != null && superClazz != null) {
                            if(!inheritedClazz.isAssignableFrom(superClazz)) {
                                throw new ClassCastException();
                            }
                        }
                        else {
                            throw new ClassCastException();
                        }
                    }
                    catch(ClassNotFoundException eee) {
                        if(!generatedSuperClasses.contains(jf.getInheritedField(gmo.getId()).getType())) {
                            throw  new ClassCastException();
                        }
                    }
                }
            }

            if(createField) {
                //if gmo has a TagHandler, add a field to the JavaFile with this TagHandler (used for the databinding generation)
                if(th != null && th.getClassToGenerate() != null) {
                    jf.addField(new JavaField(Modifier.PROTECTED,
                            clazz.getName(),
                            gmo.getId(), gmo.getJavadoc(), th),
                            gmo.isJavaBean());
                }
                else {                    
                    jf.addField(new JavaField(Modifier.PROTECTED,
                            (classes != null && classes.containsKey(gmo.getClassDescriptor().toString())) ? gmo.getClassDescriptor().toString() + "Abstract" : gmo.getClassDescriptor().toString(),
                            gmo.getId(), gmo.getJavadoc(), superTh),
                            gmo.isJavaBean());
                }
            }
            //if gmo does not represents the first tag
            if (gmo.getParent() != null) {
                if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                    settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
                }
                if(createField || gmo.getConstructor() != null) {
                    componentsCreation.append(gmo.getId()).append(" = new ");
                    if(classes != null && classes.containsKey(gmo.getClassDescriptor().toString())) {
                        componentsCreation.append(gmo.getClassDescriptor().toString() + "Impl");
                    }
                    else if(clazz != null) {
                        componentsCreation.append(clazz.getName());
                    }
                    else {
                        componentsCreation.append(gmo.getClassDescriptor().toString());
                    }
                    componentsCreation.append("(").append(gmo.getConstructor() != null ? gmo.getConstructor() : "").append(");\n");
                    if(jf.getInheritedField(gmo.getId()) != null) {
                        componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                    }
                }
                componentsSettings.append(gmo.getId()).append("Settings();\n");

                if (gmo.getStyleSheets() != null) {
                    styleSheets.addAll(gmo.getStyleSheets());
                }

                if(clazz != null) {
                    constraint = processAttributes(clazz, gmo, settingsMethod, seh, th);
                }
                else {
                    constraint = processAttributes(jFile, superClazz, gmo, settingsMethod, seh, superTh);
                }

                if((jFile != null && (jFile.getField(gmo.getId()) != null || jFile.getInheritedField(gmo.getId()) != null))
                        || (gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null)) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
                //if gmo is not the child of a cell
                else if (!ignoreTag(gmo)) {
                   if((gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("ScrollPanel"))
                            || gmo.getParent().getClassDescriptor().toString().equals("javax.swing.JScrollPane")) {
                        componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".setViewportView(").append(gmo.getId()).append(");\n");
                    }
                    //add the component to its parent
                    else if((superClazz != null && java.awt.Component.class.isAssignableFrom(superClazz) && !java.awt.Window.class.isAssignableFrom(superClazz))
                            || (gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("TreeNode"))) {
                        componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".add(").append(gmo.getId());
                        //with a constraint if it is not null
                        if (constraint != null) {
                            componentsTree.append(",").append(constraint);
                        }
                        componentsTree.append(");\n");
                    }
                }
            }
            //if gmo represents the first tag
            else {
                //set superclass and interface
                if(th != null) {
                    jf.setSuperClass(th.getClassToGenerate().getName());
                }
                else {
                    jf.setSuperClass(classes.containsKey(gmo.getClassDescriptor().getSuperClass().toString()) ? gmo.getClassDescriptor().getSuperClass().toString() + "Abstract" : gmo.getClassDescriptor().getSuperClass().toString());
                }
                jf.addInterface(gmo.getClassDescriptor().getName());

                //FIXME
                if (gmo.getStyleSheets() != null) {
                    styleSheets.addAll(gmo.getStyleSheets());
                }

                //get the inherited methods from the not generated superclass
                if(clazz != null) {
                    jf.setSuperclassIsGuixObject(false);
                    if(javax.swing.JComponent.class.isAssignableFrom(clazz)) {
                        jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "firePropertyChange",
                                new JavaArgument[]{new JavaArgument("String", "propertyName"), new JavaArgument("Object", "oldValue"), new JavaArgument("Object", "newValue")},
                                null, "super.firePropertyChange(propertyName, oldValue, newValue);", null));
                    }
                    
                    for(java.lang.reflect.Method m : clazz.getMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            JavaArgument[] args = new JavaArgument[m.getParameterTypes().length];
                            for(int i = 0 ; i < m.getParameterTypes().length ; i++) {
                                args[i] = new JavaArgument(m.getParameterTypes()[i].getName(), "arg" + i);
                            }
                            jf.addInheritedMethod(new JavaMethod(m.getModifiers(), m.getReturnType().getName(), m.getName(), args, null, "", null));
                        }
                    }
                    processAttributes(clazz, gmo, settingsMethod, seh, th);
                }
                //get the inherited methods and fiel from the generated superclass
                else if(jFile != null) {
                    jf.setSuperclassIsGuixObject(true);
                    for(JavaMethod m : jFile.getInheritedMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            jf.addInheritedMethod(m);
                        }
                    }
                    for(JavaMethod m : jFile.getMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            jf.addInheritedMethod(m);
                        }
                    }
                    for(JavaField f : jFile.getInheritedFields()) {
                        if(Modifier.isPublic(f.getModifiers()) || Modifier.isProtected(f.getModifiers())) {
                            jf.addInheritedField(f);
                        }
                    }
                    for(JavaField f : jFile.getFields()) {
                        if(Modifier.isPublic(f.getModifiers()) || Modifier.isProtected(f.getModifiers())) {
                            jf.addInheritedField(f);
                        }
                    }
                    processAttributes(jFile, superClazz, gmo, settingsMethod, seh, superTh != null ? superTh: th);
                }
                else {
                    log.error("Allo Houston, on a un probleme !");
                }
                componentsSettings.append(gmo.getId()).append("Settings();\n");
            }
            
            //browse the children of gmo
            for (GuixModelObject child : gmo.getChildren()) {
                Map<Method, String> methodBodies = browseModelObjects(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
        }
        Map<Method, String> result = new HashMap<Method, String>();
        result.put(Method.COMPONENTS_CREATION, componentsCreation.toString());
        result.put(Method.COMPONENTS_SETTINGS, componentsSettings.toString());
        result.put(Method.COMPONENTS_TREE, componentsTree.toString());
        return result;

    }

     /**
     *
     * @return true if the tag does not represent a real object but is just useful to ease the creation of the UI
     */
    private boolean ignoreTag(GuixModelObject gmo) {
        return (gmo.getParent() != null && (gmo.getParent().getClassDescriptor().getPackageName() == null
                    && (gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("Cell")
                    || gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("Tab"))));
    }

    /**
     * Generates the code to set the attributes of the gmo
     *
     * @param clazz class represented by gmo
     * @param gmo the GuixModelObject which contains the attribute
     * @param settingsMethod the StringBuffer containing the code of the setting method for the object represented by gmo
     * @param seh the event handler
     * @param th gmo's TagHandler
     * @return the value of the layout constraint, if any
     */
    private String processAttributes(Class clazz, GuixModelObject gmo, StringBuffer settingsMethod, SwingEventHandler seh, TagHandler th) {
        String constraint = null;
        processCSSAttributes(gmo, seh, clazz);
        //browses the attributes
        for (AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
            if(th != null && th.getAttrToGenerate(attr.getName()) != null) {
                attr.setName(th.getAttrToGenerate(attr.getName()));
            }
            boolean addQuote = false, addSimpleQuote = false;
            //the attribute name with the first letter capitalized
            String capitalizedAttribute = (attr.getName().length() > 0) ? Character.toUpperCase(attr.getName().charAt(0)) + attr.getName().substring(1) : attr.getName();
            //is the attribute a "real" attribute ? false if it represents an event or a constraint
            boolean realAttribute = true;

            try {
                //checks if the attribute is an event
                if (attr.getName().startsWith("on") && Character.isUpperCase(attr.getName().charAt(2))) {
                    //the swing event does not start by "on"
                    String lowerCaseAttribute = (attr.getName().length() > 2) ? Character.toLowerCase(attr.getName().charAt(2)) + attr.getName().substring(3) : attr.getName();
                    realAttribute = !seh.addEvent(clazz, lowerCaseAttribute, attr.getValue());
                }
                //checks if it is a constraint
                else if (attr.getName().equals(CONSTRAINT_ATTRIBUTE)) {
                    constraint = attr.getValue();
                    realAttribute = false;
                }

                if (realAttribute) {
                    //get the code of the binding (removes braces mostly)
                    String binding = BindingUtils.processDataBindings(attr.getValue());
                    
                    java.lang.reflect.Method[] methods = clazz.getMethods();
                    int m = 0;
                    //checks if the setter for this attribute exists
                    while(m < methods.length && !methods[m].getName().equals("set" + capitalizedAttribute)) {
                        m++;
                    }
                    //if yes
                    if(m < methods.length) {                        
                        //if the attribute binds the value of another component attribute
                        if (binding != null) {
                            if(!bindings2Generate.containsKey(gmo.getId())) {
                                bindings2Generate.put(gmo.getId(), new HashMap<String, String>());
                            }
                            bindings2Generate.get(gmo.getId()).put(attr.getName(), binding);
                        }
                        else { 
                            //checks if the parameter of the setter is a String
                            addQuote = methods[m].getParameterTypes()[0].equals(String.class);
                            addSimpleQuote = methods[m].getParameterTypes()[0].equals(Character.class) || methods[m].getParameterTypes()[0].equals(char.class);
                            //generates the code to set the attribute to object
                            if(settingsMethod != null) {
                                settingsMethod.append(gmo.getParent() != null ? gmo.getId() : "this").append(".set").append(capitalizedAttribute).append("(");
                                if(addQuote) {
                                    settingsMethod.append("\"").append(attr.getValue()).append("\"").append(");\n");
                                }
                                else if(addSimpleQuote) {
                                    settingsMethod.append("'").append(attr.getValue()).append("'").append(");\n");
                                }
                                else {
                                    settingsMethod.append(attr.getValue()).append(");\n");
                                }
                            }
                        }
                    }
                    else if(log.isErrorEnabled()) {
                        log.error(attr.getName() + " cannot be set.");
                    }
                }
            }
            catch (IntrospectionException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            }
        }
        if(gmo.getId() != null && settingsMethod != null) {
            //generates the code for the events
            settingsMethod.append(seh.generate(gmo.getId()));
            //add to the file the creation method for this object
            jf.addMethod(
                    new JavaMethod(Modifier.PROTECTED, "void", gmo.getId() + "Settings", null, null, settingsMethod.toString(), gmo.getId() + " settings"));
        }
        //return the constraint to put the object into its parent
        return constraint;
    }

    /**
     * Generates the code to set the attributes of the gmo when gmo's class is an uncompiled java file
     *
     * @param jFile the JavaFile describing th uncompiled java file
     * @param superClazz the first not generated superclass of gmo
     * @param gmo the GuixModelObject instance of the clas represented by jFile
     * @param settingsMethod the StringBuffer containing the setting method
     * @param seh the event handler
     * @param th gmo's TagHandler
     * @return the value of the layout constraint, if any
     */
    private String processAttributes(JavaFile jFile, Class superClazz, GuixModelObject gmo, StringBuffer settingsMethod, SwingEventHandler seh, TagHandler th) {
        String constraint = null;
        processCSSAttributes(gmo, jFile);
        //browses the attributes
        for (AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
            if(th != null && th.getAttrToGenerate(attr.getName()) != null) {
                attr.setName(th.getAttrToGenerate(attr.getName()));
            }
            boolean addQuote = false, addSimpleQuote = false;
            //the attribute name with the first letter capitalized
            String capitalizedAttribute = (attr.getName().length() > 0) ? Character.toUpperCase(attr.getName().charAt(0)) + attr.getName().substring(1) : attr.getName();
            //is the attribute a "real" attribute ? false if it represents an event or a constraint
            boolean realAttribute = true;

            try {
                //checks if the attribute is an event
                if (attr.getName().startsWith("on") && Character.isUpperCase(attr.getName().charAt(2))) {
                    //the swing event does not start by "on"
                    String lowerCaseAttribute = (attr.getName().length() > 2) ? Character.toLowerCase(attr.getName().charAt(2)) + attr.getName().substring(3) : attr.getName();
                    realAttribute = !seh.addEvent(superClazz, lowerCaseAttribute, attr.getValue());
                }
                //checks if it is a constraint
                else if (attr.getName().equals(CONSTRAINT_ATTRIBUTE)) {
                    constraint = attr.getValue();
                    realAttribute = false;
                }

                if (realAttribute) {
                    JavaMethod method = null;
                    for(JavaMethod m : jFile.getMethods()) {
                        if(m.getName().equals("set" + capitalizedAttribute)) {
                            method = m;
                            break;
                        }
                    }
                    if(method == null) {
                        for(JavaMethod m : jFile.getInheritedMethods()) {
                            if(m.getName().equals("set" + capitalizedAttribute)) {
                                method = m;
                                break;
                            }
                        }
                    }
                    if(method != null) {
                        String binding = BindingUtils.processDataBindings(attr.getValue());
                        if(binding != null) {
                            if(!bindings2Generate.containsKey(gmo.getId())) {
                                bindings2Generate.put(gmo.getId(), new HashMap<String, String>());
                            }
                            bindings2Generate.get(gmo.getId()).put(attr.getName(), binding);
                        }
                        else {
                            //checks if the parameter of the setter is a String
                            addQuote = method.getArguments()[0].getType().equals(String.class.getName());
                            addSimpleQuote = method.getArguments()[0].getType().equals(Character.class.getName()) || method.getArguments()[0].getType().equals(char.class.getName());
                            //generates the code to set the attribute to object
                            settingsMethod.append(gmo.getParent() != null ? gmo.getId() : "this").append(".set").append(capitalizedAttribute).append("(");
                            if(addQuote) {
                                settingsMethod.append("\"").append(attr.getValue()).append("\"").append(");\n");
                            }
                            else if(addSimpleQuote) {
                                settingsMethod.append("'").append(attr.getValue()).append("'").append(");\n");
                            }
                            else {
                                settingsMethod.append(attr.getValue()).append(");\n");
                            }
                        }
                    }
                    else if(log.isErrorEnabled()) {
                        log.error(attr.getName() + " cannot be set.");
                    }
                }
            }
            catch (IntrospectionException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            }
        }
        //if gmo does not represents the first tag
        if(gmo.getId() != null && settingsMethod != null) {
            //add to the file the creation method for this object
            jf.addMethod(new JavaMethod(Modifier.PROTECTED, "void", gmo.getId() + "Settings", null, null, settingsMethod.toString(), gmo.getId() + " settings"));
        }
        //return the constraint to put the object into its parent
        return constraint;
    }

    /**
     * Transform the css attributes into AttributeDescriptors
     *
     * @param gmo the GuixModelObject which has got the css attributes
     * @param seh a SwingEventHandlerTest
     * @param clazz gmo's class
     */
    private void processCSSAttributes(GuixModelObject gmo, SwingEventHandler seh, Class clazz) {
        Map<String, String> cssIds = new HashMap<String, String>();
        Map<String, String> cssStyleClasses = new HashMap<String, String>();
        Map<String, String> cssJavaClasses = new HashMap<String, String>();
        List<AttributeDescriptor> ads = new ArrayList<AttributeDescriptor>();
        for (StyleSheet ss : styleSheets) {
            for (Selector sel : ss.getSelectors()) {
                if((sel.getId() == null || sel.getId().equals(gmo.getId()))
                        && (sel.getStyleClass() == null || sel.getStyleClass().equals(gmo.getStyleClass()))
                        && (sel.getJavaClassName() == null || sel.getJavaClassName().equals(gmo.getClassDescriptor().getName()))) {
                    boolean realAttribute = true;
                    if (sel.getPseudoClass() != null && sel.getPseudoClass().startsWith("on") && Character.isUpperCase(sel.getPseudoClass().charAt(2))) {
                        //the swing event does not start by "on"
                        String lowerCaseAttribute = (sel.getPseudoClass().length() > 2) ? Character.toLowerCase(sel.getPseudoClass().charAt(2)) + sel.getPseudoClass().substring(3) : sel.getPseudoClass();
                        StringBuffer eventAction = new StringBuffer();

                        for (Rule rule : sel.getRules()) {
                            for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                java.lang.reflect.Method[] methods = clazz.getMethods();
                                String capitalizedAttribute = (entry.getKey().length() > 0) ? Character.toUpperCase(entry.getKey().charAt(0)) + entry.getKey().substring(1) : entry.getKey();
                                int m = 0;
                                //checks if the setter for this attribute exists
                                while(m < methods.length && !methods[m].getName().equals("set" + capitalizedAttribute)) {
                                    m++;
                                }
                                //if yes
                                if(m < methods.length) {
                                    //checks if the parameter of the setter is a String
                                    boolean addQuote = methods[m].getParameterTypes()[0].equals(String.class);
                                    //generates the code to set the attribute to object
                                    eventAction.append(gmo.getId()).append(".set").append(capitalizedAttribute).append("(").append(addQuote ? "\"" : "").append(entry.getValue()).append(addQuote ? "\"" : "").append(");\n");
                                                                                }
                                else if(log.isErrorEnabled()) {
                                    log.error(entry.getKey() + " cannot be set.");
                                }
                            }
                        }
                        try {
                            realAttribute = !seh.addEvent(clazz, lowerCaseAttribute, eventAction.toString());
                        }
                        catch (IntrospectionException ex) {
                            if(log.isErrorEnabled()) {
                                log.error("Error while adding event " + sel.getPseudoClass());
                            }
                        }
                    }
                    if(realAttribute) {
                        if (sel.getId() != null && sel.getId().equals(gmo.getId())) {
                            for (Rule rule : sel.getRules()) {
                                for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                    cssIds.put(entry.getKey(), entry.getValue());
                                }
                            }
                        }
                        else if (sel.getStyleClass() != null && sel.getStyleClass().equals(gmo.getStyleClass())) {
                            for (Rule rule : sel.getRules()) {
                                for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                    cssStyleClasses.put(entry.getKey(), entry.getValue());
                                }
                            }
                        }
                        else if (sel.getJavaClassName() != null && sel.getJavaClassName().equals(gmo.getClassDescriptor().getName())) {
                            for (Rule rule : sel.getRules()) {
                                for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                    cssJavaClasses.put(entry.getKey(), entry.getValue());
                                }
                            }
                        }
                    }
                }
            }
        }
        for(Entry<String, String> entry : cssIds.entrySet()) {
            ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
        }
        for(Entry<String, String> entry : cssStyleClasses.entrySet()) {
            if(!cssIds.containsKey(entry.getKey())) {
                ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
            }
        }
        for(Entry<String, String> entry : cssJavaClasses.entrySet()) {
            if(!cssIds.containsKey(entry.getKey()) && !cssStyleClasses.containsKey(entry.getKey())) {
                ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
            }
        }
        gmo.getAttributeDescriptors().addAll(ads);
    }

    /**
     * Transform the css attributes into AttributeDescriptors
     *
     * @param gmo the GuixModelObject which has got the css attributes
     * @param jFile the JavaFile describing th uncompiled java file
     */
    private void processCSSAttributes(GuixModelObject gmo, JavaFile jFile) {
        Map<String, String> cssIds = new HashMap<String, String>();
        Map<String, String> cssStyleClasses = new HashMap<String, String>();
        Map<String, String> cssJavaClasses = new HashMap<String, String>();
        List<AttributeDescriptor> ads = new ArrayList<AttributeDescriptor>();
        for (StyleSheet ss : styleSheets) {
            for (Selector sel : ss.getSelectors()) {
                if((sel.getId() == null || sel.getId().equals(gmo.getId()))
                        && (sel.getStyleClass() == null || sel.getStyleClass().equals(gmo.getStyleClass()))
                        && (sel.getJavaClassName() == null || sel.getJavaClassName().equals(gmo.getClassDescriptor().getName()))) {
                    if (sel.getPseudoClass() != null && sel.getPseudoClass().startsWith("on") && Character.isUpperCase(sel.getPseudoClass().charAt(2))) {
                        //the swing event does not start by "on"
                        StringBuffer eventAction = new StringBuffer();

                        for (Rule rule : sel.getRules()) {
                            for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                String capitalizedAttribute = (entry.getKey().length() > 0) ? Character.toUpperCase(entry.getKey().charAt(0)) + entry.getKey().substring(1) : entry.getKey();
                                //checks if the setter for this attribute exists
                                JavaMethod method = null;
                                for(JavaMethod m : jFile.getMethods()) {
                                    if(m.getName().equals("set" + capitalizedAttribute)) {
                                        method = m;
                                        break;
                                    }
                                }
                                //if yes
                                if(method != null) {
                                    //checks if the parameter of the setter is a String
                                    boolean addQuote = method.getArguments()[0].getType().equals(String.class.getName());
                                    //generates the code to set the attribute to object
                                    eventAction.append(gmo.getId()).append(".set").append(capitalizedAttribute).append("(").append(addQuote ? "\"" : "").append(entry.getValue()).append(addQuote ? "\"" : "").append(");\n");
                                                                                }
                                else if(log.isErrorEnabled()) {
                                    log.error(entry.getKey() + " cannot be set.");
                                }
                            }
                        }
                    }
                    if (sel.getId() != null && sel.getId().equals(gmo.getId())) {
                        for (Rule rule : sel.getRules()) {
                            for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                cssIds.put(entry.getKey(), entry.getValue());
                            }
                        }
                    }
                    else if (sel.getStyleClass() != null && sel.getStyleClass().equals(gmo.getStyleClass())) {
                        for (Rule rule : sel.getRules()) {
                            for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                cssStyleClasses.put(entry.getKey(), entry.getValue());
                            }
                        }
                    }
                    else if (sel.getJavaClassName() != null && sel.getJavaClassName().equals(gmo.getClassDescriptor().getName())) {
                        for (Rule rule : sel.getRules()) {
                            for (Entry<String, String> entry : rule.getProperties().entrySet()) {
                                cssJavaClasses.put(entry.getKey(), entry.getValue());
                            }
                        }
                    }
                }
            }
        }
        for(Entry<String, String> entry : cssIds.entrySet()) {
            ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
        }
        for(Entry<String, String> entry : cssStyleClasses.entrySet()) {
            if(!cssIds.containsKey(entry.getKey())) {
                ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
            }
        }
        for(Entry<String, String> entry : cssJavaClasses.entrySet()) {
            if(!cssIds.containsKey(entry.getKey()) && !cssStyleClasses.containsKey(entry.getKey())) {
                ads.add(new AttributeDescriptor(entry.getKey(), entry.getValue()));
            }
        }
        gmo.getAttributeDescriptors().addAll(ads);
    }

    public Map<String, Map<String, String>> getBindings2Generate() {
        return bindings2Generate;
    }
    
}

