/*******************************************************************************
 * Copyright (c) 2015-2018 Skymind, Inc.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Apache License, Version 2.0 which is available at
 * https://www.apache.org/licenses/LICENSE-2.0.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 ******************************************************************************/

package org.nd4j.linalg.api.ops.impl.reduce;

import lombok.NoArgsConstructor;
import lombok.NonNull;
import org.nd4j.autodiff.samediff.SDVariable;
import org.nd4j.autodiff.samediff.SameDiff;
import org.nd4j.common.base.Preconditions;
import org.nd4j.imports.graphmapper.tf.TFGraphMapper;
import org.nd4j.linalg.api.buffer.DataType;
import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.linalg.api.ops.DynamicCustomOp;
import org.nd4j.linalg.api.ops.impl.reduce.bp.MeanBp;
import org.nd4j.linalg.api.ops.impl.reduce.bp.VarianceBp;
import org.nd4j.linalg.factory.Nd4j;
import org.tensorflow.framework.AttrValue;
import org.tensorflow.framework.GraphDef;
import org.tensorflow.framework.NodeDef;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

@NoArgsConstructor
public class Moments extends DynamicCustomOp {

    private int[] axes;

    public Moments(@NonNull INDArray input, int... axes){
        super(new INDArray[]{input}, null);
        this.axes = axes;
        addArgs();
    }

    public Moments(SameDiff sameDiff, SDVariable input) {
        this(sameDiff, input, null);
    }

    public Moments(SameDiff sameDiff, SDVariable input, int[] axes) {
        super(null, sameDiff, new SDVariable[] {input}, false);
        this.axes = axes;
        addArgs();
    }

    public Moments(INDArray in, INDArray outMean, INDArray outStd, int... axes){
        super(null, new INDArray[]{in}, new INDArray[]{outMean, outStd}, null, axes);
    }

    private void addArgs() {
        if(axes != null) {
            for (int axis : axes) {
                addIArgument(axis);
            }
        }
    }

    @Override
    public void initFromTensorFlow(NodeDef nodeDef, SameDiff initWith, Map<String, AttrValue> attributesForNode, GraphDef graph) {
        TFGraphMapper.initFunctionFromProperties(nodeDef.getOp(), this, attributesForNode, nodeDef, graph);
        addArgs();
    }

    @Override
    public String opName() {
        return "moments";
    }

    @Override
    public List<SDVariable> doDiff(List<SDVariable> grad){
        SDVariable dLdMean = grad.get(0);
        SDVariable dLdVar = grad.get(1);        //Note: non-bias-corrected variance
        SDVariable meanBp = new MeanBp(sameDiff, arg(), dLdMean, false, axes).outputVariable();
        SDVariable varBp = new VarianceBp(sameDiff, arg(), dLdVar, false, false, axes).outputVariable();
        return Collections.singletonList(meanBp.add(varBp));
    }

    @Override
    public List<DataType> calculateOutputDataTypes(List<DataType> dataTypes){
        Preconditions.checkState(dataTypes != null && dataTypes.size() == 1, "Expected 1 datatype for %s, got %s", getClass(), dataTypes);
        if(dataTypes.get(0).isFPType())
            return Arrays.asList(dataTypes.get(0), dataTypes.get(0));
        return Arrays.asList(Nd4j.defaultFloatingPointType(), Nd4j.defaultFloatingPointType());
    }
}
