/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common.extensions.mux;

import java.nio.ByteBuffer;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.common.extensions.mux.MuxException;
import org.eclipse.jetty.websocket.common.extensions.mux.MuxedFrame;
import org.eclipse.jetty.websocket.common.extensions.mux.op.MuxAddChannelRequest;
import org.eclipse.jetty.websocket.common.extensions.mux.op.MuxAddChannelResponse;
import org.eclipse.jetty.websocket.common.extensions.mux.op.MuxDropChannel;
import org.eclipse.jetty.websocket.common.extensions.mux.op.MuxFlowControl;
import org.eclipse.jetty.websocket.common.extensions.mux.op.MuxNewChannelSlot;

public class MuxParser {
    private static final Logger LOG = Log.getLogger(MuxParser.class);
    private MuxedFrame muxframe = new MuxedFrame();
    private Listener events;
    private long channelId;

    public Listener getEvents() {
        return this.events;
    }

    public synchronized void parse(Frame frame) {
        if (this.events == null) {
            throw new RuntimeException("No " + Listener.class + " specified");
        }
        if (!frame.hasPayload()) {
            LOG.debug("No payload data, skipping", new Object[0]);
            return;
        }
        if (frame.getType().getOpCode() != 2) {
            LOG.debug("Not a binary opcode (base frame), skipping", new Object[0]);
            return;
        }
        LOG.debug("Parsing Mux Payload of {}", new Object[]{frame});
        try {
            ByteBuffer buffer = frame.getPayload().slice();
            if (buffer.remaining() <= 0) {
                return;
            }
            if (frame.isContinuation()) {
                this.muxframe.reset();
                this.muxframe.setFin(frame.isFin());
                this.muxframe.setFin(frame.isRsv1());
                this.muxframe.setFin(frame.isRsv2());
                this.muxframe.setFin(frame.isRsv3());
                this.muxframe.setContinuation(true);
                this.parseDataFramePayload(buffer);
            } else {
                this.channelId = this.readChannelId(buffer);
                if (this.channelId == 0L) {
                    this.parseControlBlocks(buffer);
                } else {
                    this.parseDataFrame(buffer);
                }
            }
        }
        catch (MuxException e) {
            this.events.onMuxException(e);
        }
        catch (Throwable t) {
            this.events.onMuxException(new MuxException(t));
        }
    }

    private void parseControlBlocks(ByteBuffer buffer) {
        while (buffer.remaining() > 0) {
            byte b = buffer.get();
            byte opc = (byte)((byte)(b >> 5) & 0xFF);
            b = (byte)(b & 0x1F);
            try {
                switch (opc) {
                    case 0: {
                        MuxAddChannelRequest op = new MuxAddChannelRequest();
                        op.setRsv((byte)((b & 0x1C) >> 2));
                        op.setEncoding((byte)(b & 3));
                        op.setChannelId(this.readChannelId(buffer));
                        long handshakeSize = this.read139EncodedSize(buffer);
                        op.setHandshake(this.readBlock(buffer, handshakeSize));
                        this.events.onMuxAddChannelRequest(op);
                        break;
                    }
                    case 1: {
                        MuxAddChannelResponse op = new MuxAddChannelResponse();
                        op.setFailed((b & 0x10) != 0);
                        op.setRsv((byte)((byte)(b & 0xC) >> 2));
                        op.setEncoding((byte)(b & 3));
                        op.setChannelId(this.readChannelId(buffer));
                        long handshakeSize = this.read139EncodedSize(buffer);
                        op.setHandshake(this.readBlock(buffer, handshakeSize));
                        this.events.onMuxAddChannelResponse(op);
                        break;
                    }
                    case 3: {
                        int rsv = b & 0x1F;
                        long channelId = this.readChannelId(buffer);
                        long reasonSize = this.read139EncodedSize(buffer);
                        ByteBuffer reasonBuf = this.readBlock(buffer, reasonSize);
                        MuxDropChannel op = MuxDropChannel.parse(channelId, reasonBuf);
                        op.setRsv(rsv);
                        this.events.onMuxDropChannel(op);
                        break;
                    }
                    case 2: {
                        MuxFlowControl op = new MuxFlowControl();
                        op.setRsv((byte)(b & 0x1F));
                        op.setChannelId(this.readChannelId(buffer));
                        op.setSendQuotaSize(this.read139EncodedSize(buffer));
                        this.events.onMuxFlowControl(op);
                        break;
                    }
                    case 4: {
                        MuxNewChannelSlot op = new MuxNewChannelSlot();
                        op.setRsv((byte)((b & 0x1E) >> 1));
                        op.setFallback((b & 1) != 0);
                        op.setNumberOfSlots(this.read139EncodedSize(buffer));
                        op.setInitialSendQuota(this.read139EncodedSize(buffer));
                        this.events.onMuxNewChannelSlot(op);
                        break;
                    }
                    default: {
                        String err = String.format("Unknown Mux Control Code OPC [0x%X]", opc);
                        throw new MuxException(err);
                    }
                }
            }
            catch (Throwable t) {
                LOG.warn(t);
                throw new MuxException(t);
            }
        }
    }

    private void parseDataFrame(ByteBuffer buffer) {
        byte b = buffer.get();
        boolean fin = (b & 0x80) != 0;
        boolean rsv1 = (b & 0x40) != 0;
        boolean rsv2 = (b & 0x20) != 0;
        boolean rsv3 = (b & 0x10) != 0;
        byte opcode = (byte)(b & 0xF);
        if (opcode == 0) {
            this.muxframe.setContinuation(true);
        } else {
            this.muxframe.reset();
            this.muxframe.setOpCode(opcode);
        }
        this.muxframe.setChannelId(this.channelId);
        this.muxframe.setFin(fin);
        this.muxframe.setRsv1(rsv1);
        this.muxframe.setRsv2(rsv2);
        this.muxframe.setRsv3(rsv3);
        this.parseDataFramePayload(buffer);
    }

    private void parseDataFramePayload(ByteBuffer buffer) {
        int capacity = buffer.remaining();
        ByteBuffer payload = ByteBuffer.allocate(capacity);
        payload.put(buffer);
        BufferUtil.flipToFlush((ByteBuffer)payload, (int)0);
        this.muxframe.setPayload(payload);
        try {
            LOG.debug("notifyFrame() - {}", new Object[]{this.muxframe});
            this.events.onMuxedFrame(this.muxframe);
        }
        catch (Throwable t) {
            LOG.warn(t);
        }
    }

    public long read139EncodedSize(ByteBuffer buffer) {
        long ret = -1L;
        long minValue = 0L;
        int cursor = 0;
        byte b = buffer.get();
        ret = b & 0x7F;
        if (ret == 127L) {
            ret = 0L;
            minValue = 65535L;
            cursor = 8;
        } else if (ret == 126L) {
            ret = 0L;
            minValue = 127L;
            cursor = 2;
        } else {
            return ret;
        }
        while (cursor > 0) {
            ret <<= 8;
            b = buffer.get();
            ret |= (long)(b & 0xFF);
            --cursor;
        }
        if (ret <= minValue) {
            String err = String.format("Invalid 1/3/9 length 0x%X (minimum value for chosen encoding is 0x%X)", ret, minValue);
            throw new MuxException(err);
        }
        return ret;
    }

    private ByteBuffer readBlock(ByteBuffer buffer, long size) {
        if (size == 0L) {
            return null;
        }
        if (size > (long)buffer.remaining()) {
            String err = String.format("Truncated data, expected %,d byte(s), but only %,d byte(s) remain", size, buffer.remaining());
            throw new MuxException(err);
        }
        if (size > Integer.MAX_VALUE) {
            String err = String.format("[Int-Sane!] Buffer size %,d is too large to be supported (max allowed is %,d)", size, Integer.MAX_VALUE);
            throw new MuxException(err);
        }
        ByteBuffer ret = ByteBuffer.allocate((int)size);
        BufferUtil.put((ByteBuffer)buffer, (ByteBuffer)ret);
        BufferUtil.flipToFlush((ByteBuffer)ret, (int)0);
        return ret;
    }

    public long readChannelId(ByteBuffer buffer) {
        long id = -1L;
        long minValue = 0L;
        byte b = buffer.get();
        int cursor = -1;
        if ((b & 0x80) == 0) {
            return b & 0x7F;
        }
        if ((b & 0x40) == 0) {
            id = b & 0x3F;
            minValue = 127L;
            cursor = 1;
        } else if ((b & 0x20) == 0) {
            id = b & 0x1F;
            minValue = 16383L;
            cursor = 2;
        } else {
            id = b & 0x1F;
            minValue = 0x1FFFFFL;
            cursor = 3;
        }
        while (cursor > 0) {
            id <<= 8;
            b = buffer.get();
            id |= (long)(b & 0xFF);
            --cursor;
        }
        if (id <= minValue) {
            String err = String.format("Invalid Channel ID 0x%X (minimum value for chosen encoding is 0x%X)", id, minValue);
            throw new MuxException(err);
        }
        return id;
    }

    public void setEvents(Listener events) {
        this.events = events;
    }

    public static interface Listener {
        public void onMuxAddChannelRequest(MuxAddChannelRequest var1);

        public void onMuxAddChannelResponse(MuxAddChannelResponse var1);

        public void onMuxDropChannel(MuxDropChannel var1);

        public void onMuxedFrame(MuxedFrame var1);

        public void onMuxException(MuxException var1);

        public void onMuxFlowControl(MuxFlowControl var1);

        public void onMuxNewChannelSlot(MuxNewChannelSlot var1);
    }
}

