/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiStorageServiceImpl.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/VradiStorageServiceImpl.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi.services;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiConstants;
import org.chorem.vradi.VradiServiceConfiguration;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.beans.SendingHelper;
import org.chorem.vradi.beans.XmlStreamImportResult;
import org.chorem.vradi.entities.Client;
import org.chorem.vradi.entities.Form;
import org.chorem.vradi.entities.Group;
import org.chorem.vradi.entities.Infogene;
import org.chorem.vradi.entities.QueryMaker;
import org.chorem.vradi.entities.Sending;
import org.chorem.vradi.entities.Session;
import org.chorem.vradi.entities.Status;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.entities.VradiUser;
import org.chorem.vradi.entities.WebHarvestStream;
import org.chorem.vradi.entities.XmlStream;
import org.chorem.vradi.services.managers.BindingManager;
import org.chorem.vradi.services.managers.ClientManager;
import org.chorem.vradi.services.managers.FormManager;
import org.chorem.vradi.services.managers.FormTypeManager;
import org.chorem.vradi.services.managers.MailingManager;
import org.chorem.vradi.services.managers.SearchManager;
import org.chorem.vradi.services.managers.TemplateManager;
import org.chorem.vradi.services.managers.ThesaurusManager;
import org.chorem.vradi.services.tasks.TasksManager;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.chorem.vradi.VradiConstants.SendingStatus;
import static org.chorem.vradi.VradiConstants.SessionStatus;
import static org.nuiton.i18n.I18n._;

/**
 * Vradi storage implementation.
 *
 * @author morin
 * @author chatellier
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    protected ApplicationConfig config;

    /** Proxy to store and retrieve data. */
    protected WikittyProxy wikittyProxy;

    protected WikittyImportExportService importExportService;

    protected ThesaurusManager thesaurusManager;

    protected BindingManager bindingManager;

    protected FormManager formManager;

    protected FormTypeManager formTypeManager;

    protected ClientManager clientManager;

    protected SearchManager searchManager;

    protected MailingManager mailingManager;

    protected TasksManager tasksManager;

    protected FileService fileService;

    /** Remote instantiation and empty constructor for hessian. */
    public VradiStorageServiceImpl() {
        this(VradiServiceConfiguration.getConfig(), ServiceFactory.getWikittyProxy());
    }

    /**
     * This constructor exists for local wikittyProxy instanciation.
     *
     * @param config       vradi config
     * @param wikittyProxy custom wikitty proxy
     */
    protected VradiStorageServiceImpl(ApplicationConfig config, WikittyProxy wikittyProxy) {
        this.wikittyProxy = wikittyProxy;
        this.config = config;

        thesaurusManager = new ThesaurusManager(wikittyProxy);
        clientManager = new ClientManager(config, wikittyProxy);
        formTypeManager = new FormTypeManager(config, wikittyProxy);
        mailingManager = new MailingManager(config, wikittyProxy, formTypeManager);

        fileService = new FileServiceImpl(config);
        searchManager = new SearchManager(wikittyProxy, thesaurusManager);
        formManager = new FormManager(config, wikittyProxy, thesaurusManager, searchManager);
        bindingManager = new BindingManager(config, wikittyProxy, formTypeManager, formManager);
        tasksManager = new TasksManager(config, wikittyProxy, thesaurusManager, formManager, mailingManager, bindingManager);

        // init cron tasks
        tasksManager.initTasks();
        wikittyProxy.getWikittyService().addWikittyServiceListener(tasksManager, WikittyService.ServiceListenerType.ALL);
    }

    /**
     * Get {@link WikittyImportExportService} singl instance.
     *
     * @return WikittyImportExportService singleton
     */
    protected WikittyImportExportService getImportExportService() {
        if (importExportService == null) {
            ApplicationConfig config = VradiServiceConfiguration.getConfig();

            importExportService = new WikittyImportExportService(
                    config,
                    wikittyProxy.getSecurityToken(),
                    wikittyProxy.getWikittyService());

        }
        return importExportService;
    }

    /**
     * Create all default status.
     *
     * @return all status created
     */
    @Override
    public List<Status> createDefaultStatuses() throws VradiException {
        return formManager.createDefaultStatuses();
    }

    @Override
    public Session createNewSession() throws VradiException {
        return formManager.createNewSession();
    }

    @Override
    public Session bindForms(Session session) throws VradiException {
        return formManager.bindForms(session);
    }

    @Override
    public List<Sending> createAllSending(String sessionId, Group group, List<String> formsIdsToBind) throws VradiException {
        return formManager.createAllSending(sessionId, group, formsIdsToBind);
    }

    @Override
    public Sending createSending(String sessionId, User user, List<String> formsIdsToBind) throws VradiException {
        return formManager.createUserSending(sessionId, user, formsIdsToBind);
    }

    @Override
    public List<Sending> removeAllSending(String sessionId, List<String> formsIds, Group group)
            throws VradiException {
        return formManager.removeAllSending(sessionId, formsIds, group);
    }

    @Override
    public Sending removeSending(String sessionId, List<String> formsIds, User user)
            throws VradiException {
        return formManager.removeSending(sessionId, formsIds, user);
    }

    /**
     * Creates and store forms from an {@code WebHarvestStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     * <p/>
     * Input xmlStream is modified by this method (wikitty obselete).
     *
     * @param webHarvestStream stream to import
     * @return a structure containing :
     *         - the number of created forms
     *         - the number of already existing forms
     *         - the number of forms created with date parsing error
     *         - the number of forms created with number parsing error
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromWebHarvestStream(WebHarvestStream webHarvestStream)
            throws VradiException {
        return bindingManager.importFormsFromWebHarvestStream(webHarvestStream);
    }

    /**
     * Creates and store forms from an {@code XmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     * <p/>
     * Input xmlStream is modified by this method (wikitty obselete).
     *
     * @param xmlStream stream to import
     * @return a structure containing :
     *         - the number of created forms
     *         - the number of already existing forms
     *         - the number of forms created with date parsing error
     *         - the number of forms created with number parsing error
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromXmlStream(XmlStream xmlStream)
            throws VradiException {
        return bindingManager.importFormsFromXmlStream(xmlStream);
    }

    @Override
    public VradiUser createUser(VradiUser user) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateVradiUser(" + user.getLogin() + ")");
        }

        Search search = Search.query().eq(VradiUser.FIELD_WIKITTYUSER_LOGIN, user.getLogin());
        Criteria criteria = search.criteria();
        criteria.setEndIndex(1); // one is enought

        PagedResult<VradiUser> users = wikittyProxy.findAllByCriteria(VradiUser.class, criteria);

        if (users.getNumFound() > 0) {
            if (log.isDebugEnabled()) {
                log.debug("User " + user.getLogin() + " already exists");
            }
            return null;

        } else {
            return wikittyProxy.store(user);
        }
    }

    @Override
    public VradiUser loginUser(String login, String mp5password) {
        if (log.isDebugEnabled()) {
            log.debug("logVradiUser(" + login + ", " + mp5password + ")");
        }

        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_LOGIN, login)
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_PASSWORD, mp5password)
                .criteria();
        criteria.setEndIndex(1); // max 1

        VradiUser vradiUser = wikittyProxy.findByCriteria(VradiUser.class, criteria);
        return vradiUser;
    }

    /*@Override
    public void importData(File file) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("importData(file)");
        }

        // TODO: create an archive of vradi.home directory before

        try {
            String toURI = file.toURI().toString();
            log.info("importing file: " + toURI);
            
            WikittyImportExportService importExportService = getImportExportService();
            importExportService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, toURI);
    
            log.info("import done");
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", e);
            }
            throw new VradiException("Can't import data", e);
        }
    }

    @Override
    public String exportData() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("exportData()");
        }

        try {
            Search search = Search.query().keyword("*");
            Criteria criteria = search.criteria();

            WikittyImportExportService importExportService = getImportExportService();
            String export = importExportService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
            log.info("export done");
            return export;

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", e);
            }
            throw new VradiException("Can't export data", e);
        }
    }*/

    @Override
    public void importAsCSV(String uri) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Importing data as CSV with url = " + uri);
        }

        try {
            WikittyImportExportService ieService = getImportExportService();
            ieService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, uri);
        } catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", eee);
            }
            throw new VradiException("Can't import data", eee);
        }
    }

    @Override
    public String exportAsCSV(Criteria criteria) throws VradiException {

        //FIXME JC 01/09/2010 Should create a file and make him downloadable through FileService.

        if (log.isDebugEnabled()) {
            log.debug("Exporting data as CSV with criteria " + criteria);
        }

        String result = null;
        try {
            WikittyImportExportService ieService = getImportExportService();
            result = ieService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
        } catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", eee);
            }
        }
        return result;
    }

    @Override
    public String exportThesaurusAsCSV() throws VradiException {
        // creation du criteria wikitty
        // (export de tous les wikitty avec extension "thesaurus"
        Search search = Search.query().eq(Element.ELT_EXTENSION,
                                          WikittyTreeNode.EXT_WIKITTYTREENODE);
        Criteria criteria = search.criteria();
        return exportAsCSV(criteria);
    }

    @Override
    public void reindexData() {
        if (log.isDebugEnabled()) {
            log.debug("reindexData()");
        }

        wikittyProxy.getWikittyService().syncSearchEngine(null);
    }

    @Override
    public void archiveQueries(QueryMaker queryMaker)
            throws VradiException {
        clientManager.archiveQueries(queryMaker);
    }

    @Override
    public void setAssociatedFields(String extensionName,
                                    String templateName,
                                    Map<String, String> fieldMap)
            throws VradiException {

        formTypeManager.setAssociatedFields(extensionName, templateName, fieldMap);

    }

    @Override
    public String generatePDF(String formId, boolean force) throws VradiException {
        return mailingManager.generatePDF(formId, force);
    }


    @Override
    public Collection<String> getTemplateFilenames(WikittyExtension extension) {
        return formTypeManager.getTemplateFilenames(extension);
    }

    @Override
    public Map<String, String> getAssociatedFields(String extensionName,
                                                   String templateName)
            throws VradiException {
        return formTypeManager.getAssociatedFields(extensionName, templateName);
    }

    @Override
    public List<String> getDocumentFields(WikittyExtension extension, String templateName) throws VradiException {

        List<String> documentsFields;

        try {
            File templateDir = VradiServiceConfigurationHelper.getTemplatesDir(config);
            File extensionTemplate = new File(templateDir, extension.getName());
            File templateFile = new File(extensionTemplate, templateName);

            TemplateManager templateManager = new TemplateManager(config, templateFile);
            documentsFields = templateManager.getDocumentFields();
        } catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get documents fields", ex);
            }
            throw ex;
        }
        return documentsFields;
    }

    @Override
    public String sendMessages(String sessionId) throws VradiException {

        return mailingManager.sendMessages(sessionId);
    }

    @Override
    public String sendMessage(String sendingId, String sessionParagraph, Collection<File> filesToAttach) throws VradiException {
        return sendMessage(sendingId, sessionParagraph, null, false, filesToAttach);
    }

    @Override
    public String sendMessage(String sendingId, String sessionParagraph, String email, Collection<File> filesToAttach) throws VradiException {
        return sendMessage(sendingId, sessionParagraph, email, true, filesToAttach);
    }

    public String sendMessage(String sendingId, String sessionParagraph, String email, boolean reSend, Collection<File> filesToAttach) throws VradiException {
        return mailingManager.sendMessage(sendingId, sessionParagraph, email, reSend, filesToAttach);
    }

    @Override
    public void receiveMessages() throws VradiException {
        mailingManager.receiveMails();
    }

    @Override
    public Session stopSentMail(Session session) throws VradiException {

        // Flag status to stop traitement
        session.setStatus(SessionStatus.STOPPED.getValue());

        // Update
        session = wikittyProxy.store(session);

        return session;
    }

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     */
    @Override
    public void generatePDFForSession(String sessionId) throws VradiException {
        mailingManager.generatePDFForSession(sessionId);
    }

    /**
     * Get all the groups a user is in.
     *
     * @param userId the user we want the groups of
     * @return the group for our user
     * @throws VradiException for various possible errors
     */
    public List<Group> getGroupsForUser(String userId) throws VradiException {
        return clientManager.getGroupsForUser(userId);
    }

    @Override
    public String exportClientDB() throws VradiException {

        // creation du criteria wikitty
        // (export de tous les wikitty avec extension "client"
        // "groupement" ou "user"
        Search restriction = Search.query().or(); //definit un "groupement" qui se fera en "or"
        restriction.eq(Element.ELT_EXTENSION, Group.EXT_GROUP);
        restriction.eq(Element.ELT_EXTENSION, Client.EXT_CLIENT);
        restriction.eq(Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = restriction.criteria();

        // export
        String vscContent = exportAsCSV(criteria);

        return vscContent;
    }

    @Override
    public void deleteSession(String sessionId) throws VradiException {
        formManager.deleteSession(sessionId);
    }

    @Override
    public boolean checkTemplateExist(String extensionName) throws VradiException {
        boolean fileExists;

        try {
            WikittyExtension extension = formTypeManager.getFormType(extensionName);

            String templateName = formTypeManager.getFormTypeTemplateName(extension);

            if (templateName == null) {
                return false;
            }

            // just verify file existence
            // File file = storageService.getTemplate(extensionName, template);
            File template = formTypeManager.getTemplate(extensionName, templateName);

            fileExists = template != null && template.exists();
        } catch (Exception eee) {
            throw new VradiException("Can't check if template existe : ", eee);
        }
        return fileExists;
    }

    @Override
    public boolean removeFormsFromSession(String sessionId, List<String> formsIds) throws VradiException {
        return formManager.removeFormsFromSession(sessionId, formsIds);
    }

    @Override
    public Sending revertDeletion(String sendingId, List<String> formsIdsToUpdate) {
        return formManager.revertDeletion(sendingId, formsIdsToUpdate);
    }

    @Override
    public Sending revertUserDeletion(String sessionId, String userId, String formId) {
        return formManager.revertUserDeletion(sessionId, userId, formId);
    }

    @Override
    public List<Sending> revertGroupDeletion(String sessionId, String groupId, String formId) {
        return formManager.revertGroupDeletion(sessionId, groupId, formId);
    }

    @Override
    public Form updateForms(Form form, List<String> thesaurusIdsToAttach) throws VradiException {
        return formManager.updateForm(form, thesaurusIdsToAttach);
    }
}
