package org.chorem.pollen.votecounting;

/*
 * #%L
 * Pollen :: VoteCounting (Api)
 * $Id: VoteCounting.java 3715 2012-10-01 05:11:10Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.1/pollen-votecounting-api/src/main/java/org/chorem/pollen/votecounting/VoteCounting.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import org.chorem.pollen.votecounting.model.ChoiceToVoteRenderType;

import java.util.Locale;

/**
 * Contract of a vote counting.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.6
 */
public interface VoteCounting<S extends VoteCountingStrategy> {

    /**
     * Obtains a fresh instance of a vote coutning strategy.
     *
     * @return a new instance of vote counting strategy for this type
     *         of vote counting.
     */
    S newStrategy();

    /**
     * Obtains the unique id of this strategy.
     *
     * @return the unique id of this strategy.
     */
    int getId();

    /**
     * Get the value to display for a given vote value in the vote page.
     *
     * @param voteValue the vote value to display
     * @return the string representation of a vote value
     */
    String getDisplayVoteValue(Integer voteValue);

    /**
     * Test if the given value is a vote value (says users has filled it).
     *
     * @param voteValue the vote value to test
     * @return {@code true} if the given value is persisted.
     */
    boolean isChoiceInVote(Integer voteValue);

    /**
     * Get the vote counting strategy name to display in UI.
     *
     * @param locale the locale used to render the strategy name
     * @return the localized vote counting strategy name
     */
    String getName(Locale locale);

    /**
     * Get the vote counting short help to display in UI.
     *
     * @param locale the locale used to render the strategy name
     * @return the localized vote counting short help
     */
    String getShortHelp(Locale locale);

    /**
     * Get the vote counting strategy help to display in UI.
     *
     * @param locale the locale used to render the strategy name
     * @return the localized vote counting strategy help
     */
    String getHelp(Locale locale);

    /**
     * Tests if the given vote value is valid.
     *
     * @param voteValue the vote value to test
     * @return {@code true} if the given vote value is valid, {@code false}
     *         otherwise.
     */
    boolean isVoteValueValid(Integer voteValue);

    /**
     * If a vote value is not valid, gets the localized message of the error.
     *
     * @param locale     the locale used to render the error message
     * @param choiceName name of thechoice
     * @param voteValue  value fo the vote
     * @return the localized validation message
     */
    String getVoteValueNotValidMessage(Locale locale,
                                       String choiceName,
                                       String voteValue);

    /**
     * Tests if the total values of a vote is valid.
     *
     * @param totalValues the given total values
     * @return {@code true} if the total values of a vote is valid,
     *         {@code false} otherwhise.
     */
    boolean isTotalVoteValueValid(int totalValues);

    /**
     * If the total values of a vote is not valid, gets the localized error
     * message.
     *
     * @param locale the locale used to render the error message
     * @return the localized validation message
     */
    String getTotalVoteValueNotValidMessage(Locale locale);

    /**
     * Tests if a given vote value is null or not.
     *
     * @param value the vote value to test
     * @return {@code true} if the given vote value is null (or considered as
     *         null), {@code false} otherwise.
     */
    boolean isVoteValueNull(Integer value);

    /**
     * Gets the type of editor used to render a vote value.
     *
     * @return the type of editor used to render a vote value.
     * @see ChoiceToVoteRenderType
     */
    ChoiceToVoteRenderType getVoteValueEditorType();
}
