/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PreventRuleService.java 3257 2012-04-13 11:28:02Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/PreventRuleService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.services.PollenServiceSupport;

import java.util.Date;

/**
 * Manage prevent rules of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PreventRuleService extends PollenServiceSupport {
    
    public static final String SCOPE_VOTE = "vote";
    
    public static final String SCOPE_REMINDER = "rappel";

    /** Logger. */
    private static final Log log = LogFactory.getLog(PreventRuleService.class);

    public PreventRule createAddVotePreventRule() {
        PreventRule rule = newInstance(getDAO(PreventRule.class));
        rule.setScope(SCOPE_VOTE);
        rule.setSensibility(1);
        rule.setRepeated(true);
        return rule;
    }

    public PreventRule createRemindPreventRule(int sensibility) {
        PreventRule rule = newInstance(getDAO(PreventRule.class));
        rule.setScope(SCOPE_REMINDER);
        rule.setSensibility(sensibility);
        rule.setRepeated(false);
        return rule;
    }

    public void onVoteAdded(Poll poll, PollUrl pollVoteUrl, PollUrl modifURL) {

        if (poll.getEndDate() != null) {

            for (PreventRule rule : poll.getPreventRule()) {
                onVoteAdded(poll,
                            rule,
                            pollVoteUrl,
                            modifURL
                );
            }
        }
    }

    public void onPollToRemind(Poll poll, PollUrl pollVoteUrl) {

        // Parcours des preventRules de chaque sondage
        // envoi d'un email si endDate-nowDate <= preventRuleSensibility
        if (poll.getEndDate() != null) {

            Date now = new Date();

            // timeValue = endDate-nowDate
            long timeValue = poll.getEndDate().getTime() - now.getTime();

            if (log.isDebugEnabled()) {
                log.debug("Now: " + now.getTime() + " End: "
                          + poll.getEndDate().getTime() + " timeValue: "
                          + timeValue + "(" + timeValue / 3600000 + "h)");
            }

            int hourBeforeEnd = (int) (timeValue / 3600000);

            for (PreventRule rule : poll.getPreventRule()) {

                onPollToRemind(poll,
                               rule,
                               pollVoteUrl,
                               hourBeforeEnd
                );
            }
        }
    }

    protected void onVoteAdded(Poll poll,
                               PreventRule preventRule,
                               PollUrl pollVoteUrl,
                               PollUrl modifURL) {

        // send to creator a email to say hey a new vote was added!

        if (execute(SCOPE_VOTE, preventRule, 1)) {

            // ok can execute this task

            EmailService emailService = newService(EmailService.class);

            // let's send an email
            emailService.onVoteAdded(poll, pollVoteUrl, modifURL);

            // and update prevent rule
            commitTransaction("Could not update prevent rule");
        }
    }

    protected void onPollToRemind(Poll poll,
                                  PreventRule preventRule,
                                  PollUrl pollVoteUrl,
                                  int timeValue) {

        // send to all voting a email to say hey we need to vote

        if (execute(SCOPE_REMINDER, preventRule, timeValue)) {

            // ok can execute this task

            EmailService emailService = newService(EmailService.class);

            // let's send an email
            emailService.onPollReminder(poll, pollVoteUrl);

            // and update prevent rule
            commitTransaction("Could not update prevent rule");
        }
    }

    protected boolean execute(String scope,
                              PreventRule preventRule,
                              Integer value) {

        Preconditions.checkNotNull(scope);
        Preconditions.checkNotNull(value);

        // Si la règle n'est pas active, on ne fait rien
        if (!preventRule.isActive()) {
            log.debug("Règle non active");
            return false;
        }

        // Si la règle est à exécution unique, on la désactive
        if (preventRule.isOneTime()) {
            log.debug("Règle à exécution unique : désactivation");
            preventRule.setActive(false);
        }

        if (!scope.equals(preventRule.getScope())) {

            // pas bon scope
            return false;
        }

        // Si la règle est répétitive (exécution tous les n), on met à jour la valeur
        if (preventRule.isRepeated() && preventRule.getSensibility() != 0) {
            value = value % preventRule.getSensibility()
                    + preventRule.getSensibility();
        }

        if (value != preventRule.getSensibility()) {

            // pas la bonne valeur
            return false;
        }

        // ok on peut lancer la tâche
        return true;
    }

}
