/*
 * #%L
 * Pollen :: Persistence
 * $Id: PollDAOImpl.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-persistence/src/main/java/org/chorem/pollen/business/persistence/PollDAOImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.business.persistence;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenFunctions;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaQuery;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;

import java.util.List;
import java.util.Map;

public class PollDAOImpl<E extends Poll> extends PollDAOAbstract<E> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollDAOImpl.class);

    public List<E> getPolls(TopiaFilterPagerUtil.FilterPagerBean pager) throws TopiaException {

        Preconditions.checkNotNull(pager);

        long records = count();
        pager.setRecords((int) records);

        TopiaQuery query = createQuery("e");
        TopiaFilterPagerUtil.addPagerToQuery(query, pager);

        List<E> result = findAllByQuery(query);
        return result;
    }


    public List<E> getCreatedPolls(TopiaFilterPagerUtil.FilterPagerBean pager,
                                   UserAccount user) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(user);

        TopiaQuery countQuery = createQuery("e");
        countQuery.addWhere("e." + Poll.PROPERTY_CREATOR + "." + PollAccount.PROPERTY_USER_ACCOUNT, TopiaQuery.Op.EQ, user);
        long records = countByQuery(countQuery);
        pager.setRecords((int) records);

        TopiaQuery query = TopiaFilterPagerUtil.addPagerToQuery(countQuery, pager);

        List<E> result = findAllByQuery(query);
        return result;
    }

    public List<Pair<Poll, PollAccount>> getInvitedPolls(
            TopiaFilterPagerUtil.FilterPagerBean pager,
            UserAccount userToUse) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(userToUse);

        String email = userToUse.getEmail();

        TopiaQuery query = createQuery("p").
                setSelect("p", "l." + PersonToList.PROPERTY_POLL_ACCOUNT).
                addLeftJoin("p." + Poll.PROPERTY_VOTING_LIST, "v", false).
                addLeftJoin("v." + VotingList.PROPERTY_POLL_ACCOUNT_PERSON_TO_LIST, "l", false).
                addEquals("l." + PersonToList.PROPERTY_POLL_ACCOUNT
                          + "." + PollAccount.PROPERTY_EMAIL, email);

        List<Pair<Poll, PollAccount>> result = findAllWithPollAccounts(
                query, pager);
        return result;
    }

    public List<Pair<Poll, PollAccount>> getParticipatedPolls(
            TopiaFilterPagerUtil.FilterPagerBean pager,
            UserAccount userToUse) throws TopiaException {

        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(userToUse);

        VoteDAO voteDao = PollenDAOHelper.getVoteDAO(context);

        TopiaQuery query = voteDao.createQuery("e").
                addFrom(Poll.class, "p").
                addWhere("e in elements(p." + Poll.PROPERTY_VOTE + ")").
                setSelect("p", "e." + Vote.PROPERTY_POLL_ACCOUNT).
                addWhere("e." + Vote.PROPERTY_POLL_ACCOUNT + "." +
                         PollAccount.PROPERTY_USER_ACCOUNT,
                         TopiaQuery.Op.EQ, userToUse);

        List<Pair<Poll, PollAccount>> result = findAllWithPollAccounts(
                query, pager);
        return result;
    }

    protected List<Pair<Poll, PollAccount>> findAllWithPollAccounts(TopiaQuery query,
                                                                    TopiaFilterPagerUtil.FilterPagerBean pager)
            throws TopiaException {

        List<Object[]> queryResults = getContext().findByQuery(query);
        Map<Poll, PollAccount> mapResult = Maps.newLinkedHashMap();
        for (Object[] row : queryResults) {
            Poll poll = (Poll) row[0];
            PollAccount pollAccount = (PollAccount) row[1];
            mapResult.put(poll, pollAccount);
        }

        long records = mapResult.size();
        pager.setRecords((int) records);
        TopiaFilterPagerUtil.computeRecordIndexesAndPagesNumber(pager);
        List<Map.Entry<Poll, PollAccount>> entries =
                Lists.newLinkedList(mapResult.entrySet());
        List<Pair<Poll, PollAccount>> result =
                Lists.transform(TopiaFilterPagerUtil.getPageFromList(entries, pager),
                                PollenFunctions.MAP_ENTRY_TO_PAIR_FUNCTION);
        return result;
    }

    public List<E> getRunningPolls(boolean withEndDate) throws TopiaException {

        List<E> results;
        TopiaQuery query;

        if (withEndDate) {

            query = createQuery("poll").
                    addWhere("poll.endDate is not null and poll.endDate > current_timestamp()").
                    addWhere("poll.beginDate is null or poll.beginDate < current_timestamp()");

//                results = transaction
//                        .find("from "
//                              + Poll.class.getName()
//                              + " as poll where (poll.endDate is not null and poll.endDate > current_timestamp())"
//                              + " and (poll.beginDate is null or poll.beginDate < current_timestamp())");
        } else {
            query = createQuery("poll").
                    addWhere("poll.endDate is null or poll.endDate > current_timestamp()").
                    addWhere("poll.beginDate is null or poll.beginDate < current_timestamp()");

//                results = transaction
//                        .find("from "
//                              + Poll.class.getName()
//                              + " as poll where (poll.endDate is null or poll.endDate > current_timestamp())"
//                              + " and (poll.beginDate is null or poll.beginDate < current_timestamp())");
        }

        results = findAllByQuery(query);

        if (log.isDebugEnabled()) {
            log.debug("Entities found: "
                      + ((results == null) ? "null" : results.size()));
        }
        return results;
    }
} //PollDAOImpl<E extends Poll>
