/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPersonListConverter;
import org.chorem.pollen.business.converters.DataVotingListConverter;
import org.chorem.pollen.business.dto.PersonListDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.persistence.PersonList;
import org.chorem.pollen.business.persistence.PersonListDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des listes.
 *
 * @author enema
 * @author rannou
 * @version $Id: ServiceListImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServiceListImpl implements ServiceList {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;
    private VotingListDAO votingListDAO = null;
    private PersonListDAO personListDAO = null;
    private DataVotingListConverter votingListConverter = new DataVotingListConverter();
    private DataPersonListConverter personListConverter = new DataPersonListConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceListImpl.class);

    public ServiceListImpl() {
    }

    @Override
    public String createVotingList(VotingListDTO votingList) {
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            VotingList votingListEntity = votingListDAO.create();

            votingListConverter.setTransaction(transaction);
            votingListConverter.populateVotingListEntity(votingList,
                    votingListEntity);

            // Création des pollAccount de la liste si nécessaire
            ServicePollAccount spa = new ServicePollAccountImpl();
            //spa.createPollAccounts(votingList.getPollAccountDTOs(), transaction);
            for (PollAccountDTO account : votingList.getPollAccountDTOs()) {
                if (spa.findPollAccountById(account.getId()) == null) {
                    account.setId(spa.createPollAccount(account));
                }
            }

            // Création des PersonToList
            votingListConverter.populatePersonVotingList(votingList,
                    votingListEntity);

            topiaId = votingListEntity.getTopiaId();

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return "";
        }
    }

    @Override
    public List<VotingList> createVotingLists(List<VotingListDTO> votingLists,
            String pollId, TopiaContext trans) {
        if (votingLists.isEmpty() || pollId.length() == 0) {
            return null;
        }

        List<VotingList> result = null;
        try {
            transaction = trans;

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            votingListConverter.setTransaction(transaction);

            result = new ArrayList<VotingList>();
            for (VotingListDTO votingList : votingLists) {
                VotingList votingListEntity = votingListDAO.create();
                votingList.setPollId(pollId);
                votingListConverter.populateVotingListEntity(votingList,
                        votingListEntity);

                // Création des pollAccount de la liste si nécessaire
                ServicePollAccount spa = new ServicePollAccountImpl();
                //spa.createPollAccounts(votingList.getPollAccountDTOs(), trans);
                for (PollAccountDTO account : votingList.getPollAccountDTOs()) {
                    if (spa.findPollAccountById(account.getId()) == null) {
                        account.setId(spa.createPollAccount(account));
                    }
                }

                // Création des PersonToList
                votingListConverter.populatePersonVotingList(votingList,
                        votingListEntity);
                result.add(votingListEntity);
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public boolean updateVotingList(VotingListDTO votingList) {
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            VotingList entity = votingListDAO.findByTopiaId(votingList.getId());

            votingListConverter.setTransaction(transaction);
            votingListConverter.populateVotingListEntity(votingList, entity);

            votingListDAO.update(entity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + votingList.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deleteVotingList(String votingListId) {
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            VotingList entity = votingListDAO.findByTopiaId(votingListId);

            votingListDAO.delete(entity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + votingListId);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public VotingListDTO findVotingListById(String votingListId) {
        VotingListDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            VotingList votingListEntity = votingListDAO
                    .findByTopiaId(votingListId);

            if (votingListEntity != null) {
                votingListConverter.setTransaction(transaction);
                result = votingListConverter
                        .createVotingListDTO(votingListEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<VotingListDTO> findVotingListByName(String name) {
        List<VotingListDTO> results = null;
        List<VotingList> votingListEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            votingListEntities = votingListDAO.findAllByName(name);

            votingListConverter.setTransaction(transaction);
            results = votingListConverter
                    .createVotingListDTOs(votingListEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<VotingListDTO> selectVotingList(Map<String, Object> properties) {
        List<VotingListDTO> results = null;
        List<VotingList> votingListEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            votingListDAO = PollenModelDAOHelper.getVotingListDAO(transaction);

            if (properties == null) {
                votingListEntities = votingListDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : toutes les listes ont été sélectionnés !");
                }
            } else {
                votingListEntities = votingListDAO
                        .findAllByProperties(properties);
            }
            votingListConverter.setTransaction(transaction);
            results = votingListConverter
                    .createVotingListDTOs(votingListEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public String createPersonList(PersonListDTO personList) {
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);
            PersonList entity = personListDAO.create();

            personListConverter.setTransaction(transaction);
            personListConverter.populatePersonListEntity(personList, entity);

            // Création des comptes de la liste
            ServicePollAccount spa = new ServicePollAccountImpl();
            for (PollAccountDTO account : personList.getPollAccountDTOs()) {
                account.setPersonListId(entity.getTopiaId());
            }
            entity.setPollAccount(spa.createPollAccounts(personList
                    .getPollAccountDTOs(), transaction));

            topiaId = entity.getTopiaId();

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return "";
        }
    }

    @Override
    public boolean createPersonLists(List<PersonListDTO> personLists,
            String userId, TopiaContext trans) {
        if (personLists.isEmpty() || userId.length() == 0) {
            return false;
        }

        try {
            transaction = trans;

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            personListConverter.setTransaction(transaction);
            for (PersonListDTO personList : personLists) {
                PersonList entity = personListDAO.create();

                // Création des comptes de la liste
                ServicePollAccount spa = new ServicePollAccountImpl();
                for (PollAccountDTO account : personList.getPollAccountDTOs()) {
                    account.setPersonListId(entity.getTopiaId());
                }
                entity.setPollAccount(spa.createPollAccounts(personList
                        .getPollAccountDTOs(), transaction));

                personList.setUserId(userId);
                personListConverter
                        .populatePersonListEntity(personList, entity);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean updatePersonList(PersonListDTO personList) {
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            PersonList entity = personListDAO.findByTopiaId(personList.getId());

            personListConverter.setTransaction(transaction);
            personListConverter.populatePersonListEntity(personList, entity);

            personListDAO.update(entity);

            // mise à jour ou création des comptes
            ServicePollAccount spa = new ServicePollAccountImpl();
            for (PollAccountDTO pollAccountDTO : personList
                    .getPollAccountDTOs()) {
                pollAccountDTO.setPersonListId(entity.getTopiaId());
                boolean updated = spa.updatePollAccount(pollAccountDTO);
                if (!updated) {
                    pollAccountDTO.setId(spa.createPollAccount(pollAccountDTO));
                }

                if (log.isDebugEnabled()) {
                    log.debug("PollAccount " + pollAccountDTO.getVotingId()
                            + " (" + pollAccountDTO.getId() + ") updated: "
                            + updated);
                }
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + personList.getId());
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deletePersonList(String personListId) {
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            PersonList entity = personListDAO.findByTopiaId(personListId);

            personListDAO.delete(entity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + personListId);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public PersonListDTO findPersonListById(String personListId) {
        PersonListDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            PersonList personListEntity = personListDAO
                    .findByTopiaId(personListId);

            if (personListEntity != null) {
                personListConverter.setTransaction(transaction);
                result = personListConverter
                        .createPersonListDTO(personListEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PersonListDTO> findPersonListByName(String name) {
        List<PersonListDTO> results = null;
        List<PersonList> personListEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            personListEntities = personListDAO.findAllByName(name);

            personListConverter.setTransaction(transaction);
            results = personListConverter
                    .createPersonListDTOs(personListEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PersonListDTO> findPersonListByUser(String userId) {
        List<PersonListDTO> results = null;
        List<PersonList> personListEntities = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            personListEntities = new ArrayList<PersonList>();
            personListEntities.addAll(user.getFavoriteList());

            personListConverter.setTransaction(transaction);
            results = personListConverter
                    .createPersonListDTOs(personListEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PersonListDTO> selectPersonList(Map<String, Object> properties) {
        List<PersonListDTO> results = null;
        List<PersonList> personListEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            personListDAO = PollenModelDAOHelper.getPersonListDAO(transaction);

            if (properties == null) {
                personListEntities = personListDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : toutes les listes ont été sélectionnés !");
                }
            } else {
                personListEntities = personListDAO
                        .findAllByProperties(properties);
            }
            personListConverter.setTransaction(transaction);
            results = personListConverter
                    .createPersonListDTOs(personListEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public String createPersonToList(PollAccountDTO account) {
        //TODO
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean updatePersonToList(PollAccountDTO account) {
        //TODO
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean deletePersonToList(String accountId) {
        //TODO
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}