/*
 * #%L
 * Lima Swing
 * *
 * $Id: LetteringViewHandler.java 3674 2013-05-03 14:59:15Z athimel $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/lettering/LetteringViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.lettering;

import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.beans.LetteringFilterImpl;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.api.EntryBookService;
import org.chorem.lima.business.api.FinancialPeriodService;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;

import javax.swing.JComboBox;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;


/**
 * Handler associated with financial transaction view.
 *
 * @author chatellier
 * @version $Revision: 3674 $
 *          <p/>
 *          Last update : $Date: 2013-05-03 16:59:15 +0200 (Fri, 03 May 2013) $
 *          By : $Author: athimel $
 */
public class LetteringViewHandler{

    protected LetteringView view;
    protected LetteringTable table;

    /** Transaction service. */
    protected FiscalPeriodService fiscalPeriodService;
    protected FinancialPeriodService financialPeriodService;
    protected AccountService accountService;
    protected FinancialTransactionService financialTransactionService;
    protected EntryBookService entryBookService;

    protected LetteringFilterImpl filter;

    protected BigDecimal debit = BigDecimal.ZERO;
    protected BigDecimal credit = BigDecimal.ZERO;
    protected BigDecimal solde = BigDecimal.ZERO;
    protected LettringSelectionModel lettringSelectionModel;
    protected LetteringEditModel editModel;
    protected enum ButtonMode {DELETTRED, LETTRED, EQUALIZED, ALL}
    private static final Log log = LogFactory.getLog(LetteringViewHandler.class);

    public LetteringViewHandler(LetteringView view) {
        this.view = view;
        financialPeriodService = LimaServiceFactory.getService(FinancialPeriodService.class);
        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        accountService = LimaServiceFactory.getService(AccountService.class);
        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);
        entryBookService = LimaServiceFactory.getService(EntryBookService.class);
    }

    /**
     * Init all combo box in view.
     */
    public void init() {
        filter = view.getFilterModel();
        editModel = view.getEditModel();
        lettringSelectionModel = view.getLettringSelectionModel();
        loadComboAndRows();

        filter.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateAllEntries();
            }
        });
    }

    public void balanceAndActions() {
        if (log.isDebugEnabled()) {
            log.debug("balanceAndActions");
        }
        if (view.getTable().getSelectedRows().length == 0) {
            onButtonModeChanged(ButtonMode.ALL);
            onBalanceChanged(null);
        } else if (!letteringNotExist(view.getTable().getSelectedRow())) {

            //lettred entries
            onBalanceChanged(null);
            setValuesForSelectedEntries();

            //For U.I. buttons (Lettering and unlettering)
            onButtonModeChanged(ButtonMode.DELETTRED);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("unlettred entries");
            }
            int[] selectedRows = view.getTable().getSelectedRows();
            if (selectedRows.length == 2) {
                if (log.isDebugEnabled()) {
                    log.debug("2 rows selected");
                }
                /*Treatment only if one of values contains decimals*/
                LetteringTableModel tableModel = view.getTableModel();
                Entry firstSelectedEntry = tableModel.getEntryAt(selectedRows[0]);
                Entry secondSelectedEntry = tableModel.getEntryAt(selectedRows[1]);

                /*Get decimals*/
                BigDecimal firstSelectedEntryAmount = firstSelectedEntry.getAmount();
                BigDecimal secondSelectedEntryAmount = secondSelectedEntry.getAmount();

                if ( secondSelectedEntry.getDebit() != firstSelectedEntry.getDebit()
                     && (firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ZERO) >0
                         && firstSelectedEntryAmount.subtract(secondSelectedEntryAmount).abs().compareTo(BigDecimal.ONE) <0) ) {
                    onButtonModeChanged(ButtonMode.EQUALIZED);
                }
            }else {
                if (log.isDebugEnabled()) {
                    log.debug("!2 rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }

            //Unlettred entries
            onBalanceChanged(null);
            //treatment unuseful if no rows are selected
            if (!view.getLettringSelectionModel().isSelectionEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("Rows selected");
                }
                setValuesForSelectedEntries();
                onButtonModeChanged(ButtonMode.LETTRED);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No Rows selected");
                }
                onButtonModeChanged(ButtonMode.ALL);
            }
        }
    }

    /**return true if lettering is null, or not null but empty
     * @param row index of the line to test
     * @return boolean
     * */
    public boolean letteringNotExist(int row){
        boolean emptyOrNull = false;
        if (row != -1) {
            Entry entry = view.getTableModel().getEntryAt(row);
            String lettering = entry.getLettering();
            emptyOrNull = (lettering==null||lettering.isEmpty());
        }
        return emptyOrNull;
    }

    public void onButtonModeChanged(ButtonMode buttonMode) {

        switch (buttonMode) {
            case DELETTRED :
                editModel.setLettred(false);
                editModel.setUnLettred(true);
                break;
            case LETTRED:
                editModel.setUnLettred(false);
                editModel.setLettred(true);
                break;
            case EQUALIZED:
                editModel.setEqualized(true);
                break;
            default:
                editModel.setLettred(false);
                editModel.setUnLettred(false);
                editModel.setEqualized(false);
        }
    }

    public void setValuesForSelectedEntries() {
        Entry selectedEntry;
        LetteringTableModel tableModel = view.getTableModel();
        for (int i = 0; i < tableModel.getRowCount(); i ++){
            if (view.getLettringSelectionModel().isSelectedIndex(i)){
                selectedEntry = tableModel.getEntryAt(i);
                //Set values for calculation (By LetteringEditModel) of balance
                onBalanceChanged(selectedEntry);
            }
        }
    }

    public void onBalanceChanged(Entry balance) {
        if (balance == null) {
            editModel.setCredit(BigDecimal.ZERO);
            editModel.setDebit(BigDecimal.ZERO);
            editModel.setSolde(BigDecimal.ZERO, false);
        } else {
            balanceCalculation(balance.getAmount(), balance.getDebit());
        }
    }

    /**Allow to add / subtract credit / debit and balance
     * @param amount debit or credit
     * @param debit it indicate if amount is debit or not
     * */
    public void balanceCalculation(BigDecimal amount, boolean debit){

        BigDecimal debitVal = debit ? amount : BigDecimal.ZERO;
        BigDecimal creditVal = debit ? BigDecimal.ZERO : amount;

        if (log.isDebugEnabled()) {
            log.debug("Balance calculation");
        }

        if (debitVal.equals(BigDecimal.ZERO)){

            if (!creditVal.equals(BigDecimal.ZERO)){

                editModel.setCredit(creditVal);
                editModel.setSolde(creditVal, true);
            }
        }else if (creditVal.equals(BigDecimal.ZERO)){
            editModel.setDebit(debitVal);
            editModel.setSolde(debitVal, false);
        }else{
            onBalanceChanged(null);
        }
    }

    public void loadComboAndRows(){

        List<Account> allAccounts = accountService.getAllAccounts();
        view.getAccountComboBoxModel().setObjects(allAccounts);

        if (!allAccounts.isEmpty()) {

            view.getAccountComboBox().setSelectedItem(allAccounts.get(0));
        }

        //By default, we have the beginning of the fiscal period (Or of current
        //date if no fiscal period) and the end of the current date
        FiscalPeriod fiscalPeriod = fiscalPeriodService.getLastFiscalPeriod();
        Date defaultDateBegFiscalPeriod;

        Calendar calendar = Calendar.getInstance();
        int dernierJourMoisCourant = calendar.getActualMaximum(Calendar.DATE);
        int premierJourMoisCourant = calendar.getActualMinimum(Calendar.DATE);

        if (fiscalPeriod != null){
            defaultDateBegFiscalPeriod = fiscalPeriodService.getLastFiscalPeriod().getBeginDate();
        }else{
           defaultDateBegFiscalPeriod = DateUtils.setDays(new Date(), premierJourMoisCourant);
        }

        Date defaultDateEndCurrent = DateUtils.setDays(new Date(), dernierJourMoisCourant);


        view.getPickerDebut().setDate(defaultDateBegFiscalPeriod);
        view.getPickerFin().setDate(defaultDateEndCurrent);
        filter.setDateStart(defaultDateBegFiscalPeriod);
        filter.setDateEnd(defaultDateEndCurrent);
        filter.setDisplayUnlettred(true);

        updateAllEntries();
    }

    protected List<Entry> findAllEntries(LetteringFilterImpl filter){
        if (filter != null) {
            List<Entry> entries =
                    financialTransactionService.getAllEntrieByDatesAndAccountAndLettering(filter);
            return  entries;
        }
        return null;
    }

    public void updateAllEntries() {

        if (isFilterValid()) {

            List<Entry> entries = findAllEntries(filter);
            List<String> journalEntrees = new ArrayList<String>();
            List<Date> datesEntree = new ArrayList<Date>();

            for (Entry entry : entries){
                FinancialTransaction financialTransaction = entry.getFinancialTransaction();
                datesEntree.add(financialTransaction.getTransactionDate());
                if (financialTransaction.getEntryBook() == null ||
                    financialTransaction.getEntryBook().getLabel() == null) {
                    journalEntrees.add("");
                } else {
                    journalEntrees.add(financialTransaction.getEntryBook().getLabel());
                }

            }

            view.getTableModel().updateEntries(entries, datesEntree, journalEntrees);
        }
        onBalanceChanged(null);
    }

    /**To make the difference between two selected entries and
     * create a new entry with the result (debit or credit).
     * It allow to letter somme entries with different debit and credit
     * */
    public void roundAndCreateEntry() {

        LetteringTableModel tableModel = view.getTableModel();

        int[] selectedRows = view.getTable().getSelectedRows();
        if (selectedRows.length == 2) {
            /*Treatment only if one of values contains decimals*/
            Entry firstSelectedEntry = tableModel.getEntryAt(selectedRows[0]);
            Entry secondSelectedEntry = tableModel.getEntryAt(selectedRows[1]);

            Entry[] newEntriesFormEqualizing = financialTransactionService.getEntriesFromEqualizing(firstSelectedEntry, secondSelectedEntry);

            /*Add new entries to the model and the table*/
            Entry newSameAccountEntry = newEntriesFormEqualizing[0];
            Entry newCostOrProductEntry = newEntriesFormEqualizing[1];
            tableModel.addEntry(newSameAccountEntry, newSameAccountEntry.getFinancialTransaction().getTransactionDate());
            tableModel.addEntry(newCostOrProductEntry, newCostOrProductEntry.getFinancialTransaction().getTransactionDate());

            /*Re-select the two entries (firstSelectedEntry and secondSelectedEntry)
            * and the new sameAccountEntry
            * */
            view.getLettringSelectionModel().selectRoundedAndNewEntries(selectedRows[0], selectedRows[1], newSameAccountEntry);
        }
    }

    /**To test if the filter contain an account and a period
     * @return true if filter is valid
     * */
    protected boolean isFilterValid() {
        boolean valid = false;

        if (filter.getAccount() != null && filter.getDateStart() != null && filter.getDateEnd() != null) {
            valid = true;
        }

        return valid;
    }

    /**
     * Select previous value in combo box.
     *
     * @param comboBox combo box
     */
    public void back(JComboBox comboBox) {
        int row = comboBox.getSelectedIndex();

        if (row > 0) {
            comboBox.setSelectedIndex(row - 1);
        }
        view.getLettringSelectionModel().clearSelection();
    }

    /**
     * Select next value in combo box.
     *
     * @param comboBox combo box
     */
    public void next(JComboBox comboBox) {
        int size = comboBox.getModel().getSize();
        int row = comboBox.getSelectedIndex();

        if (row < size - 1) {
            comboBox.setSelectedIndex(row + 1);
        }
        view.getLettringSelectionModel().clearSelection();
    }

    /**Add a group of three letters to n entries*/
    public void addLetter() {
        String newLetters = financialTransactionService.getNextLetters();
        changeLetter(newLetters);
        onButtonModeChanged(ButtonMode.DELETTRED);
    }

    /**Remove a group of three letters to n entries*/
    public void removeLetter() {
        changeLetter(null);
        onButtonModeChanged(ButtonMode.LETTRED);
    }

    /**Add or remove a group of three letters to n entries*/
    protected void changeLetter(String newLetters) {

        int[] entrieSelected = view.getTable().getSelectedRows();

        view.getTableModel().updateLettersSelectedEntries(entrieSelected, newLetters);

        for (int indexEntry : entrieSelected){
            financialTransactionService.updateEntry(view.getTableModel().getEntryAt(indexEntry));
        }
    }

}
