/*
 * #%L
 * Lima Swing
 * *
 * $Id: LetteringTableModel.java 3647 2012-09-18 15:59:06Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/lettering/LetteringTableModel.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.lettering;

import org.chorem.lima.LimaContext;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.nuiton.util.decorator.Decorator;
import org.nuiton.util.decorator.DecoratorProvider;

import javax.swing.table.AbstractTableModel;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Basic transaction table model.
 * <p/>
 * Le modele est filtré sur {@link #selectedBeginDate} et
 * {@link #selectedEndDate} (montée en charge !).
 *
 * @author ore
 * @author chatellier
 */
public class LetteringTableModel extends AbstractTableModel{

    /** serialVersionUID. */
    private static final long serialVersionUID = 1L;

    /** Begin Date. */
    protected Date selectedBeginDate;

    /** EndDate. */
    protected Date selectedEndDate;

    protected List<Entry> entries;
    protected List<Date> datesEntree;
    protected List<String> entryBooks;

    /**
     * To decorate account objects.
     */
    protected final Decorator<Account> accountDecorator;

    public LetteringTableModel() {
        DecoratorProvider decoratorProvider =
                LimaContext.get().getDecoratorProvider();
        accountDecorator = decoratorProvider.getDecoratorByType(Account.class);
    }

    public List<Entry> getEntries(){
        return entries;
    }

    public Date getDateEntree(int row) {
        return datesEntree.get(row);
    }

    public Integer getIndexOfEntry(Entry entrySearch){
        return entries.indexOf(entrySearch);
    }

    public int getNumberOfEntries(){
        return entries.size();
    }

    public void updateEntries(List<Entry> entries,List<Date> datesEntree, List<String> entryBooks) {

        this.entries = entries;
        setDate(datesEntree);
        setJournal(entryBooks);
        fireTableDataChanged();
    }

    protected void setDate(List<Date> datesEntree){
        this.datesEntree=datesEntree;
    }

    protected void setJournal(List<String> entryBooks){
        this.entryBooks = entryBooks;
    }

    @Override
    public int getColumnCount() {
        return 7;
    }

    public Entry getEntryAt(int row) {
        if (row == -1) {
            return null;
        }

        return entries.get(row);
    }

    @Override
    public Class<?> getColumnClass(int column) {

        Class<?> result = String.class;

        switch (column) {
            case 0:
                result = Date.class;
                break;
            case 1:
                result = String.class;
                break;
            case 2:
                result = String.class;
                break;
            case 3:
                result = Account.class;
                break;
            case 4:
                result = String.class;
                break;
            case 5:
                result = BigDecimal.class;
                break;
            case 6:
                result = BigDecimal.class;
                break;
        }

        return result;
    }

    @Override
    public String getColumnName(int column) {
        String result = "n/a";

        switch (column) {
            case 0:
                result = _("lima.table.date");
                break;
            case 1:
                result = _("lima.table.account");
                break;
            case 2:
                result = _("lima.table.voucher");
                break;
            case 3:
                result = _("lima.table.description");
                break;
            case 4:
                result = _("lima.table.letter");
                break;
            case 5:
                result = _("lima.table.debit");
                break;
            case 6:
                result = _("lima.table.credit");
                break;
        }

        return result;
    }

    @Override
    public int getRowCount() {
        if (entries != null){
            return entries.size();
        }
        return 0;
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;

        if (entries.size() > 0){
            Entry currentEntry = entries.get(row);
            switch (column) {
                case 0:
                    result = datesEntree.get(row); // date
                    break;
                case 1: // account
                    Account acc = currentEntry.getAccount();
                    result = accountDecorator.toString(acc);
                    break;
                case 2:
                    result = currentEntry.getVoucher();
                    break;
                case 3:
                    result = currentEntry.getDescription();
                    break;
                case 4:
                    result = currentEntry.getLettering();
                    break;
                case 5:
                    result = currentEntry.getDebit() ? currentEntry.getAmount() : BigDecimal.ZERO;
                    break;
                case 6:
                    result = currentEntry.getDebit() ? BigDecimal.ZERO : currentEntry.getAmount();
                    break;
            }
        }
        return result;
    }

    /**
     * Only cell of "Letter" could be editable
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    public Object getElementAt(int row) {
        return entries.get(row);
    }

    /**add one entry*/
    public void addEntry(Entry entry, Date dateEntree) {
        entries.add(entry);
        datesEntree.add(dateEntree);
        int row = entries.indexOf(entry);
        fireTableRowsInserted(row, row);
    }

    /**update selected entries*/
    public void updateLettersSelectedEntries(int[] entries, String letters) {
        for (int rowEntry : entries){
            getEntryAt(rowEntry).setLettering(letters);
            fireTableRowsUpdated(rowEntry, rowEntry);
        }
    }

}
