/*
 * #%L
 * Lima Swing
 * 
 * $Id: ImportExport.java 3631 2012-08-22 15:43:11Z mallon $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/importexport/ImportExport.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.importexport;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.util.concurrent.ExecutionException;

import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingWorker;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.api.ExportService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.business.utils.ImportExportEntityEnum;
import org.chorem.lima.enums.EncodingEnum;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.jdesktop.swingx.painter.BusyPainter;

/**
 * Import export helper.
 * Calling service with specified file to import and displaying wait view to user.
 * 
 * @author echatellier
 * Date : 24 avr. 2012
 */
public class ImportExport {

    private static final Log log = LogFactory.getLog(ImportExport.class);

    /** Parent view. */
    protected Component viewComponent;

    /**
     * @deprecated etat de la classe, non multi thread, gerer ca différement.
     */
    protected EncodingEnum encodingEnum;

    protected ImportService importService;

    protected ExportService exportService;

    private ImportExportWaitView waitView;

    public ImportExport(Component view) {
        viewComponent = view;

        //services
        importService = LimaServiceFactory.getService(ImportService.class);
        exportService = LimaServiceFactory.getService(ExportService.class);

        //create the wait dialog panel
        waitView = new ImportExportWaitView();
        waitView.waitDialog.setSize(400, 200);
        waitView.setLocationRelativeTo(view);
        BusyPainter busyPainter = waitView.getBusylabel().getBusyPainter();
        busyPainter.setHighlightColor(new Color(44, 61, 146).darker());
        busyPainter.setBaseColor(new Color(168, 204, 241).brighter());
    }

    /**
     * Call the appropriate method in business service.
     *
     * @param importExportMethode
     * @param file
     * @param verbose
     */
    public void importExport(ImportExportEnum importExportMethode, String file, boolean verbose) {
        final ImportExportEnum importExportMethodeF = importExportMethode;
        encodingEnum = EncodingEnum.UTF8;
        if (file.equals("")) {
            file = chooseFile(importExportMethode.getImportMode(), importExportMethode);
        }
        //if export cancel
        if (!file.equals("")) {
            final EncodingEnum charset = encodingEnum;
            final String filePath = file;
            final Boolean verboseMode = verbose;
            final Boolean importMode = importExportMethode.getImportMode();
            new SwingWorker<String, Void>() {
                @Override
                protected String doInBackground() {
                    String datas;
                    String result = "";
                    switch (importExportMethodeF) {
                        case CSV_ALL_EXPORT:
                            datas = exportService.exportAsCSV();
                            createFile(filePath, charset.getEncoding(), datas);
                            break;
                        case CSV_ACCOUNTCHARTS_EXPORT:
                            datas = exportService.exportAccountsChartAsCSV();
                            createFile(filePath, charset.getEncoding(), datas);
                            break;
                        case CSV_ENTRYBOOKS_EXPORT:
                            datas = exportService.exportEntryBookChartAsCSV();
                            createFile(filePath, charset.getEncoding(), datas);
                            break;
                        case CSV_FINANCIALSTATEMENTS_EXPORT:
                            datas = exportService.exportFinancialStatementChartAsCSV();
                            createFile(filePath, charset.getEncoding(), datas);
                            break;
                        case CSV_VAT_EXPORT:
                            datas = exportService.exportVatStatementChartAsCSV();
                            createFile(filePath, charset.getEncoding(), datas);
                            break;
                        case EBP_ACCOUNTCHARTS_EXPORT:
                            //For windows ebp
                            datas = exportService.exportAccountsAsEBP();
                            createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), datas);
                            break;
                        case EBP_ENTRIES_EXPORT:
                            //For windows ebp
                            datas = exportService.exportEntriesAsEBP();
                            createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), datas);
                            break;
                        case CSV_ALL_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAllAsCSV(datas);
                            break;
                        case CSV_ACCOUNTCHARTS_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsCSV(datas, ImportExportEntityEnum.ACCOUNT);
                            break;
                        case CSV_ENTRYBOOKS_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsCSV(datas, ImportExportEntityEnum.ENTRYBOOK);
                            break;
                        case CSV_FINANCIALSTATEMENTS_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsCSV(datas, ImportExportEntityEnum.FINANCIALSTATEMENT);
                            break;
                        case CSV_VAT_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsCSV(datas, ImportExportEntityEnum.VATSTATEMENT);
                            break;
                        case CSV_ENTRIES_IMPORT:
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsCSV(datas, ImportExportEntityEnum.ENTRY);
                            break;
                        case PDF_VAT_IMPORT:
                            int response = JOptionPane.showConfirmDialog(waitView,
                                                                         _("lima.importexport.usevatpdf"),
                                                                         _("lima.common.confirmation"), JOptionPane.YES_NO_OPTION);
                            datas = extractFile(filePath, charset.getEncoding());
                            result = importService.importAsPDF(filePath, ImportExportEntityEnum.VATPDF, response == JOptionPane.YES_OPTION);
                            break;
                        case EBP_ACCOUNTCHARTS_IMPORT:
                            //For windows ebp
                            datas = extractFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
                            result = importService.importAccountsChartFromEbp(datas);
                            break;
                        case EBP_ENTRIES_IMPORT:
                            //For windows ebp
                            datas = extractFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
                            result = importService.importEntriesFromEbp(datas);
                            break;
                    }
                    return result;
                }

                @Override
                protected void done() {
                    try {

                        if (log.isDebugEnabled()) {
                            log.debug("importMode : " + importMode);
                        }

                        //hidde wait dialog panel
                        waitView.setVisible(false);
                        // display result dialog
                        if (verboseMode) {
                            String result = get();
                            if (result.equals(null)) {
                                JOptionPane.showMessageDialog(viewComponent, _("lima.ui.importexport.importerror"),
                                        _("lima.ui.importexport.importtitle"), JOptionPane.ERROR_MESSAGE);
                                
                            } else {
                                if (importMode) {
                                    if (log.isDebugEnabled()) {
                                        log.debug("importMode");
                                        log.debug("importExportMethodeF : " + importExportMethodeF);
                                    }
                                    //special message when importing a VAT PDF
                                    if (importExportMethodeF.equals(ImportExportEnum.PDF_VAT_IMPORT)) {
                                        if (log.isDebugEnabled()) {
                                            log.debug("PDF_VAT_IMPORT");
                                        }
                                        JOptionPane.showMessageDialog(
                                                waitView,
                                                _("lima.ui.importexport.import.vatpdfimport"),
                                                _("lima.ui.importexport.import"),
                                                JOptionPane.INFORMATION_MESSAGE);
                                    } else {
                                        if (log.isDebugEnabled()) {
                                            log.debug("import.terminated");
                                        }
                                        JOptionPane.showMessageDialog(
                                                waitView,
                                                _("lima.ui.importexport.import.terminated"),
                                                _("lima.ui.importexport.import"),
                                                JOptionPane.INFORMATION_MESSAGE);
                                    }
                                } else {
                                    if (log.isDebugEnabled()) {
                                        log.debug("export.terminated");
                                    }
                                    JOptionPane.showMessageDialog(
                                            waitView,
                                            _("lima.ui.importexport.export.terminated"),
                                            _("lima.ui.importexport.export"),
                                            JOptionPane.INFORMATION_MESSAGE);
                                }
                            }
                        }
                    } catch (InterruptedException ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't get result message", ex);
                        }
                    } catch (ExecutionException ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't get result message", ex);
                        }
                    }
                }
            }.execute();
            waitView.setVisible(true);
        }
    }


    /**
     * open choose file dialog with appropriate file mode view
     * folders for export or folders+files for import
     *
     * @param importMode
     * @param importExportMethode
     * @return
     */
    public String chooseFile(Boolean importMode, ImportExportEnum importExportMethode) {
        String filePath = "";
        JFileChooser chooser = new JFileChooser();
        //Encoding option

        JComboBox comboBox = new JComboBox(EncodingEnum.descriptions());
        if (importExportMethode.getEncodingOption()) {
            JPanel panel = new JPanel();
            panel.setLayout(new BorderLayout());
            panel.add(new JLabel(_("lima.importexport.choiceencoding")), BorderLayout.WEST);
            panel.add(comboBox, BorderLayout.CENTER);
            ((Container) chooser.getComponent(2)).add(panel, BorderLayout.SOUTH);
        }

        String approveButtonText;
        if (importMode) {
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            approveButtonText = _("lima.ui.importexport.import");
            chooser.setDialogTitle(approveButtonText);
            chooser.setApproveButtonText(approveButtonText);
        } else {
            approveButtonText = _("lima.ui.importexport.export");
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setDialogTitle(approveButtonText);
            chooser.setApproveButtonText(approveButtonText);
        }
        if (chooser.showOpenDialog(viewComponent) == JFileChooser.APPROVE_OPTION) {
            filePath = chooser.getSelectedFile().getAbsolutePath();

            if (importExportMethode.getEncodingOption()) {
                encodingEnum = EncodingEnum.valueOfDescription((String) comboBox.getSelectedItem());
            }
        }

        return filePath;
    }


    /**
     * Get csv datas in string and write file.
     * 
     * @param filePath
     * @param charset
     * @param datas
     */
    public void createFile(String filePath, String charset, String datas) {
        File file = new File(filePath);
        BufferedWriter out = null;
        try {
            out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), charset));
            out.write(datas);
            out.flush();
            out.close();
        } catch (IOException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't write file " + filePath, eee);
            }
        } finally {
            IOUtils.closeQuietly(out);
        }
        
    }

    /**
     * Open csv file and get his datas on a string.
     * 
     * @param filePath
     * @param charset
     * @return
     */
    public String extractFile(String filePath, String charset) {
        StringWriter sw = new StringWriter();
        BufferedReader in = null;
        try {
            InputStream is;

            // FIXME echatellier 20101129 remove this hack and modify resources reading mecanism
            if (filePath != null && filePath.indexOf(".jar!") > 0) {
                String resourcesPath = filePath.substring(filePath.indexOf(".jar!") + 5);
                is = ImportExport.class.getResourceAsStream(resourcesPath);
            } else {
                is = new FileInputStream(filePath);
            }

            in = new BufferedReader(new InputStreamReader(is, charset));
            IOUtils.copy(in, sw);
            in.close();
        } catch (IOException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't read file " + filePath, eee);
            }
        } finally {
            IOUtils.closeQuietly(in);
        }

        return sw.toString();
    }
}
