/*
 * #%L
 * Lima Swing
 * 
 * $Id: FiscalPeriodViewHandler.java 3607 2012-08-14 13:54:14Z mallon $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/fiscalperiod/FiscalPeriodViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.fiscalperiod;

import static org.nuiton.i18n.I18n._;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.DefaultListSelectionModel;
import javax.swing.JOptionPane;
import javax.swing.event.ListSelectionEvent;

import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.EntryBookImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;
import org.jdesktop.swingx.JXTable;
import org.nuiton.util.DateUtil;

/**
 * Handler pour la gestion des exercices (creation/fermeture).
 *
 * @author chatellier
 * @version $Revision: 3607 $
 * 
 * Last update : $Date: 2012-08-14 15:54:14 +0200 (Tue, 14 Aug 2012) $
 * By : $Author: mallon $
 */
public class FiscalPeriodViewHandler implements ServiceListener {

    private static final Log log = LogFactory.getLog(FiscalPeriodViewHandler.class);

    protected FiscalPeriodView view;

    /** Service. */
    protected FiscalPeriodService fiscalPeriodService;

    protected FinancialTransactionService financialTransactionService;

    /**
     * Constructor.
     * 
     * @param view managed view
     */
    public FiscalPeriodViewHandler(FiscalPeriodView view) {
        this.view = view;

        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);

        LimaServiceFactory.addServiceListener(FiscalPeriodService.class, this);
    }

    public void init() {
        loadAllFiscalPeriod();
    }

    /**
     * Reload fiscal period list from service and update ui.
     */
    public void loadAllFiscalPeriod() {
        List<FiscalPeriod> periods = fiscalPeriodService.getAllFiscalPeriods();
        view.getFiscalPeriodTableModel().setFiscalPeriods(periods);
    }

    /**
     * Display add period form and perform add.
     */
    public void addFiscalPeriod() {

        FiscalPeriodTableModel model = (FiscalPeriodTableModel) view.getFiscalPeriodTable().getModel();

        AddPeriod addPeriodDialog = new AddPeriod(view);
        addPeriodDialog.setLocationRelativeTo(view);
        addPeriodDialog.setVisible(true);

        FiscalPeriod fiscalPeriod = addPeriodDialog.getPeriod();
        if (fiscalPeriod != null) {             
           
            //check if fiscalperiod have 12 months, ask a confirmation
            int nbMonth = DateUtil.getDifferenceInMonths(fiscalPeriod.getBeginDate(),
                fiscalPeriod.getEndDate());
            int response = JOptionPane.YES_OPTION;
            if (nbMonth != 12) {
                response = JOptionPane.showConfirmDialog(view,
                   _("lima.ui.fiscalperiod.addfiscalperiod.morethan12"),
                   _("lima.ui.fiscalperiod.addfiscalperiodtitle"),
                   JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
            }

            if (response == JOptionPane.YES_OPTION) {
                fiscalPeriod = fiscalPeriodService.createFiscalPeriod(fiscalPeriod);
                model.addFiscalPeriod(fiscalPeriod);
            }
        }
    }

    /**
     * Undisplay period and perform delete.
     */
    public void deleteFiscalPeriod() {

         JXTable fiscalPeriodeTable = view.getFiscalPeriodTable();       
         FiscalPeriodTableModel model = (FiscalPeriodTableModel) view.getFiscalPeriodTable().getModel();      

         //check if the user want to delete the fiscal year
         int response = JOptionPane.showConfirmDialog(view,
                 _("lima.ui.fiscalperiod.delete.confirmation"),
                 _("lima.ui.fiscalperiod.delete.title"), JOptionPane.YES_NO_OPTION,
                 JOptionPane.WARNING_MESSAGE);

         int selectedRow = fiscalPeriodeTable.getSelectedRow();
         FiscalPeriod selectedFiscalPeriod = model.getFiscalPeriodAt(selectedRow);

         if (response == JOptionPane.YES_OPTION) {
             fiscalPeriodService.deleteFiscalPeriod(selectedFiscalPeriod);

             model.deleteFiscalPeriod(selectedFiscalPeriod);
             view.getModel().setBlockEnabled(false);
             view.getModel().setDeleteEnabled(false);
         }
    }

    /**
     * Init add period view.
     * 
     * @param addPeriod view to init
     */
    public void init(AddPeriod addPeriod) {
        addPeriod.getRootPane().setDefaultButton(addPeriod.getOkButton());

        // set begin date picker
        Date endDate = null;
        Calendar calendarBegin = Calendar.getInstance();
        // set begindate to JAN 1 - 0:00.000 of this years
        Date beginDate = calendarBegin.getTime();
        beginDate = DateUtils.truncate(beginDate, Calendar.DATE);
        try {
            FiscalPeriod result = fiscalPeriodService.getLastFiscalPeriod();
            if (result != null){
                beginDate = result.getEndDate();
            }
        }
        catch (LimaException eee){
            if(log.isDebugEnabled()){
                log.debug("Can't get last fiscal period", eee);
            }
        }
        beginDate = DateUtils.addDays(beginDate, 1);
        addPeriod.getBeginDatePicker().setDate(beginDate);
        
        // get end date
        endDate = DateUtils.addYears(beginDate, 1);
        endDate = DateUtils.addDays(endDate, -1);
        addPeriod.getEndDatePicker().setDate(endDate);
    }

    /**
     * Block selected fiscal period,
     * and make new fiscal period and retained earnings
     * after user confirmations
     */
    public void blockFiscalPeriod() {
        JXTable fiscalPeriodeTable = view.getFiscalPeriodTable();        
        FiscalPeriodTableModel model = view.getFiscalPeriodTableModel();
        
        int selectedRow = fiscalPeriodeTable.getSelectedRow();
        FiscalPeriod selectedFiscalPeriod = model.getFiscalPeriodAt(selectedRow);

        //check if the user want to block the fiscal year
        int answerBlock = JOptionPane.showConfirmDialog(view,
                _("lima.ui.fiscalperiod.block.confirmation"),
                _("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                JOptionPane.WARNING_MESSAGE);
        
        if (answerBlock == JOptionPane.YES_OPTION) {
            boolean retainedEarnings = false;
            EntryBook entryBook = null;

            //check if the user wants to create a new fiscal year
            int answerCreate = JOptionPane.showConfirmDialog(view,
                                _("lima.ui.fiscalperiod.block.newyear"),
                                _("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE);

            // create a new fiscal year
            if (answerCreate == JOptionPane.YES_OPTION) {
                addFiscalPeriod();

                //report if they are at least one transaction to be report
                if (fiscalPeriodService.isRetainedEarnings(selectedFiscalPeriod)) {

                    //check if the user wants to report datas
                    int answerRetainedEarnings = JOptionPane.showConfirmDialog(view,
                            _("lima.ui.fiscalperiod.block.addretainedearnings"),
                            _("lima.ui.fiscalperiod.block.title"), JOptionPane.YES_NO_OPTION,
                            JOptionPane.QUESTION_MESSAGE);

                    if (answerRetainedEarnings == JOptionPane.YES_OPTION) {
                        retainedEarnings = true;

                        //Sets EntryBook
                        EntryBook newEntryBook = new EntryBookImpl();
                        RetainedEarningsEntryBookForm entryBookForm =
                                new RetainedEarningsEntryBookForm(view);
                        entryBookForm.setEntryBook(newEntryBook);

                        // jaxx constructor don't call super() ?
                        entryBookForm.setLocationRelativeTo(view);
                        entryBookForm.setVisible(true);
                        entryBook = entryBookForm.getEntryBook();
                    }
                }
            }

            FiscalPeriod fiscalPeriodBlocked = fiscalPeriodService.retainedEarningsAndBlockFiscalPeriod(selectedFiscalPeriod, entryBook, retainedEarnings);
            model.updateFiscalPeriod(fiscalPeriodBlocked);

            view.getModel().setBlockEnabled(false);
            view.getModel().setDeleteEnabled(false);
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        //refresh on import datas
        if (methodeName.contains("importAll")) {
            loadAllFiscalPeriod();
        }
    }
    
    /**
     * Manage the differents buttons for fiscal period
     * */
    protected void onSelectionChanged(ListSelectionEvent listSelectionEvent){
        if(!listSelectionEvent.getValueIsAdjusting()) {
            DefaultListSelectionModel listSelectionModel = (DefaultListSelectionModel)listSelectionEvent.getSource();

            int selectedRow = listSelectionEvent.getFirstIndex();
            if (!listSelectionModel.isSelectedIndex(selectedRow)) {
                selectedRow = listSelectionEvent.getLastIndex();
            }

            FiscalPeriodTableModel fiscalPeriodTableModel = view.getFiscalPeriodTableModel();
            if (fiscalPeriodTableModel.getRowCount() != selectedRow) {
                FiscalPeriod fiscalPeriodAt = fiscalPeriodTableModel.getFiscalPeriodAt(selectedRow);

                List<FinancialTransaction> financialTransactionList = financialTransactionService.getAllFinancialTransactions(fiscalPeriodAt);

                boolean enableBlock = false;
                boolean enableDelete = false;

                if (log.isDebugEnabled()) {
                    log.debug("reloadEnablingButton");
                }

              if (selectedRow != -1){
                  if (log.isDebugEnabled()) {
                      log.debug("selectedFiscalPeriod != null");
                  }

                  //no action possible for fiscal period closed
                  if (!fiscalPeriodAt.getLocked()){
                      if (log.isDebugEnabled()) {
                          log.debug("selectedFiscalPeriod not locked");
                      }
                      enableBlock = true;

                      //only an empty and open fiscal period may be deleted
                      enableDelete = financialTransactionList.size() <= 0;
                   }
                }
                view.getModel().setBlockEnabled(enableBlock);
                view.getModel().setDeleteEnabled(enableDelete);
            }
        }
    }  
        
}
