/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialTransactionTableModel.java 3698 2013-08-02 15:06:11Z Bavencoff $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/financialtransaction/FinancialTransactionTableModel.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.financialtransaction;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.utils.EntryComparator;
import org.chorem.lima.business.utils.FinancialTransactionComparator;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FinancialTransactionImpl;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.common.Column;
import org.chorem.lima.ui.common.TableModelWithGroup;

import javax.swing.table.AbstractTableModel;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Basic transaction table model.
 *
 * @author ore
 * @author chatellier
 * @version $Revision: 3698 $
 */
public class FinancialTransactionTableModel extends AbstractTableModel  implements TableModelWithGroup {

    /** serialVersionUID. */
    private static final long serialVersionUID = -7495388454688562991L;

    protected static final Log log = LogFactory.getLog(FinancialTransactionTableModel.class);

    /** Service (just to update setValueAt(). */
    protected FinancialTransactionService financialTransactionService;

    protected List<Column> columns;

    protected List<Entry> entries;

    protected List<FinancialTransaction> transactions;

    public FinancialTransactionTableModel() {
        columns = new ArrayList<Column>();
        initColumns();

        financialTransactionService =
                LimaServiceFactory.getService(FinancialTransactionService.class);
        entries = new ArrayList<Entry>();
        transactions = new ArrayList<FinancialTransaction>();
    }

    protected void initColumns() {
        columns.add(new DateColumn(this));
        columns.add(new VoucherColumn(this));
        columns.add(new AccountColumn(this));
        columns.add(new DescriptionColumn(this));
        columns.add(new DebitColumn(this));
        columns.add(new CreditColumn(this));
    }

    public Column getColumn(int column) {
        return columns.get(column);
    }

    public void setTransactions(List<FinancialTransaction> transactions) {
        this.transactions.clear();
        this.transactions.addAll(transactions);
        sortEntries();
    }

    public void sortEntries() {
        Collections.sort(transactions, new FinancialTransactionComparator());
        entries.clear();
        for (FinancialTransaction transaction : transactions) {
            if (transaction.sizeEntry() > 0) {
                List<Entry> entries = new ArrayList<Entry>(transaction.getEntry());
                Collections.sort(entries, new EntryComparator());
                this.entries.addAll(entries);
            } else {
                Entry entry = new EntryImpl();
                entry.setFinancialTransaction(transaction);
                entry = financialTransactionService.createEntry(entry);
                transaction.addEntry(entry);
                financialTransactionService.updateFinancialTransaction(transaction);
                this.entries.add(entry);
            }
        }
        fireTableDataChanged();
    }

    @Override
    public int getColumnCount() {
        return columns.size();
    }

    @Override
    public int getRowCount() {
        return entries.size();
    }

    @Override
    public Class<?> getColumnClass(int column) {
        Class result = null;
        if (column >= 0 && column < columns.size()) {
            result = columns.get(column).getColumnClass();
        }
        return result;
    }

    @Override
    public String getColumnName(int column) {
        String result = "n/a";
        if (column >= 0 && column < columns.size()) {
            result = columns.get(column).getColumnName();
        }
        return  result;
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        if (column >= 0 && column < columns.size()) {
            result = columns.get(column).getValueAt(row);
        }
        return  result;
    }

    /**
     * To set cells editable or not
     * different condition for entry or financial transaction
     */
    @Override
    public boolean isCellEditable(int row, int column) {
        boolean result = false;
        if (column >= 0 && column < columns.size()) {
            result = columns.get(column).isCellEditable(row);
        }
        return  result;
    }

    /** to modifiy financialtransaction or entry */
    @Override
    public void setValueAt(Object value, int row, int column) {
        boolean update = false;
        if (column >= 0 && column < columns.size()) {
            update = columns.get(column).setValueAt(value, row);
        }
        // some modification must update all other
        // first row modification update following rows
        if (update) {
            fireTableCellUpdated(row, column);
        }
    }

    public Entry getEntryAt(int row) {
        return entries.get(row);
    }

    public FinancialTransaction getTransactionAt(int row) {
        Entry entry = entries.get(row);
        return entry.getFinancialTransaction();
    }

    public int indexOf(Entry entry) {
        return entries.indexOf(entry);
    }

    public int indexOf(FinancialTransaction transaction) {
        int index = 0;
        Collection<Entry> entriesTransaction = transaction.getEntry();
        for (Entry entry : entries) {
            if (!entriesTransaction.contains(entry)) {
                index++;
            } else {
                break;
            }
        }
        return index;
    }

    /**
     * Insert new entry.
     * 
     * @param entry entry to insert
     */
    public Entry addEntry(Entry entry) {
        Entry newEntry = null;
        FinancialTransaction transaction = entry.getFinancialTransaction();
        if (transactions.contains(transaction)) {
            newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(transaction);
            newEntry.setVoucher(entry.getVoucher());
            newEntry.setAccount(entry.getAccount());
            newEntry.setDescription(entry.getDescription());
            newEntry.setAmount(entry.getAmount());
            newEntry.setDebit(entry.getDebit());
            newEntry = financialTransactionService.createEntry(newEntry);

            transaction.addEntry(newEntry);
            financialTransactionService.updateFinancialTransaction(transaction);
            int row = indexOf(transaction) + transaction.sizeEntry() - 1;
            entries.add(row, newEntry);
            fireTableRowsInserted(row, row);
        }
        return newEntry;
    }

    /**
     * Delete selected row in table (could be transaction or entry).
     * <p/>
     * Called by model.
     *
     * @param row
     */
    public void removeTransaction(int row) {
        FinancialTransaction transaction = getTransactionAt(row);
        int firstRow = indexOf(transaction);
        int lastRow = firstRow + transaction.sizeEntry() - 1;
        financialTransactionService.removeFinancialTransaction(transaction);
        entries.removeAll(transaction.getEntry());
        transactions.remove(transaction);
        fireTableRowsDeleted(firstRow, lastRow);
    }

    public void removeEntry(int row) {
        Entry entry = getEntryAt(row);
        FinancialTransaction transaction = entry.getFinancialTransaction();
        if (transaction.sizeEntry() > 1) {
            financialTransactionService.removeEntry(entry);
            transaction.removeEntry(entry);
            entries.remove(entry);
        } else {
            financialTransactionService.removeFinancialTransaction(transaction);
            entries.removeAll(transaction.getEntry());
            transactions.remove(transaction);
        }
        fireTableRowsDeleted(row, row);
    }

    public FinancialTransaction addTransaction(FinancialTransaction transaction) {
        FinancialTransaction newTransaction = new FinancialTransactionImpl();
        newTransaction.setEntryBook(transaction.getEntryBook());
        newTransaction.setTransactionDate(transaction.getTransactionDate());
        newTransaction = financialTransactionService.createFinancialTransaction(newTransaction);

        if (transaction.getEntry() == null || transaction.getEntry().isEmpty()) {
            Entry newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(newTransaction);
            newEntry = financialTransactionService.createEntry(newEntry);
            newTransaction.addEntry(newEntry);

            newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(newTransaction);
            newEntry = financialTransactionService.createEntry(newEntry);
            newTransaction.addEntry(newEntry);
        } else {
            for (Entry entry : transaction.getEntry()) {
                Entry newEntry = new EntryImpl();
                newEntry.setFinancialTransaction(newTransaction);
                newEntry.setVoucher(entry.getVoucher());
                newEntry.setAccount(entry.getAccount());
                newEntry.setDescription(entry.getDescription());
                newEntry.setAmount(entry.getAmount());
                newEntry.setDebit(entry.getDebit());
                newEntry = financialTransactionService.createEntry(newEntry);
                newTransaction.addEntry(newEntry);
            }
            financialTransactionService.updateFinancialTransaction(newTransaction);
        }

        int indexFirstEntry = entries.size();
        transactions.add(newTransaction);
        entries.addAll(newTransaction.getEntry());
        int indexLastEntry = entries.size() - 1;
        fireTableRowsInserted(indexFirstEntry, indexLastEntry);
        return newTransaction;
    }

    public BigDecimal getBalanceTransactionInRow(int row) {
        FinancialTransaction transaction = getTransactionAt(row);
        BigDecimal debit = transaction.getAmountDebit();
        BigDecimal credit = transaction.getAmountCredit();
        BigDecimal balance = debit.subtract(credit);
        return balance;
    }

    public int size() {
        int result = 0;
        if (entries != null) {
            result = entries.size();
        }
        return result;
    }


    @Override
    public int indexGroupAt(int row) {
        FinancialTransaction transaction = getTransactionAt(row);
        int index = transactions.indexOf(transaction);
        return index;

    }

    public void updateEntry(Entry entry) {
        if (log.isDebugEnabled()) {
            log.debug("Update Entry");
        }
        financialTransactionService.updateEntry(entry);
    }

    public void updateTransaction(FinancialTransaction transaction) {
        if (log.isDebugEnabled()) {
            log.debug("Update transaction");
        }
        financialTransactionService.updateFinancialTransaction(transaction);
    }

    public void fireTransaction(FinancialTransaction transaction) {
        int firstRow = indexOf(transaction);
        int lastRow = firstRow + transaction.sizeEntry() - 1;
        fireTableRowsUpdated(firstRow, lastRow);
    }
}
