/*
 * #%L
 * Lima Swing
 * 
 * $Id: EntryBookViewHandler.java 3426 2012-05-16 13:02:27Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/entrybook/EntryBookViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.entrybook;

import static org.nuiton.i18n.I18n._;

import java.util.List;

import javax.swing.JOptionPane;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.EntryBookService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.EntryBookImpl;
import org.chorem.lima.enums.EntryBooksChartEnum;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.importexport.ImportExport;
import org.jdesktop.swingx.JXTable;

/**
 * Handler for entry book view.
 *
 * @author chatellier
 * @version $Revision: 3426 $
 *          <p/>
 *          Last update : $Date: 2012-05-16 15:02:27 +0200 (Wed, 16 May 2012) $
 *          By : $Author: echatellier $
 */
public class EntryBookViewHandler implements ServiceListener {

    private static final Log log = LogFactory.getLog(EntryBookViewHandler.class);

    protected EntryBookView view;

    protected EntryBookService entryBookService;

    public EntryBookViewHandler(EntryBookView view) {
        this.view = view;
        entryBookService = LimaServiceFactory.getService(EntryBookService.class);
        LimaServiceFactory.addServiceListener(ImportService.class, this);
    }

    public void init() {
        loadAllEntryBooks();
    }

    /**
     * Load all available entry book into table.
     */
    protected void loadAllEntryBooks() {

        List<EntryBook> entryBooks = entryBookService.getAllEntryBooks();
        if (log.isDebugEnabled()) {
            log.debug(String.format("Loaded %d entry books from service", entryBooks.size()));
        }
        
        EntryBookTableModel model = new EntryBookTableModel(entryBooks);
        EntryBookTable table = view.getEntryBooksTable();
        table.setModel(model);
    }

    /**
     * Add new entry book (display blocking dialog and perform add).
     */
    public void addEntryBook() {

        EntryBook newEntryBook = new EntryBookImpl();
        EntryBookForm entryBookForm = new EntryBookForm(view);
        entryBookForm.setEntryBook(newEntryBook);
        entryBookForm.setLocationRelativeTo(view);
        entryBookForm.setVisible(true);

        newEntryBook = entryBookForm.getEntryBook();
        // null == cancel action
        if (newEntryBook != null) {

            // service call
            newEntryBook = entryBookService.createEntryBook(newEntryBook);
            
            // ui refresh
            JXTable entryBookTable = view.getEntryBooksTable();
            EntryBookTableModel entryBookTableModel = (EntryBookTableModel) entryBookTable.getModel();
            entryBookTableModel.addEntryBook(newEntryBook);
        }
    }

    /**
     * Update existing entry book.
     */
    public void updateEntryBook() {

        JXTable entryBookTable = view.getEntryBooksTable();
        EntryBookTableModel entryBookTableModel = (EntryBookTableModel) entryBookTable.getModel();
        int selectedRow = entryBookTable.getSelectedRow();

        EntryBook selectedEntryBook = entryBookTableModel.getEntryBookAtRow(selectedRow);
        EntryBookForm entryBookForm = new EntryBookForm(view);
        entryBookForm.setAddState(false);
        entryBookForm.setEntryBook(selectedEntryBook);
        entryBookForm.setLocationRelativeTo(view);
        entryBookForm.setVisible(true);

        // null == cancel action
        selectedEntryBook = entryBookForm.getEntryBook();
        if (selectedEntryBook != null) {
            // service call
            selectedEntryBook = entryBookService.updateEntryBook(selectedEntryBook);

            // ui refresh
            entryBookTableModel.updateEntryBook(selectedEntryBook);
        }
    }

    /**
     * Delete selected entry book.
     */
    public void deleteEntryBook() {
        JXTable entryBookTable = view.getEntryBooksTable();
        int selectedRow = entryBookTable.getSelectedRow();
        EntryBookTableModel entryBookTableModel = (EntryBookTableModel) entryBookTable.getModel();

        EntryBook selectedEntryBook = entryBookTableModel.getEntryBookAtRow(selectedRow);

        int response = JOptionPane.showConfirmDialog(view,
            _("lima.ui.entrybook.deletemessage", selectedEntryBook.getLabel()),
            _("lima.ui.entrybook.deletetitle"), JOptionPane.YES_NO_OPTION,
            JOptionPane.QUESTION_MESSAGE);

        if (response == JOptionPane.YES_OPTION) {

            entryBookService.removeEntryBook(selectedEntryBook);
            entryBookTableModel.removeEntryBook(selectedEntryBook);
        }
        
    }

    /**
     * Import entry book.
     * 
     * UI will be refreshed by {@link #notifyMethod(String, String)} called
     * by service listener.
     */
    public void importEntryBooks() {

        EntryBookImportForm form = new EntryBookImportForm(view);
        form.setLocationRelativeTo(view);
        form.setVisible(true);

        Object value = form.getButtonGroup().getSelectedValue();
        // if action confirmed
        if (value != null) {
            EntryBooksChartEnum defaultEntryBooksEnum = (EntryBooksChartEnum) value;
            ImportExport importExport = new ImportExport(view);
            importExport.importExport(ImportExportEnum.CSV_ENTRYBOOKS_IMPORT,
                                      defaultEntryBooksEnum.getFilePath(), true);
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {

        //refresh on import datas
        if (methodeName.contains("importEntryBooks")
            || methodeName.contains("importAll")
            || methodeName.contains("importAs")) {
            loadAllEntryBooks();
        }
    }
}
