/*
 * #%L
 * Lima :: Swing
 * $Id: IntervalPanel.java 3615 2012-08-20 12:52:54Z mallon $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/common/IntervalPanel.java $
 * %%
 * Copyright (C) 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package org.chorem.lima.ui.common;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.apache.commons.lang3.time.DateUtils;
import org.chorem.lima.entity.FinancialPeriod;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.enums.ComboBoxDatesEnum;
import org.jdesktop.swingx.JXDatePicker;

/**
 * Panel permettant de selectionner un interval de date suivant plusieurs type
 * possible:
 * <ul>
 * <li>date d'un exercice
 * <li>date d'une perdiode comptable
 * <li>deux dates
 * </ul>
 * 
 * Il supporte les listeners, donc on peut être notifié d'un changement
 * d'interval.
 * 
 * @author echatellier
 */
public class IntervalPanel extends JPanel implements ActionListener, ItemListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 5973427750255668999L;

    protected List<IntervalListener> listeners;

    protected JComboBox typeComboBox;

    protected CardLayout typeLayout;

    protected JPanel specificTypePanel;

    protected FiscalPeriodComboBoxModel fiscalPeriodComoboBoxModel;

    protected FinancialPeriodComboBoxModel financialPeriodComoboBoxModel;

    protected JXDatePicker beginDatePicker;

    protected JXDatePicker endDatePicker;

    public IntervalPanel() {
        super(new BorderLayout());
        listeners = new ArrayList<IntervalListener>();
        initUI();
    }

    public void addIntervalListener(IntervalListener listener) {
        listeners.add(listener);
    }

    public void removeIntervalListener(IntervalListener listener) {
        listeners.remove(listener);
    }

    private void initUI() {

        // type combo box
        typeComboBox = new JComboBox(ComboBoxDatesEnum.descriptions());
        add(typeComboBox, BorderLayout.WEST);
        typeComboBox.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                ComboBoxDatesEnum type = ComboBoxDatesEnum.valueOfDescription((String) typeComboBox.getSelectedItem());
                // show corresponding component in layout
                typeLayout.show(specificTypePanel, type.name());
            }
        });

        typeLayout = new CardLayout();
        specificTypePanel = new JPanel(typeLayout);

        fiscalPeriodComoboBoxModel = new FiscalPeriodComboBoxModel();
        JComboBox fiscalPeriodComboBox = new JComboBox(fiscalPeriodComoboBoxModel);
        fiscalPeriodComboBox.setRenderer(new FiscalPeriodListRenderer());
        fiscalPeriodComboBox.addItemListener(this);
        specificTypePanel.add(fiscalPeriodComboBox, ComboBoxDatesEnum.FISCAL_PERIOD.name());

        financialPeriodComoboBoxModel = new FinancialPeriodComboBoxModel();
        JComboBox financialPeriodComboBox = new JComboBox(financialPeriodComoboBoxModel);
        financialPeriodComboBox.setRenderer(new FinancialPeriodListRenderer());
        financialPeriodComboBox.addItemListener(this);
        specificTypePanel.add(financialPeriodComboBox, ComboBoxDatesEnum.FINANCIAL_PERIOD.name());

        JPanel periodPanel = new JPanel(new GridBagLayout());
        JLabel beginLabel = new JLabel(_("lima.common.begindate"));
        periodPanel.add(beginLabel, new GridBagConstraints(0, 0, 1, 1, 0, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(1, 1, 1, 1), 0, 0));

        beginDatePicker = new JXDatePicker();
        beginDatePicker.addActionListener(this);
        periodPanel.add(beginDatePicker, new GridBagConstraints(1, 0, 1, 1, 0, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(1, 1, 1, 1), 0, 0));

        JLabel endLabel = new JLabel(_("lima.common.enddate"));
        periodPanel.add(endLabel, new GridBagConstraints(2, 0, 1, 1, 0, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(1, 1, 1, 1), 0, 0));

        endDatePicker = new JXDatePicker();
        endDatePicker.addActionListener(this);
        periodPanel.add(endDatePicker, new GridBagConstraints(3, 0, 1, 1, 0, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(1, 1, 1, 1), 0, 0));
        specificTypePanel.add(periodPanel, ComboBoxDatesEnum.PERIOD.name());

        // get begin date
        Date beginDate = DateUtils.truncate(new Date(), Calendar.YEAR);
        beginDatePicker.setDate(beginDate);
        endDatePicker.setDate(new Date());
        
        add(specificTypePanel, BorderLayout.CENTER);
    }

    /**
     * Init component combo models.
     * 
     * @param fiscalPeriods fiscal periods
     * @param financialPeriods financial periods
     */
    public void init(List<FiscalPeriod> fiscalPeriods, List<FinancialPeriod> financialPeriods) {
        fiscalPeriodComoboBoxModel.setObjects(fiscalPeriods);
        financialPeriodComoboBoxModel.setObjects(financialPeriods);
    }

    /**
     * Return selected begin date.
     * 
     * @return begin date, or {@code null} with no valid selection.
     */
    public Date getBeginDate() {
        Date beginDate = null;

        ComboBoxDatesEnum type = ComboBoxDatesEnum.valueOfDescription((String) typeComboBox.getSelectedItem());
        switch (type) {
            case PERIOD:
                beginDate = beginDatePicker.getDate();
                break;
            case FINANCIAL_PERIOD: {
                FinancialPeriod period = (FinancialPeriod)financialPeriodComoboBoxModel.getSelectedItem();
                if (period != null) {
                    beginDate = period.getBeginDate();
                }
                break;
            }
            case FISCAL_PERIOD: {
                FiscalPeriod period = (FiscalPeriod)fiscalPeriodComoboBoxModel.getSelectedItem();
                if (period != null) {
                    beginDate = period.getBeginDate();
                }
                break;
            }
        }

        return beginDate;
    }

    /**
     * Return selected end date.
     * 
     * @return end date, or {@code null} with no valid selection.
     */
    public Date getEndDate() {
        Date endDate = null;

        ComboBoxDatesEnum type = ComboBoxDatesEnum.valueOfDescription((String) typeComboBox.getSelectedItem());
        switch (type) {
            case PERIOD:
                endDate = endDatePicker.getDate();
                break;
            case FINANCIAL_PERIOD: {
                FinancialPeriod period = (FinancialPeriod)financialPeriodComoboBoxModel.getSelectedItem();
                if (period != null) {
                    endDate = period.getEndDate();
                }
                break;
            }
            case FISCAL_PERIOD: {
                FiscalPeriod period = (FiscalPeriod)fiscalPeriodComoboBoxModel.getSelectedItem();
                if (period != null) {
                    endDate = period.getEndDate();
                }
                break;
            }
        }

        return endDate;
    }

    @Override
    public void itemStateChanged(ItemEvent e) {
        if (e.getStateChange() == ItemEvent.SELECTED) {
            fireEvent();
        }
    }

    protected void fireEvent() {
        for (IntervalListener l : listeners) {
            l.intervalChanged(null);
        }
    }

    @Override
    public void actionPerformed(ActionEvent e) {
        fireEvent();
    }
}
