/*
 * #%L
 * Lima Swing
 * 
 * $Id: MainViewHandler.java 3649 2012-09-18 17:14:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/ui/MainViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.editor.config.ConfigUIHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaConfig;
import org.chorem.lima.LimaContext;
import org.chorem.lima.business.api.HttpServerService;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.account.AccountView;
import org.chorem.lima.ui.accountsreports.AccountsReportsView;
import org.chorem.lima.ui.balance.BalanceReportsView;
import org.chorem.lima.ui.entrybook.EntryBookView;
import org.chorem.lima.ui.entrybooksreports.EntryBooksReportsView;
import org.chorem.lima.ui.financialperiod.FinancialPeriodView;
import org.chorem.lima.ui.financialstatementchart.FinancialStatementChartView;
import org.chorem.lima.ui.financialstatementreport.FinancialStatementReportView;
import org.chorem.lima.ui.financialtransaction.FinancialTransactionView;
import org.chorem.lima.ui.financialtransactionsearch.FinancialTransactionSearchView;
import org.chorem.lima.ui.financialtransactionunbalanced.FinancialTransactionUnbalancedView;
import org.chorem.lima.ui.fiscalperiod.FiscalPeriodView;
import org.chorem.lima.ui.home.HomeView;
import org.chorem.lima.ui.identity.IdentityForm;
import org.chorem.lima.ui.importexport.ImportExport;
import org.chorem.lima.ui.ledger.LedgerView;
import org.chorem.lima.ui.lettering.LetteringView;
import org.chorem.lima.ui.vatchart.VatChartView;
import org.chorem.lima.ui.vatreports.VatReportView;
import org.nuiton.util.DesktopUtil;
import org.nuiton.widget.SwingSession;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.UIManager;
import javax.swing.border.LineBorder;
import javax.swing.plaf.BorderUIResource;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.net.URL;
import java.util.Locale;

import static org.nuiton.i18n.I18n._;

/**
 * Handler for main view.
 * <p/>
 * Date: 8 nov. 2009
 * Time: 09:59:54
 *
 * @author chemit
 * @version $Revision: 3649 $
 *          <p/>
 *          Last update : $Date: 2012-09-18 19:14:25 +0200 (Tue, 18 Sep 2012) $
 *          By : $Author: tchemit $
 */
public class MainViewHandler {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(MainViewHandler.class);

    protected SwingSession swingSession;

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param fullscreen  flag pour indiquer si on doit ouvrir l'ui en model console (pleine ecran).
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public MainView initUI(LimaContext rootContext, boolean fullscreen) {
        // show main ui
        MainView ui = new MainView(rootContext);

        swingSession = new SwingSession(getLimaStateFile(), false);

        LimaContext.MAIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);

        // set fullscreen propery on main ui
        ui.getGraphicsConfiguration().getDevice().setFullScreenWindow(fullscreen ? ui : null);

        return ui;
    }

    //Get xml lima state file
    protected File getLimaStateFile() {

        LimaConfig limaConfig = LimaConfig.getInstance();

        File limaStateFile = limaConfig.getLimaStateFile();

        return limaStateFile;
    }

    public void changeLanguage(MainView mainUI, Locale newLocale) {
        LimaConfig config = mainUI.getConfig();

        // sauvegarde de la nouvelle locale
        config.setLocale(newLocale);

        // rechargement i18n
        LimaContext.get().initI18n(config);

        // on recharge l'ui
        reloadUI(LimaContext.get(), config.isFullScreen());
    }

    /**
     * Ferme l'application.
     *
     * @param ui l'ui principale de l'application
     */
    public void close(MainView ui) {
        /*Save Lima state*/
        swingSession.save();
        if (log.isInfoEnabled()) {
            log.info("Lima quitting...");
        }
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }
        try {
            LimaContext.get().close();

            Runtime.getRuntime().halt(0);

        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("error while closing " + ex.getMessage(), ex);
            }
            Runtime.getRuntime().halt(1);
        }
    }

    /**
     * Méthode pour changer de mode d'affichage.
     * <p/>
     * Si <code>fullscreen</code> est à <code>true</code> alors on passe en
     * mode console (c'est à dire en mode plein écran exclusif), sinon on
     * passe en mode fenetré normal.
     *
     * @param ui         l'ui principale de l'application
     * @param fullscreen le nouvel état requis.
     */
    public void changeScreen(MainView ui, boolean fullscreen) {
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }

        ui.getConfig().setFullscreen(fullscreen);

        reloadUI(LimaContext.get(), fullscreen);
    }

    public void showConfig(JAXXContext context) {
        MainView ui = getUI(context);
        final LimaConfig config = ui.getConfig();

        ConfigUIHelper helper = new ConfigUIHelper(config);

        helper.registerCallBack(LimaConfig.Option.COLOR_SELECTION_FOCUS.key, _("lima.config.colorselectionfocus"), new ImageIcon(), new Runnable() {
            @Override
            public void run() {
                UIManager.put("Table.focusCellHighlightBorder", new BorderUIResource(new LineBorder(config.getColorSelectionFocus(), 2)));
            }
        });

        helper.addCategory(_("lima.config.category.directories"), _("lima.config.category.directories.description"));
        helper.addOption(LimaConfig.Option.CONFIG_FILE);

        helper.addCategory(_("lima.config.category.other"), _("lima.config.category.other.description"));
        helper.addOption(LimaConfig.Option.FULL_SCREEN);
        helper.addOption(LimaConfig.Option.LOCALE);
        helper.addOption(LimaConfig.Option.DECIMAL_SEPARATOR);
        helper.addOption(LimaConfig.Option.SCALE);
        helper.addOption(LimaConfig.Option.THOUSAND_SEPARATOR);
        helper.addOption(LimaConfig.Option.CURRENCY);
        /*Pas de 'callBack' sur le changement de comportement lors de l'édition d'une cellule,
        * car les deux éditeurs concernés rappellent la config dans un listener ('focusGained')*/
        helper.addOption(LimaConfig.Option.COMPORTMENT_EDITING_CELL);
        helper.addOption(LimaConfig.Option.COLOR_SELECTION_FOCUS);
        helper.setOptionCallBack(LimaConfig.Option.COLOR_SELECTION_FOCUS.key);

        helper.buildUI(context, _("lima.config.category.directories"));
        helper.displayUI(ui, false);
    }

    public void gotoSite(JAXXContext context) {

        LimaConfig config = getUI(context).getConfig();

        URL siteURL = config.getOptionAsURL("application.site.url");
        try {
            DesktopUtil.browse(siteURL.toURI());
        } catch (Exception e) {
            throw new RuntimeException("Can't open lima website at " + siteURL, e);
        }
    }

    public void showAbout(JAXXContext context) {
        MainView ui = getUI(context);
        AboutPanel about = new AboutPanel();
        about.setTitle(_("lima.title.about"));
        about.setAboutText(_("lima.title.about.description"));
        about.setBottomText(ui.getConfig().getCopyrightText());
        about.setIconPath("/icons/lima.png");
        about.setLicenseFile("META-INF/lima-LICENSE.txt");
        about.setThirdpartyFile("META-INF/lima-THIRD-PARTY.txt");

        JScrollPane component = new JScrollPane();
        JEditorPane translateArea = new JEditorPane();
        translateArea.setContentType("text/html");
        translateArea.setEditable(false);
        if (translateArea.getFont() != null) {
            translateArea.setFont(translateArea.getFont().deriveFont((float) 11));
        }
        translateArea.setBorder(null);
        String text = _("lima.about.translate.content");
        translateArea.setText(text);
        component.getViewport().add(translateArea);
        about.getTabs().add(_("lima.title.about.translate"), component);
        about.init();
        about.showInDialog(ui, true);
    }

    /**
     * Permet de recharger l'ui principale et de changer de le mode d'affichage.
     *
     * @param rootContext le contexte applicatif
     * @param fullscreen  le type de fenetre à reouvrir
     */
    protected void reloadUI(LimaContext rootContext, boolean fullscreen) {

        // must remove all properties listener on config
        LimaContext.CONFIG_DEF.getContextValue(rootContext).removeJaxxPropertyChangeListener();

        // scan main ui
        MainView ui = getUI(rootContext);

        if (ui != null) {
            LimaContext.MAIN_UI_ENTRY_DEF.removeContextValue(rootContext);

            ui.dispose();

            ui.setVisible(false);
        }

        ui = initUI(rootContext, fullscreen);

        // show ui
        ui.setVisible(true);
        showHomeView(rootContext);
    }

    /**
     * Test if there is some modification on screen,
     *
     * @param rootContext the context
     * @return <code>true</code> if no more modification is detected
     * @throws IllegalArgumentException if rootContext is null
     */
    protected boolean ensureModification(JAXXContext rootContext) throws IllegalArgumentException {
        if (rootContext == null) {
            throw new IllegalArgumentException("rootContext can not be null");
        }
        MainView ui = getUI(rootContext);
        if (ui == null) {
            // no ui, so no modification
            return true;
        }
        // check ui is not modified
        return true;
    }

    protected MainView getUI(JAXXContext context) {
        if (context instanceof MainView) {
            return (MainView) context;
        }
        MainView ui = LimaContext.MAIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    public void showHomeView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        HomeView homeView = new HomeView(mainView);
        showTab(mainView, _("lima.tab.home"), homeView, false);
    }

    public void showIdentity(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        IdentityForm identityForm = new IdentityForm();
        identityForm.setLocationRelativeTo(mainView);
        identityForm.setVisible(true);
    }

    /**
     * Show account tree table view to create or modify accounts
     *
     * @param rootContext the root application context
     */
    public void showAccountView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        AccountView accountView = new AccountView(mainView);
        showTab(mainView, _("lima.charts.account"), accountView);
        swingSession.add(accountView);
    }

    /**
     * Show account tree table view to create or modify accounts
     *
     * @param rootContext the root application context
     */
    public void showEntryBookView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        EntryBookView entryBookView = new EntryBookView(mainView);
        showTab(mainView, _("lima.common.entrybooks"), entryBookView);
        swingSession.add(entryBookView);
    }

    /**
     * Show fiscal period view to create or block a period
     *
     * @param rootContext the root application context
     */
    public void showFiscalPeriodView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FiscalPeriodView fiscalPeriodView = new FiscalPeriodView(mainView);
        showTab(mainView, _("lima.charts.fiscalyear"), fiscalPeriodView);
        swingSession.add(fiscalPeriodView);
    }

    /**
     * Show financial period view to create or block a period
     *
     * @param rootContext the root application context
     */
    public void showFinancialPeriodView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialPeriodView financialPeriodView =
                new FinancialPeriodView(mainView);
        showTab(mainView, _("lima.charts.financialperiod"), financialPeriodView);
        swingSession.add(financialPeriodView);
    }

    /**
     * Show financial statement view to generate balance sheet and income statement
     *
     * @param rootContext the root application context
     */
    public void showFinancialStatementView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialStatementChartView financialStatementView =
                new FinancialStatementChartView(mainView);
        showTab(mainView, _("lima.charts.financialstatement"), financialStatementView);
        swingSession.add(financialStatementView);
    }

    /**
     * Show vat chart to generate vat document
     *
     * @param rootContext the root application context
     */
    public void showVatChartView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        VatChartView vatChartView = new VatChartView(mainView);
        showTab(mainView, _("lima.charts.vatstatement"), vatChartView);
        swingSession.add(vatChartView);
    }

    /**
     * Show account table report to view an account on a period
     *
     * @param rootContext the root application context
     */
    public void showAccountReports(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        AccountsReportsView accountsReportsView =
                new AccountsReportsView(mainView);
        showTab(mainView, _("lima.reports.accounts"), accountsReportsView);
        swingSession.add(accountsReportsView);
    }

    /**
     * Show entry book table report to view an account on a period
     *
     * @param rootContext the root application context
     */
    public void showEntryBookReports(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        EntryBooksReportsView entryBooksReportsView =
                new EntryBooksReportsView(mainView);
        showTab(mainView, _("lima.reports.entrybooks"), entryBooksReportsView);
        swingSession.add(entryBooksReportsView);
    }


    /**
     * Show financial transactions view to create entries
     *
     * @param rootContext the root application context
     */
    public void showTransactionView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionView transactionView =
                new FinancialTransactionView(mainView);
        showTab(mainView, _("lima.entries.writetransaction"),
                transactionView);
        swingSession.add(transactionView);
    }

    public void showTransactionUnbalancedView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionUnbalancedView searchResultView =
                new FinancialTransactionUnbalancedView(mainView);
        showTab(mainView, _("lima.entries.searchunbalancedtransaction"),
                searchResultView);
        swingSession.add(searchResultView);
    }

    public void showTransactionSearchView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionSearchView searchView =
                new FinancialTransactionSearchView(mainView);
        showTab(mainView, _("lima.entries.searchtransaction"), searchView);
        swingSession.add(searchView);
    }

    public void showLetteringView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        LetteringView letteringView = new LetteringView(mainView);
        showTab(mainView, _("lima.entries.lettering"), letteringView);
        swingSession.add(letteringView);
    }

    public void showBalanceView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        BalanceReportsView balanceView = new BalanceReportsView(mainView);
        showTab(mainView, _("lima.reports.balance"), balanceView);
        swingSession.add(balanceView);
    }

    public void showLedgerView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        LedgerView ledgerView = new LedgerView(mainView);
        showTab(mainView, _("lima.reports.ledger"), ledgerView);
        swingSession.add(ledgerView);
    }

    public void showFinancialStatementReportsView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialStatementReportView financialStatementReportView =
                new FinancialStatementReportView(mainView);
        showTab(mainView, _("lima.reports.financialstatement"),
                financialStatementReportView);
        swingSession.add(financialStatementReportView);
    }

    public void showVatReportView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        VatReportView vatReportView = new VatReportView(mainView);
        showTab(mainView, _("lima.reports.vat"), vatReportView);
        swingSession.add(vatReportView);
    }

    public void showImportExportView(JAXXContext rootContext,
                                     ImportExportEnum type) {
        MainView mainView = getUI(rootContext);
        ImportExport importExport = new ImportExport(mainView);
        importExport.importExport(type, "", true);
    }

    public void loadURI(MainView ui) {
        int port = LimaServiceFactory.getService(
                HttpServerService.class).getHttpPort();
        String address = LimaConfig.getInstance().getHostAdress();
        String url = "http://" + address + ":" + port + "/";

        if (log.isDebugEnabled()) {
            log.debug("URL : " + url);
        }

        SwingUtil.openLink(url);
    }

    /**
     * Show a new closable tab.
     *
     * @param ui        main ui
     * @param name      name of tab to show
     * @param container the tab container
     * @param canClose  {@code false} if tab can't be closed
     */
    protected void showTab(MainView ui,
                           String name,
                           Component container,
                           boolean canClose) {
        final JTabbedPane contentTabbedPane = ui.getContentTabbedPane();

        // if contentTabbedPane doesnot yet contains tab
        if (contentTabbedPane.indexOfTab(name) == -1) {
            ClosableTabHeader closableHeader = new ClosableTabHeader();
            closableHeader.setTitle(name);
            closableHeader.setCanClose(canClose);
            contentTabbedPane.addTab(name, container);
            contentTabbedPane.setSelectedComponent(container);
            contentTabbedPane.setTabComponentAt(
                    contentTabbedPane.indexOfTab(name), closableHeader);
            closableHeader.getCloseTab().addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    JButton button = (JButton) e.getSource();
                    ClosableTabHeader closableTab =
                            (ClosableTabHeader) button.getParent();
                    String name = closableTab.getTitle();
                    contentTabbedPane.remove(contentTabbedPane.indexOfTab(name));
                }
            });
            // if contentTabbedPane contains tab
        } else {
            contentTabbedPane.setSelectedIndex(contentTabbedPane.indexOfTab(name));
        }
    }

    /**
     * Show a new closable tab.
     *
     * @param ui        main ui
     * @param name      name of tab to show
     * @param container the tab container
     */
    protected void showTab(MainView ui, String name, Component container) {
        showTab(ui, name, container, true);
    }
}
