/*
 * #%L
 * Lima Swing
 * 
 * $Id: LimaConfig.java 3674 2013-05-03 14:59:15Z athimel $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.7/lima-swing/src/main/java/org/chorem/lima/LimaConfig.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima;

import jaxx.runtime.JAXXUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.actions.MiscAction;
import org.chorem.lima.business.api.OptionsService;
import org.chorem.lima.entity.LimaCallaoDAOHelper;
import org.chorem.lima.service.LimaServiceFactory;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;
import org.nuiton.util.converter.ConverterUtil;

import java.awt.Color;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.util.Locale;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * La configuration de l'application.
 *
 * @author chemit
 * @version $Revision: 3674 $
 *          <p/>
 *          Last update : $Date: 2013-05-03 16:59:15 +0200 (Fri, 03 May 2013) $
 *          By : $Author: athimel $
 */
public class LimaConfig extends ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(LimaConfig.class);

    protected static LimaConfig instance;

    private static final String configFile = "lima-config.properties";

    protected OptionsService optionsService;

    /** La version du logiciel. */
    protected Version version;

    /**
     * Get the configuration file
     * @return name of the config. file
     * */
    /*public String getConfigFile() {
        return configFile;
    }*/

    /**
     * Get copyright text (include version).
     *
     * @return copyright text
     */
    public String getCopyrightText() {
        return "Version " + getVersion() + " Codelutin @ 2008-2012";
    }

    /**
     * Version as string.
     *
     * @return le nombre global ex: 3.2.0.0
     */
    public String getVersion() {
        return version.toString();
    }

    /**
     * Lima config constructor.
     * <p/>
     * Define all default options and action alias.
     */
    public LimaConfig() {

        // set defaut option (included configuration file name : important)
        loadDefaultOptions(Option.values());

        // set action alias
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }

        // ajout des alias (can be set in option enum ?)
        addAlias("--disableui", "--launchui false");

        // initialisation des répertoires
        // TODO what is it for ?
        //FileUtil.setCurrentDirectory(getLimaUserDirectory());
        //getLimaUserDirectory().mkdirs();

    }

    public static LimaConfig getInstance() {
        if (instance == null) {
            instance = new LimaConfig();
            instance.loadConfiguration(configFile);
        }

        return instance;
    }

    protected void loadConfiguration(String configFileName) {

        instance.setConfigFileName(configFileName);
        try {
            instance.parse();
        } catch (ArgumentsParserException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't read configuration", ex);
            }
        }
        instance.setOption(TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES,
                           LimaCallaoDAOHelper.getImplementationClassesAsString());
    }

    @Override
    public ApplicationConfig parse(String... args) throws ArgumentsParserException {

        // super parse, read config file etc...
        super.parse(args);

        // on ne peut pas connaitre la version avant la lecture du fichier
        // on supprime tout ce qui est apres - (-SNAPSHOT, -rc-1 ,...)
        String sVersion = VersionUtil.removeSnapshot(getOption("application.version"));
        version = VersionUtil.valueOf(sVersion);

        return this;

    }

    /**
     * Get application locale.
     *
     * @return configuration application locale
     */
    public Locale getLocale() {
        String local = getOption(Option.LOCALE.key);
        Locale result = ConverterUtil.convert(Locale.class, local);
        return result;
    }

    /**
     * Locale setter for command line parameters.
     *
     * @param locale new locale
     */
    public void setLocale(String locale) {
        setOption(Option.LOCALE.key, locale);
    }

    /**
     * Change locale (not command line version).
     * Save user file.
     *
     * @param newLocale new locale
     */
    public void setLocale(Locale newLocale) {
        setOption(Option.LOCALE.key, newLocale.toString());
        saveForUser();
        firePropertyChange("locale", null, newLocale);
    }

    /**
     * Get application decimal separator
     *
     * @return configuration application decimal separator
     */
    public char getDecimalSeparator() {
        char decimalSeparator = getOption(Option.DECIMAL_SEPARATOR.key).charAt(0);
        return decimalSeparator;
    }

    /**
     * Change decimal separator
     * Save user file.
     *
     * @param decimalSeparator new DecimalSeparator
     */
    public void setDecimalSeparator(String decimalSeparator) {
        setOption(Option.DECIMAL_SEPARATOR.key, decimalSeparator);
        saveForUser();
        firePropertyChange("decimalSeparator", null, decimalSeparator);
    }

    /**
     * Get application scale
     *
     * @return configuration application scale
     */
    public int getScale() {
        return getOptionAsInt(Option.SCALE.key);
    }

    /**
     * Change scale
     * Save user file.
     *
     * @param scale new Scale
     */
    public void setScale(String scale) {
        setOption(Option.SCALE.key, scale);
        saveForUser();
        firePropertyChange("scale", null, scale);
        if (log.isInfoEnabled()) {
            log.info("new scale" + scale);
        }
        optionsService.setScale(scale);
    }

    /**
     * Get application thousand separator
     *
     * @return configuration application thousand separator
     */
    public char getThousandSeparator() {
        return getOption(Option.THOUSAND_SEPARATOR.key).charAt(0);
    }

    /**
     * Change the thousand separator
     * Save user file.
     *
     * @param thousandSeparator new thousandSeparator
     */
    public void setThousandSeparator(String thousandSeparator) {
        setOption(Option.THOUSAND_SEPARATOR.key, thousandSeparator);
        saveForUser();
        firePropertyChange("thousandSeparator", null, thousandSeparator);
    }

    /**
     * currency configuration boolean
     *
     * @return {@code true} if the currency must be displayed
     */
    public boolean getCurrency() {
        return getOptionAsBoolean(Option.CURRENCY.key);
    }

    /**
     * Change the currency displaying
     *
     * @param currency the new currency to set in configuration
     */
    public void setCurrency(String currency) {
        setOption(Option.CURRENCY.key, currency);
        saveForUser();
        firePropertyChange("currency", null, currency);
    }

    /**
     * Launch ui configuration value.
     *
     * @return {@code true} if ui must be displayed
     */
    public boolean isLaunchui() {
        boolean launchUI = getOptionAsBoolean(Option.LAUNCH_UI.key);
        return launchUI;
    }

    /**
     * Launch ui setter for command line parameters.
     *
     * @param launchui new lauch ui value
     */
    public void setLaunchui(String launchui) {
        setOption(Option.LAUNCH_UI.key, launchui);
    }

    /**
     * Change full screen mode.
     * Save user file.
     *
     * @param fullscreen new fullscreen mode
     */
    public void setFullscreen(boolean fullscreen) {
        boolean oldValue = isFullScreen();
        setOption(Option.FULL_SCREEN.key, fullscreen + "");
        saveForUser();
        firePropertyChange("fullscreen", oldValue, fullscreen);
    }

    public boolean isFullScreen() {
        Boolean result = getOptionAsBoolean(Option.FULL_SCREEN.key);
        return result != null && result;
    }

    /**
     * Get support email address.
     *
     * @return support email
     */
    public String getSupportEmail() {
        return getOption(Option.SUPPORT_EMAIL.key);
    }

    /**
     * Return true if ejb mode is configured as remote.
     *
     * @return {@code true} if remote mode should be used
     */
    public boolean isEJBRemoteMode() {
        boolean result = getOptionAsBoolean(Option.OPENEJB_REMOTEMODE.key);
        return result;
    }

    public File getDataDirectory() {
        File result = getOptionAsFile(Option.DATA_DIR.key);
        return result;
    }

    public File getLimaStateFile() {
        File result = getOptionAsFile(Option.LIMA_STATE_FILE.key);
        return result;
    }

    public File getResourcesDirectory() {
        File result = getOptionAsFile(Option.RESOURCES_DIRECTORY.key);
        return result;
    }

    public File getI18nDirectory() {
        File result = getOptionAsFile(Option.I18N_DIRECTORY.key);
        return result;
    }

    public String getHostAdress() {
        return getOption(Option.LIMA_HOST_ADDRESS.key);
    }

    public void setColorSelectionFocus(String color) {
        setOption(Option.COLOR_SELECTION_FOCUS.key, color);
    }

    public Color getColorSelectionFocus() {
        return getOptionAsColor((Option.COLOR_SELECTION_FOCUS.key));
    }

    public void setComportmentEditingCell(String comportmentEditingCell) {
        setOption(Option.COMPORTMENT_EDITING_CELL.key, comportmentEditingCell);
    }

    public String getComportmentEditingCell() {
        return getOption((Option.COMPORTMENT_EDITING_CELL.key));
    }

    /** Used in ???? */
    public static final String[] DEFAULT_JAXX_PCS = {"fullScreen", "locale", "decimalSeparator", "scale", "thousandSeparator", "currency"};

    /** Used in ???? */
    public void removeJaxxPropertyChangeListener() {
        PropertyChangeListener[] toRemove = JAXXUtil.findJaxxPropertyChangeListener(DEFAULT_JAXX_PCS, getPropertyChangeListeners());
        if (toRemove == null || toRemove.length == 0) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("before remove : " + getPropertyChangeListeners().length);
            log.debug("toRemove : " + toRemove.length);

        }
        for (PropertyChangeListener listener : toRemove) {
            removePropertyChangeListener(listener);
        }
        if (log.isDebugEnabled()) {
            log.debug("after remove : " + getPropertyChangeListeners().length);
        }
    }

    /**
     * Lima option definition.
     * <p/>
     * Contains all lima configuration key, with defaut value and
     * information for jaxx configuration frame ({@link #type},
     * {@link #_transient}, {@link #_final}...)
     */
    public enum Option implements OptionDef {


        CONFIG_FILE(CONFIG_FILE_NAME, n_("lima.config.configFileName.description"), "lima-config.properties", String.class, true, true),
        DATA_DIR("lima.data.dir", n_("lima.config.data.dir.description"), "${user.home}/.lima", File.class, false, false),
        RESOURCES_DIRECTORY("lima.resources.dir", n_("lima.config.resources.dir.description"), "${lima.data.dir}/resources-${application.version}", String.class, false, false),
        I18N_DIRECTORY("lima.i18n.dir", n_("lima.config.i18n.dir.description"), "${lima.resources.dir}/i18n", String.class, false, false),
        LOCALE("lima.ui.locale", n_("lima.config.locale.description"), "fr_FR", Locale.class, false, false),
        DECIMAL_SEPARATOR("lima.data.bigDecimal.decimalSeparator", n_("lima.config.decimalseparator.description"), ",", String.class, false, false),
        SCALE("lima.data.bigDecimal.scale", n_("lima.config.scale.description"), "2", Integer.class, false, false),
        THOUSAND_SEPARATOR("lima.thousandSeparator", n_("limma.config.thousandseparator.description"), " ", String.class, false, false),
        CURRENCY("lima.config.currency", n_("lima.config.currency.description"), "none", Boolean.class, false, false),
        FULL_SCREEN("lima.ui.fullscreen", n_("lima.config.ui.fullscreen.description"), "false", Boolean.class, false, false),
        LAUNCH_UI("lima.ui.launchui", n_("lima.config.ui.flaunchui.description"), "true", Boolean.class, true, true),
        SUPPORT_EMAIL("lima.misc.supportemail", n_("lima.misc.supportemail.description"), "support@codelutin.com", String.class, false, false),
        OPENEJB_REMOTEMODE("lima.openejb.remotemode", n_("lima.openejb.remotemode.description"), "false", String.class, false, false),
        LIMA_HOST_ADDRESS("lima.host.address", n_("lima.config.host.adress"), "localhost", String.class, false, false),
        LIMA_STATE_FILE("lima.ui.state.file", n_("lima.config.state.file"), "${lima.data.dir}/limaState.xml", String.class, false, false),
        COLOR_SELECTION_FOCUS("lima.ui.table.cell.colorSelectionFocus", n_("lima.config.colorselectionfocus"), "#000000", Color.class, false, false),
        COMPORTMENT_EDITING_CELL("lima.ui.table.cell.comportmentEditingCell",n_("lima.config.comportmenteditingcell"), "ALL", ComportmentEditingCellEnum.class, false, false);

        public enum ComportmentEditingCellEnum {ALL, NOTHING}

        public final String key;

        public final String description;

        public String defaultValue;

        public final Class<?> type;

        public boolean _transient;

        public boolean _final;

        Option(String key, String description, String defaultValue,
               Class<?> type, boolean _transient, boolean _final) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this._final = _final;
            this._transient = _transient;
        }

        @Override
        public boolean isFinal() {
            return _final;
        }

        @Override
        public void setFinal(boolean _final) {
            this._final = _final;
        }

        @Override
        public boolean isTransient() {
            return _transient;
        }

        @Override
        public void setTransient(boolean _transient) {
            this._transient = _transient;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public String getDescription() {
            return _(description);
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public Class<?> getType() {
            return type;
        }
    }

    /** Lima action definition. */
    public enum Action {

        HELP(n_("lima.action.commandline.help"), MiscAction.class.getName() + "#help", "-h", "--help");

        /** Before init action step. */
        public static final int BEFORE_EXIT_STEP = 0;

        /** After init action step. */
        public static final int AFTER_INIT_STEP = 1;

        public String description;

        public String action;

        public String[] aliases;

        Action(String description, String action, String... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }

        public String getDescription() {
            return _(description);
        }
    }

    /**
     * Override save action to propagate some option to server.
     */
    @Override
    public void save(File file,
                     boolean forceAll,
                     String... excludeKeys) throws IOException {

        super.save(file, forceAll, excludeKeys);

        // propagate scale option to serveur option
        optionsService = LimaServiceFactory.getService(OptionsService.class);

        // scale server option
        String scaleOption = getOption("scale");
        if (StringUtils.isNotBlank(scaleOption)) {
            optionsService.setScale(scaleOption);
        }
    }
}
