/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialPeriodViewHandler.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.2/lima-swing/src/main/java/org/chorem/lima/ui/financialperiod/FinancialPeriodViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.financialperiod;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.FinancialPeriodService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.business.utils.FinancialPeriodComparator;
import org.chorem.lima.entity.ClosedPeriodicEntryBook;
import org.chorem.lima.service.LimaServiceFactory;
import org.jdesktop.swingx.JXTable;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.Collections;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler for FinancialPeriodView.
 *
 * @author chatellier
 * @version $Revision: 3777 $
 * 
 * Last update : $Date: 2014-04-14 09:29:08 +0200 (Mon, 14 Apr 2014) $
 * By : $Author: sbavencoff $
 */
public class FinancialPeriodViewHandler implements ServiceListener {

    private static final Log log = LogFactory.getLog(FinancialPeriodViewHandler.class);

    protected FinancialPeriodService financialPeriodService;

    protected FinancialPeriodView view;

    protected FinancialPeriodViewHandler(FinancialPeriodView view) {
        this.view = view;
        
        financialPeriodService = LimaServiceFactory.getService(FinancialPeriodService.class);
        LimaServiceFactory.addServiceListener(ImportService.class, this);
        LimaServiceFactory.addServiceListener(FiscalPeriodService.class, this);
    }

    /**
     * Init view.
     */
    public void init() {

        InputMap inputMap = view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();

        // add action on Ctrl + B
        String binding = "close-FinancialPeriod";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_B, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                blockFinancialPeriod();
            }
        });


        // get data from service
        List<ClosedPeriodicEntryBook> closedPeriodicEntryBooks = financialPeriodService.
                getAllClosedPeriodicEntryBooksFromUnblockedFiscalPeriod();
        Collections.sort(closedPeriodicEntryBooks, new FinancialPeriodComparator());
        if (log.isDebugEnabled()) {
            log.debug(String.format("Loaded %d closed periodic entry book", closedPeriodicEntryBooks.size()));
        }

        // notify ui
        view.getFinancialPeriodTableModel().setValues(closedPeriodicEntryBooks);
    }

    /**
     * Block selected period.
     */
    public void blockFinancialPeriod() {
        JXTable financialPeriodeTable = view.getFinancialPeriodTable();
        int selectedRow = financialPeriodeTable.getSelectedRow();
        FinancialPeriodTableModel model = (FinancialPeriodTableModel) view.getFinancialPeriodTable().getModel();

        // blocked it
        ClosedPeriodicEntryBook closedPeriodicEntryBook = model.get(selectedRow);
        int response = JOptionPane.showConfirmDialog(view, t("lima.ui.financialperiod.blockmessage"),
            t("lima.ui.financialperiod.blocktitle"),
            JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

        if (response == JOptionPane.YES_OPTION) {
            closedPeriodicEntryBook = financialPeriodService.blockClosedPeriodicEntryBook(closedPeriodicEntryBook);
            model.setValue(selectedRow, closedPeriodicEntryBook);
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodName) {
        if (serviceName.contains("FiscalPeriod") || methodName.contains("importAll")) {
            init();
        }
    }
}
