/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialPeriodComboBoxModel.java 3137 2011-05-20 14:26:33Z vsalaun $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.5/lima-swing/src/main/java/org/chorem/lima/ui/combobox/FinancialPeriodComboBoxModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.combobox;

import java.util.List;
import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.FinancialPeriodServiceMonitorable;
import org.chorem.lima.business.FiscalPeriodServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.entity.FinancialPeriod;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.financialperiod.FinancialPeriodTableModel;

/**
 * Opened financial period combo box model.
 * 
 * @author chatellier
 * @version $Revision: 3137 $
 * 
 * Last update : $Date: 2011-05-20 16:26:33 +0200 (Fri, 20 May 2011) $
 * By : $Author: vsalaun $
 */
public class FinancialPeriodComboBoxModel extends AbstractListModel implements ComboBoxModel, ServiceListener {
    
    private static final long serialVersionUID = 1L;

    private static final Log log =
        LogFactory.getLog(FinancialPeriodTableModel.class);

    protected Object selectedFinancialPeriod;
    
    protected List<FinancialPeriod> datasCache;
    
    protected FinancialPeriodServiceMonitorable financialPeriodService;
    
    protected FiscalPeriod selectedFiscalPeriod;
    
    private boolean displayAllPeriods;

    /**
     * Constructor for locked period list.
     */
    public FinancialPeriodComboBoxModel() {
        this(false);
    }

    /**
     * Constructor for period list.
     * 
     * @param all if {@code true}, display all available period, otherwize
     *  display only locked ones
     */
    public FinancialPeriodComboBoxModel(boolean displayAllPeriods) {
        
        this.displayAllPeriods = displayAllPeriods;
        
    	financialPeriodService =
            LimaServiceFactory.getInstance().getService(
                    FinancialPeriodServiceMonitorable.class);
        financialPeriodService.addListener(this);
        LimaServiceFactory.getInstance().getService(
                FiscalPeriodServiceMonitorable.class).addListener(this);
        LimaServiceFactory.getInstance().getService(
                ImportServiceMonitorable.class).addListener(this);
        if (displayAllPeriods) {
            datasCache=getAllDataList();
        }
        else {
            datasCache=getDataList();
        }
	}

	public void setFiscalPeriod(FiscalPeriod fiscalPeriod){
        selectedFiscalPeriod=fiscalPeriod;
        if (displayAllPeriods) {
            datasCache=getAllDataList();
        }
        else {
            datasCache=getDataList();
        }
        fireContentsChanged(this, 0, datasCache.size());
    }
    
    @Override
    public int getSize() {
        return datasCache.size();
    }

    @Override
    public Object getElementAt(int index) {
        return datasCache.get(index);
    }

    @Override
    public void setSelectedItem(Object anItem) {
        selectedFinancialPeriod = anItem;
    }

    @Override
    public Object getSelectedItem() {
        return selectedFinancialPeriod;
    }
    
    //get unblocked financial periods data list 
    public List<FinancialPeriod> getDataList(){
        List<FinancialPeriod> result = null;
        try {
            if (selectedFiscalPeriod != null){
                result = financialPeriodService.getFinancialPeriods(selectedFiscalPeriod.getBeginDate(), selectedFiscalPeriod.getEndDate());
            }
            else {
                result = financialPeriodService.getUnblockedFinancialPeriods();
            }
        }
        catch (LimaException ex) {
            if (log.isDebugEnabled()){
                log.debug("Can't get Financial Period list",ex);
            }
        }
        return result;   
    }
    
    //get all financial periods data list
    public List<FinancialPeriod> getAllDataList() {
    	List<FinancialPeriod> result = null;
        try {
            if (selectedFiscalPeriod != null){
                result = financialPeriodService.getFinancialPeriods(selectedFiscalPeriod.getBeginDate(), selectedFiscalPeriod.getEndDate());
            }
            else {
                result = financialPeriodService.getAllFinancialPeriods();
            }
        }
        catch (LimaException ex) {
            if (log.isDebugEnabled()){
                log.debug("Can't get Financial Period list",ex);
            }
        }
        return result; 
    }
    
    
    public void refresh(){
        datasCache = getDataList();
        fireContentsChanged(this, 0, datasCache.size());
    }
    
    public void refreshAll() {
        datasCache = getAllDataList();
        fireContentsChanged(this, 0, datasCache.size());
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        if (serviceName.contains("Period") || methodeName.contains("importAll")){
            if (displayAllPeriods) {
                datasCache=getAllDataList();
            }
            else {
                datasCache=getDataList();
            }
        }    
    }
}
