/*
 * #%L
 * Lima :: Swing
 * 
 * $Id: VatReportTableModel.java 3278 2011-08-16 14:09:44Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.5.1/lima-swing/src/main/java/org/chorem/lima/ui/vatreports/VatReportTableModel.java $
 * %%
 * Copyright (C) 2008 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package org.chorem.lima.ui.vatreports;

import static org.nuiton.i18n.I18n._;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.beans.VatStatementAmounts;
import org.chorem.lima.business.FinancialTransactionServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ReportServiceMonitorable;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.VatStatementServiceMonitorable;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.ErrorHelper;
import org.nuiton.topia.TopiaException;

public class VatReportTableModel extends AbstractTableModel implements ServiceListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = -6301817853711018389L;
    
    /** log. */
    private static final Log log = LogFactory
            .getLog(VatReportTableModel.class);
    
    /** Services. */
    protected ReportServiceMonitorable reportService;
    protected VatStatementServiceMonitorable vatStatementService;
    protected FinancialTransactionServiceMonitorable financialTransactionService;
    protected ImportServiceMonitorable importService;
    
    /** selected financial period */
    protected Date selectedBeginDate;
    protected Date selectedEndDate;
    
    /** data cache */
    protected List<VatStatementAmounts> cacheDataList;
    
    /**
     * variable used for the document edition,
     * if "true" then the document will be automatically filled in
     * if "false" the the document will be empty
     */
    protected String autocomplete;
    
    
    public VatReportTableModel() {
        reportService =
            LimaServiceFactory.getInstance().getService(
                ReportServiceMonitorable.class);
        
        //add listeners
        vatStatementService =
            LimaServiceFactory.getInstance().getService(
                VatStatementServiceMonitorable.class);
        vatStatementService.addListener(this);
        financialTransactionService = 
            LimaServiceFactory.getInstance().getService(
                FinancialTransactionServiceMonitorable.class);
        financialTransactionService.addListener(this);
        importService =
            LimaServiceFactory.getInstance().getService(
                ImportServiceMonitorable.class);
        importService.addListener(this);
        
        //sets autocomplete mode to true on start
        autocomplete = "true";
    }
    
    protected List<VatStatementAmounts> getDataList() throws TopiaException {
        
        List<VatStatementAmounts> list = null;
        
        if ((selectedBeginDate != null ) || (selectedEndDate != null)) {
            try {
                list = vatStatementService.vatStatementReport(
                        getBeginDate(),
                        getEndDate());
            }
            catch (LimaException eee) {
                if (log.isErrorEnabled()) {
                    log.debug("Can't update model", eee);
                }
                ErrorHelper.showErrorDialog(_("lima.vatreport.listerror"), eee);
            }
        }
        return list;
    }

    public void refresh(){
        try {
            cacheDataList = getDataList();
        } catch (TopiaException eee) {
            if (log.isErrorEnabled()) {
                log.debug("Can't update model", eee);
            }
            ErrorHelper.showErrorDialog("Can't get VAT list", eee);
        }
        fireTableDataChanged();
    }
    
    @Override
    public Class<?> getColumnClass(int column) {

        Class<?> result = null;

        switch (column) {
        case 0:
            result = String.class; //label
            break;
        case 1:
            result = BigDecimal.class; //solde
            break;
        }

        return result;
    }
    
    @Override
    public String getColumnName(int column) {
        String result = "n/a";

        switch (column) {
        case 0:
            result = _("lima.table.label"); //label
            break;
        case 1:
            result = _("lima.table.solde"); //solde
            break;
        }

        return result;
    }

    @Override
    public int getRowCount() {
        int result = 0;
        
        if (cacheDataList != null) {
            result = cacheDataList.size();
        }

        return result;
    }

    @Override
    public int getColumnCount() {
        return 2;
    }
    
    public VatStatementAmounts getElementAt(int row){
        VatStatementAmounts currentRow = cacheDataList.get(row);
        return currentRow;
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        // just prevent too much result
        if (cacheDataList != null) {
            VatStatementAmounts value = cacheDataList.get(row);
            switch (column) {
                case 0:
                    result = value.getLabel();
                    break;
                case 1:
                    if ((value.getAmount() != null) && (!value.isHeader())) {
                        result = value.getAmount();
                    } else {
                        result = null;
                    }
                    break;
            }
        }
        return result;
    }

    public void setBeginDate(Date date) {
        selectedBeginDate = date;
    }
    
    public Date getBeginDate() {
        return selectedBeginDate;
    }
    
    public void setEndDate(Date date) {
        selectedEndDate = date;
    }
    
    public Date getEndDate() {
        return selectedEndDate;
    }
    
    public String getAutocomplete() {
        return autocomplete;
    }
    
    public void setAutocomplete() {
        if (getAutocomplete().equals("true")) {
            autocomplete = "false";
        } else {
            autocomplete = "true";
        }
    }
    
    @Override
    public boolean isCellEditable(int row, int column) {
        return false;
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        //automatic refresh when FinancialTransaction, VatStatement chart or import for VAT services have been used by the user
        if (serviceName.contains("FinancialTransaction") || methodeName.contains("VatStatement") || methodeName.contains("import") ) {
            refresh();
        }
    }

}
