/*
 * #%L
 * Lima Swing
 * 
 * $Id: MainViewHandler.java 3059 2010-12-01 09:28:32Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.4.3/lima-swing/src/main/java/org/chorem/lima/ui/MainViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui;

import static org.nuiton.i18n.I18n._;
import java.awt.Desktop;
import java.net.URL;
import java.util.Locale;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.editor.config.ConfigUIHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaConfig;
import org.chorem.lima.LimaContext;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.ui.account.AccountView;
import org.chorem.lima.ui.accountsreports.AccountsReportsView;
import org.chorem.lima.ui.balance.BalanceView;
import org.chorem.lima.ui.entrybook.EntryBookView;
import org.chorem.lima.ui.entrybooksreports.EntryBooksReportsView;
import org.chorem.lima.ui.financialperiod.FinancialPeriodView;
import org.chorem.lima.ui.fiscalperiod.FiscalPeriodView;
import org.chorem.lima.ui.financialstatementchart.FinancialStatementChartView;
import org.chorem.lima.ui.financialstatementreport.FinancialStatementReportView;
import org.chorem.lima.ui.financialtransaction.FinancialTransactionView;
import org.chorem.lima.ui.financialtransactionsearch.FinancialTransactionSearchView;
import org.chorem.lima.ui.financialtransactionunbalanced.FinancialTransactionUnbalancedView;
import org.chorem.lima.ui.home.HomeView;
import org.chorem.lima.ui.identity.IdentityForm;
import org.chorem.lima.ui.importexport.ImportExport;
import org.chorem.lima.ui.ledger.LedgerView;
import org.chorem.lima.ui.lettering.LetteringView;
import org.chorem.lima.util.ErrorHelper;

/**
 * Handler for main view.
 * 
 * Date: 8 nov. 2009
 * Time: 09:59:54
 * 
 * @author chemit
 * @version $Revision: 3059 $
 * 
 * Last update : $Date: 2010-12-01 10:28:32 +0100 (mer., 01 déc. 2010) $
 * By : $Author: echatellier $
 */
public class MainViewHandler {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(MainViewHandler.class);

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param fullscreen  flag pour indiquer si on doit ouvrir l'ui en model console (pleine ecran).
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public MainView initUI(LimaContext rootContext, boolean fullscreen) {
        // show main ui
        MainView ui = new MainView(rootContext);

        LimaContext.MAIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);

        // set fullscreen propery on main ui
        ui.getGraphicsConfiguration().getDevice().setFullScreenWindow(fullscreen ? ui : null);

        return ui;
    }

    public void changeLanguage(MainView mainUI, Locale newLocale) {
        LimaConfig config = mainUI.getConfig();

        // sauvegarde de la nouvelle locale
        config.setLocale(newLocale);

        // rechargement i18n
        LimaContext.get().initI18n();

        // on recharge l'ui
        reloadUI(LimaContext.get(), config.isFullScreen());
    }

    /**
     * Ferme l'application.
     *
     * @param ui l'ui principale de l'application
     */
    public void close(MainView ui) {
        if (log.isInfoEnabled()) {
            log.info("Lima quitting...");
        }
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }
        try {
            LimaContext.get().close();

            Runtime.getRuntime().halt(0);

        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("error while closing " + ex.getMessage(), ex);
            }
            Runtime.getRuntime().halt(1);
        }
    }

    /**
     * Méthode pour changer de mode d'affichage.
     * <p/>
     * Si <code>fullscreen</code> est à <code>true</code> alors on passe en
     * mode console (c'est à dire en mode plein écran exclusif), sinon on
     * passe en mode fenetré normal.
     *
     * @param ui         l'ui principale de l'application
     * @param fullscreen le nouvel état requis.
     */
    public void changeScreen(MainView ui, final boolean fullscreen) {
        boolean canContinue = ensureModification(ui);
        if (!canContinue) {
            return;
        }

        ui.getConfig().setFullscreen(fullscreen);

        reloadUI(LimaContext.get(), fullscreen);
    }

    public void showConfig(JAXXContext context) {
        MainView ui = getUI(context);
        LimaConfig config = context.getContextValue(LimaConfig.class);

        //TODO pepin 2010-08-20 add all options

        ConfigUIHelper helper = new ConfigUIHelper(config);
        helper.addCategory(_("lima.config.category.directories"), _("lima.config.category.directories.description"));
        helper.addOption(LimaConfig.Option.CONFIG_FILE);
        
        helper.addCategory(_("lima.config.category.other"), _("lima.config.category.other.description"));
        helper.addOption(LimaConfig.Option.FULL_SCREEN);
        helper.addOption(LimaConfig.Option.LOCALE);
        
        helper.buildUI(context, _("lima.config.category.directories"));
        helper.displayUI(ui, false);
    }

    public void showHelp(JAXXContext context, String helpId) {

    }

    public void closeHelp(JAXXContext context) {

    }

    public void gotoSite(JAXXContext rootContext) {

        LimaConfig config = rootContext.getContextValue(LimaConfig.class);

        URL siteURL = config.getOptionAsURL("application.site.url");

        log.info("goto " + siteURL);
        if (Desktop.isDesktopSupported() && Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception ex) {
                if (log.isDebugEnabled()) {
                    log.error("Can't open lima website", ex);
                }
                ErrorHelper.showErrorDialog("Can't open lima website", ex);
            }
        }
    }

    public void showAbout(MainView ui) {

        AboutPanel about = new AboutPanel();
        about.setTitle(_("lima.title.about"));
        about.setAboutText(_("lima.title.about.description"));
        about.setBottomText(ui.getConfig().getCopyrightText());
        about.setIconPath("/icons/lima.png");
        about.setLicenseFile("META-INF/lima-swing-LICENSE.txt");
        about.setThirdpartyFile("META-INF/lima-swing-THIRD-PARTY.txt");
        about.init();
        about.showInDialog(ui, true);
    }

    /**
     * Permet de recharger l'ui principale et de changer de le mode d'affichage.
     *
     * @param rootContext le contexte applicatif
     * @param fullscreen  le type de fenetre à reouvrir
     */
    protected void reloadUI(LimaContext rootContext, boolean fullscreen) {

        // must remove all properties listener on config
        rootContext.getContextValue(LimaConfig.class).removeJaxxPropertyChangeListener();

        // scan main ui
        MainView ui = getUI(rootContext);

        if (ui != null) {
            LimaContext.MAIN_UI_ENTRY_DEF.removeContextValue(rootContext);

            ui.dispose();

            ui.setVisible(false);
        }

        ui = initUI(rootContext, fullscreen);

        // show ui
        ui.setVisible(true);
    }

    /**
     * Test if there is some modification on screen,
     *
     * @param rootContext the context
     * @return <code>true</code> if no more modification is detected
     * @throws IllegalArgumentException if rootContext is null
     */
    protected boolean ensureModification(JAXXContext rootContext) throws IllegalArgumentException {
        if (rootContext == null) {
            throw new IllegalArgumentException("rootContext can not be null");
        }
        MainView ui = getUI(rootContext);
        if (ui == null) {
            // no ui, so no modification
            return true;
        }
        // check ui is not modified
        return true;
    }

    protected MainView getUI(JAXXContext context) {
        if (context instanceof MainView) {
            return (MainView) context;
        }
        MainView ui = LimaContext.MAIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }

    public void showHomeView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        HomeView homeView = new HomeView(mainView);
        mainView.showTab(_("lima.tab.home"), homeView, false);
    }
    
    public void showIdentity(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        IdentityForm identityForm = new IdentityForm();
        identityForm.setLocationRelativeTo(mainView);
        identityForm.setVisible(true);
    }
    
    /**
     * Show account tree table view to create or modify accounts
     * @param rootContext
     */
    public void showAccountView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        AccountView accountView = new AccountView(mainView);
        mainView.showTab(_("lima.charts.account"), accountView);
    }
    
    /**
     * Show account tree table view to create or modify accounts
     * @param rootContext
     */
    public void showEntryBookView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        EntryBookView entryBookView= new EntryBookView(mainView);
        mainView.showTab(_("lima.common.entrybooks"), entryBookView);
    }

    /**
     * Show fiscal period view to create or block a period
     * @param rootContext
     */
    public void showFiscalPeriodView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FiscalPeriodView fiscalPeriodView = new FiscalPeriodView(mainView);
        mainView.showTab(_("lima.charts.fiscalyear"), fiscalPeriodView);
    }
    
    /**
     * Show financial period view to create or block a period
     * @param rootContext
     */
    public void showFinancialPeriodView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialPeriodView financialPeriodView =
            new FinancialPeriodView(mainView);
        mainView.showTab(_("lima.charts.financialperiod"), financialPeriodView);
    }
    
    /**
     * Show financial statement view to generate balance sheet and income statement
     * @param rootContext
     */
    public void showFinancialStatementView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialStatementChartView financialStatementView =
            new FinancialStatementChartView(mainView);
        mainView.showTab(_("lima.charts.financialstatement"), financialStatementView);
    }    
    
    /**
     * Show account table report to view an account on a period
     * @param rootContext
     */
    public void showAccountReports(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        AccountsReportsView accountsReportsView =
            new AccountsReportsView(mainView);
        mainView.showTab(_("lima.reports.accounts"), accountsReportsView);
    }
    
    /**
     * Show entry book table report to view an account on a period
     * @param rootContext
     */
    public void showEntryBookReports(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        EntryBooksReportsView entryBooksReportsView= new EntryBooksReportsView(mainView);
        mainView.showTab(_("lima.reports.entrybooks"), entryBooksReportsView);
    }
    
    
    /**
     * Show financial transactions view to create entries
     * @param rootContext
     */
    public void showTransactionView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionView transactionView = new FinancialTransactionView(mainView);
        mainView.showTab(_("lima.entries.writetransaction"), transactionView);
    }
    
    public void showTransactionUnbalancedView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionUnbalancedView searchResultView = new FinancialTransactionUnbalancedView(mainView);
        mainView.showTab(_("lima.entries.searchunbalancedtransaction"), searchResultView);
    }
    
    public void showTransactionSearchView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialTransactionSearchView searchView = new FinancialTransactionSearchView(mainView);
        mainView.showTab(_("lima.entries.searchtransaction"), searchView);
    }
    
    public void showLetteringView(JAXXContext rootContext){
         MainView mainView = getUI(rootContext);
         LetteringView letteringView = new LetteringView(mainView);
         mainView.showTab(_("lima.entries.lettering"), letteringView);
    }
    
    public void showBalanceView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
       BalanceView balanceView = new BalanceView(mainView);
       mainView.showTab(_("lima.reports.balance"), balanceView);
    }
    
    public void showLedgerView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
       LedgerView ledgerView = new LedgerView(mainView);
       mainView.showTab(_("lima.reports.ledger"), ledgerView);
    }
    
    public void showFinancialStatementReportsView(JAXXContext rootContext) {
        MainView mainView = getUI(rootContext);
        FinancialStatementReportView financialStatementReportView = new FinancialStatementReportView(mainView);
        mainView.showTab(_("lima.reports.financialstatement"), financialStatementReportView);
    }

    public void showImportExportView(JAXXContext rootContext, ImportExportEnum type) {
        MainView mainView = getUI(rootContext);
        ImportExport importExport = ImportExport.getInstance(mainView);
        importExport.importExport(type, "", true);
    }
}
