/*
 * #%L
 * jTimer
 * 
 * $Id: ProjectsAndTasksCellRenderer.java 2853 2012-04-01 16:37:54Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2007 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.jtimer.ui.treetable;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.image.ImageObserver;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.data.DataEventListener;
import org.chorem.jtimer.data.TimerCore;
import org.chorem.jtimer.entities.TimerProject;
import org.chorem.jtimer.entities.TimerTask;
import org.jdesktop.swingx.JXTreeTable;

/**
 * Gere l'affichage des noeuds de l'arbre.
 * 
 * Comme une icone quand la taches est lancee...
 * 
 * @author chatellier
 * @version $Revision: 2853 $
 * 
 * Last update : $Date: 2012-04-01 18:37:54 +0200 (Sun, 01 Apr 2012) $
 * By : $Author: echatellier $
 */
public class ProjectsAndTasksCellRenderer extends DefaultTreeCellRenderer
        implements DataEventListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 1383276150996517529L;

    /** Logger. */
    private static Log log = LogFactory.getLog(ProjectsAndTasksCellRenderer.class);

    /** Running task icon. */
    protected ImageIcon runningIcon;

    /** Running tasks. */
    protected Collection<TimerTask> runningTasks;

    /** Tree table reference for image observer. */
    protected JXTreeTable treeTable;

    /** Node image nodeObserver for running task animated icon. */
    protected NodeImageObserver nodeObserver;

    /**
     * Constructor.
     * 
     * @param treeTable Tree table reference for image observer
     * @param core TimerCore
     */
    public ProjectsAndTasksCellRenderer(JXTreeTable treeTable, TimerCore core) {
        this.treeTable = treeTable;

        // init
        runningTasks = new HashSet<TimerTask>();

        URL runnigIconUrl = ProjectsAndTasksCellRenderer.class
                .getResource("/org/chorem/jtimer/resources/running.gif");
        runningIcon = new ImageIcon(runnigIconUrl);
        nodeObserver = new NodeImageObserver();
        runningIcon.setImageObserver(nodeObserver);

        // be notified on events
        core.getData().addDataEventListener(this);
    }

    /*
     * @see org.jdesktop.swingx.renderer.DefaultTreeRenderer#getTreeCellRendererComponent(javax.swing.JTree, java.lang.Object, boolean, boolean, boolean, int, boolean)
     */
    @Override
    public Component getTreeCellRendererComponent(JTree tree, Object value,
            boolean selected, boolean expanded, boolean leaf, int row,
            boolean hasFocus) {

        Object localValue = value;

        // if this is a task
        if (value instanceof TimerTask) {
            TimerTask task = (TimerTask) value;

            // add style in project in sync
            String taskName = task.getName();
            if (task instanceof TimerProject
                    && ((TimerProject) task).isSynchronized()) {
                taskName = taskName
                        .substring(TimerProject.SYNCHRONIZED_PROJECT_NAME_PREFIX
                                .length());
            }

            // task name should not be "null"
            localValue = taskName;
        }

        // le fait en 2 temps car sinon, on voit de temps en temps
        // le toString() suivit du setName()
        super.getTreeCellRendererComponent(tree, localValue,
                selected, expanded, leaf, row, hasFocus);

        // if this is a task
        if (value instanceof TimerTask) {
            TimerTask task = (TimerTask) value;

            // add style in project in sync
            if (task instanceof TimerProject
                    && ((TimerProject) task).isSynchronized()) {
                setFont(getFont().deriveFont(Font.ITALIC));
            } else {
                setFont(getFont().deriveFont(Font.PLAIN));
            }

            // add icon if task is running  
            if (runningTasks.contains(task)) {
                setIcon(runningIcon);
                nodeObserver.addRow(row);
            }
            else {
                // force no default icon
                setIcon(null);
                nodeObserver.removeRow(row);
            }

            // add color if task is closed
            if (task.isClosed()) {
                // fix selection color
                if (selected) {
                    setForeground(Color.GRAY.brighter());
                } else {
                    setForeground(Color.GRAY);
                }
            }
        }

        return this;
    }

    /**
     * Enclosed class to manage gif image refresh.
     */
    class NodeImageObserver implements ImageObserver {

        protected Set<Integer> rows = Collections.synchronizedSet(new HashSet<Integer>());

        public void addRow(Integer row) {
            rows.add(row);
        }

        public void removeRow(Integer row) {
            rows.remove(row);
        }

        /*
         * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int, int, int, int)
         */
        @Override
        public boolean imageUpdate(Image img, int flags, int x, int y, int w,
                int h) {
            if ((flags & (FRAMEBITS | ALLBITS)) != 0) {
                for (Integer row : rows) {
                    Rectangle rowBounds = treeTable.getCellRect(row, 0, true);
                    if (rowBounds != null) {
                        treeTable.repaint(rowBounds);
                    }
                }
            }
            return (flags & (ALLBITS | ABORT)) == 0;
        }
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#addProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void addProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#addTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void addTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#dataLoaded(java.util.Collection)
     */
    @Override
    public void dataLoaded(Collection<TimerProject> projects) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#deleteProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void deleteProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#deleteTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void deleteTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#modifyProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void modifyProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#modifyTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void modifyTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#changeClosedState(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void changeClosedState(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#startTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void startTask(TimerTask task) {

        if (log.isDebugEnabled()) {
            log.debug("startTask on " + task.getName());
        }

        // remember running task
        runningTasks.add(task);
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#stopTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void stopTask(TimerTask task) {

        if (log.isDebugEnabled()) {
            log.debug("stopTask on " + task.getName());
        }

        // remember don't running task
        runningTasks.remove(task);
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#postMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void moveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#preMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preMoveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#postMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void postMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void preMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#setTaskTime(org.chorem.jtimer.entities.TimerTask, java.util.Date, java.lang.Long)
     */
    @Override
    public void setTaskTime(TimerTask task, Date date, Long time) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#setAnnotation(org.chorem.jtimer.entities.TimerTask, java.util.Date, java.lang.String)
     */
    @Override
    public void setAnnotation(TimerTask task, Date date, String annotation) {

    }
}
