/*
 * #%L
 * jTimer
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2007 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ws.xmlrpc;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.data.TimerDataManager;
import org.chorem.jtimer.entities.ConnectionData;
import org.chorem.jtimer.entities.TimerProject;
import org.chorem.jtimer.entities.TimerTask;
import org.chorem.jtimer.ws.ConnectionDataHandler;
import org.chorem.jtimer.ws.exception.WebServiceException;

/**
 * WSDaemon. Do synchronization with remote service.
 * 
 * @author chatellier
 * @version $Revision: 2602 $
 * 
 * Last update : $Date: 2009-06-15 19:13:17 +0200 (lun., 15 juin 2009) $
 * By : $Author: echatellier $
 */
public class WSDaemon extends java.util.TimerTask {

    /** Log */
    private static Log log = LogFactory.getLog(WSDaemon.class);

    /** Data manager */
    protected TimerDataManager dataManager;

    /** Web service implementation */
    protected ChoremXMLRPCClient managementService;

    /** Connection handler */
    protected ConnectionDataHandler connectionDataHandler;

    /**
     * Constructor.
     * 
     * @param managementService managementService to use
     * @param dataManager data
     */
    public WSDaemon(ChoremXMLRPCClient managementService, TimerDataManager dataManager) {
        // data manager
        this.dataManager = dataManager;

        // web service impl
        this.managementService = managementService;
    }

    /**
     * Set connection handler.
     * 
     * @param connectionDataHandler connection handler
     */
    public void setConnectionDataHandler(
            ConnectionDataHandler connectionDataHandler) {

        this.connectionDataHandler = connectionDataHandler;
    }

    /*
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {

        // fix NPE (happen in tests)
        // but must not happen in main line
        if (connectionDataHandler == null) {
            if (log.isDebugEnabled()) {
                log.debug("No handler set, can't synchronize");
            }
            
            return;
        }
        
        // get connection info
        // if can't be found, ask user
        ConnectionData cData = connectionDataHandler
                .getConnectionInformation(managementService);

        if (cData != null) {
            String login = cData.getLogin();

            if (log.isInfoEnabled()) {
                log.info("Refresh...");
            }

            // managementService.getUserProjects();
            try {
                String userId = managementService.login(login, "");

                if (log.isDebugEnabled()) {
                    log.debug("UserId = " + userId);
                }

                if (userId == null || userId.isEmpty()) {
                    // reask user for connection identifier
                    if (log.isWarnEnabled()) {
                        log.warn("TODO : Unknown user identifier, should reask here");
                    }
                } else {
                    // first, get updated project list
                    refreshProjectList(userId);
                    if (log.isInfoEnabled()) {
                        log.info("Project list refreshed");
                    }

                    // now send local time to server
                    managementService.synchronize(userId, dataManager
                            .getProjectsList());
                    if (log.isInfoEnabled()) {
                        log.info("Times updated");
                    }

                    if (log.isInfoEnabled()) {
                        log.info("Refresh complete");
                    }
                }

            } catch (WebServiceException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't synchronize with remote service", e);
                }
            }
        } else {
            if (log.isInfoEnabled()) {
                log.info("Can't do synchronization");
            }
        }
    }

    /**
     * Refresh projects and task list.
     * 
     * @param userId user id
     */
    protected void refreshProjectList(String userId) {

        try {
            List<TimerProject> projects = managementService
                    .getUserProjects(userId);

            if (projects != null) {
                // Manage each remote project
                for (TimerProject project : projects) {
                    String projectName = project.getName();

                    TimerProject localProject = getProjectByName(projectName);
                    if (localProject != null) {
                        synchronizeTask(localProject, project);
                    } else {
                        // else only notify project added ?
                        dataManager.addProject(project);
                    }
                }
            }
        } catch (WebServiceException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't synchronize with remote service", e);
            }
        }
    }

    /**
     * Found project by name in local projects.
     * 
     * @param projectName project name to find
     * @return local project found
     */
    protected TimerProject getProjectByName(String projectName) {

        TimerProject projectFound = null;

        for (TimerProject project : dataManager.getProjectsList()) {
            if (project.getName().equals(projectName)) {
                projectFound = project;
            }
        }

        return projectFound;
    }

    /**
     * Found task by name in a task subtask
     * 
     * @param task task
     * @param taskName task name to found
     * 
     * @return local task found
     */
    protected TimerTask getTaskByName(TimerTask task, String taskName) {

        TimerTask taskFound = null;

        for (TimerTask lTask : task.getSubTasks()) {
            if (lTask.getName().equals(taskName)) {
                taskFound = lTask;
            }
        }

        return taskFound;
    }

    /**
     * Synchronize two projects.
     * 
     * @param localTask local project
     * @param task other task
     */
    protected void synchronizeTask(TimerTask localTask, TimerTask task) {

        for (TimerTask lTask : task.getSubTasks()) {
            TimerTask localTaskFound = getTaskByName(localTask, lTask.getName());
            if (localTaskFound != null) {
                synchronizeTask(localTaskFound, lTask);
            } else {
                dataManager.addTask(localTask, lTask);
            }
        }
    }
}
