/*
 * #%L
 * jTimer
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2007 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
/* *##%
 * Copyright (C) 2008, 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.io.Saver;
import org.chorem.jtimer.ws.ProjectManagement;

/**
 * JTimer config class.
 * 
 * @author chatellier
 * @version $Revision: 2518 $
 * 
 * Last update : $Date: 2009-05-22 13:17:51 +0200 (ven., 22 mai 2009) $
 * By : $Author: chatellier $
 */
public class JTimerFactory {

    /** Logger */
    private static Log log = LogFactory.getLog(JTimerFactory.class);

    /** Default idle time in milliseconds */
    protected static final long DEFAULT_IDLE_TIME = 5 * 60 * 1000; // 5 min

    /** Properties */
    protected static Properties props;

    /** Configured idle time */
    protected static long idleTime = -1L;

    /** ProjectManagement */
    protected static ProjectManagement projectManagement;

    /** Saver */
    protected static Saver saver;

    /**
     * Constructeur.
     */
    protected JTimerFactory() {
        // disable instanciation
    }

    /**
     * Init.
     */
    protected static void init() {

        // init once
        if (props == null) {

            // change load order here :
            // Take :
            // - classpath one
            // - user home one
            // - ./ one

            try {
                props = new Properties();

                // read configuration file
                String fileName = getConfigurationFileName();
                
                // use file in classpath
                URL url = JTimerFactory.class.getResource("/" + fileName);
                props.load(url.openStream());

                // read file in current directory
                File currentDirectoryFile = new File(fileName);
                if (currentDirectoryFile.canRead()) {
                    props.load(new FileInputStream(currentDirectoryFile));
                }

                // and override by file in user home
                File homeUserFile = new File(getUserHomeConfigurationFilePath());
                if (homeUserFile.exists()) {
                    if (log.isInfoEnabled()) {
                        log.info("Use user home configuration in "
                                + homeUserFile.getAbsolutePath());
                    }
                    props.load(new FileInputStream(homeUserFile));
                } else {

                    // homeUserFile is $HOME/JTimer.properties
                    // on linux, parent dir is created by SAF
                    // on other, SAF dir is not this one, so
                    // could not exists
                    if (homeUserFile.getParentFile() != null
                            && !homeUserFile.getParentFile().exists()) {
                        homeUserFile.getParentFile().mkdirs();
                        if (log.isInfoEnabled()) {
                            log.info("Creating directory "
                                    + homeUserFile.getParentFile()
                                            .getAbsolutePath());
                        }
                    }

                    if (log.isInfoEnabled()) {
                        log.info("Creating new default configuration file in "
                                + homeUserFile.getAbsolutePath());
                    }

                    URL defautFileURL = JTimerFactory.class.getResource("/"
                            + getDefaultConfigurationFileName());
                    copyConfigurationFile(defautFileURL, homeUserFile);
                }

            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't read configuration file", e);
                }
            }
        }
    }

    /**
     * Configuration filepath.
     * 
     * @return Configuration filename
     */
    protected static String getUserHomeConfigurationFilePath() {
        // user home
        String filePath = System.getProperty("user.home");
        // + .jtimer directory
        // TODO use saf config
        filePath += File.separator + ".jtimer";
        // add filename
        filePath += File.separator + getConfigurationFileName();

        if (log.isDebugEnabled()) {
            log.debug("Look for configuration file in : " + filePath);
        }

        return filePath;
    }

    /**
     * Configuration filename.
     * 
     * @return Configuration filename
     */
    protected static String getConfigurationFileName() {
        String fileName = JTimer.class.getSimpleName() + ".properties";
        return fileName;
    }

    /**
     * Default configuration filename.
     * 
     * @return Configuration filename
     */
    protected static String getDefaultConfigurationFileName() {
        String fileName = JTimer.class.getSimpleName() + "_default.properties";
        return fileName;
    }

    /**
     * Copy copyFrom content in copyTo file.
     * 
     * @param copyFrom from
     * @param copyTo to
     * @throws IOException 
     */
    protected static void copyConfigurationFile(URL copyFrom, File copyTo)
            throws IOException {

        InputStream in = copyFrom.openStream();
        OutputStream out = new FileOutputStream(copyTo);

        IOUtils.copy(in, out);

        in.close();
        out.close();
    }

    /**
     * Return an implementation on a distant web service.
     * 
     * Use jtimer.service.class property to found class and
     * jtimer.service.endpoint and jtimer.service.resource to init it.
     * 
     * @return implementation on a distant web service.
     */
    public static ProjectManagement getProjectManagementService() {

        if (projectManagement == null) {

            // check that init has been done
            init();

            // get implementation class
            String implementationClass = props
                    .getProperty("jtimer.service.class");
            // get service endpoint
            String serviceEndPoint = props
                    .getProperty("jtimer.service.endpoint");
            // get service resource name
            String serviceResouceName = props
                    .getProperty("jtimer.service.resource");

            // log
            if (log.isInfoEnabled()) {
                log.info("Using service class : " + implementationClass);
                log.info(" with service endpoint : " + serviceEndPoint);
                log.info(" with service resource : " + serviceResouceName);
            }

            // By default jTimer won't synchronize
            // so implementationClass can be null
            if (implementationClass != null) {
                try {
                    // get instance
                    projectManagement = (ProjectManagement) Class.forName(
                            implementationClass).newInstance();

                    // init instance
                    projectManagement.setEndpoint(serviceEndPoint);
                    projectManagement.setResourceName(serviceResouceName);
                } catch (InstantiationException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't instanciate class : "
                                + implementationClass, e);
                    }
                } catch (IllegalAccessException e) {
                    if (log.isErrorEnabled()) {
                        log.error(
                                "Can't access class : " + implementationClass,
                                e);
                    }
                } catch (ClassNotFoundException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't found class : " + implementationClass,
                                e);
                    }
                }
            } else {
                if (log.isInfoEnabled()) {
                    log.info("No sync information given, won't synchronize");
                }
            }
        }

        return projectManagement;
    }

    /**
     * Get saver manager.
     * 
     * @return saver manager
     */
    public static Saver getFileSaver() {

        if (saver == null) {
            // check that init has been done
            init();

            // get implementation class
            String implementationClass = props
                    .getProperty("jtimer.io.saver.class");
            // get saver directory
            String saverDirectory = props
                    .getProperty("jtimer.io.saver.directory");
            // get saver save delay
            String autoSaveDelay = props
                    .getProperty("jtimer.io.saver.autosavedelay");

            // replace $HOME by user.home
            saverDirectory = getHomeReplacement(saverDirectory);

            // log
            if (log.isInfoEnabled()) {
                log.info("Using saver class : " + implementationClass);
                log.info(" with saver home directory : " + saverDirectory);
                log.info(" with auto save delay : " + autoSaveDelay);
            }

            try {
                // get instance
                saver = (Saver) Class.forName(implementationClass)
                        .newInstance();

                // init instance
                saver.setSaveDirectory(saverDirectory);

                // set delay to saver
                try {
                    long autoSaveDelayInS = Long.parseLong(autoSaveDelay);
                    saver.setAutoSaveDelay(autoSaveDelayInS * 1000);
                } catch (NumberFormatException e) {
                    if (log.isWarnEnabled()) {
                        log.warn("jtimer.io.saver.autosavedelay is non numeric value",
                                        e);
                    }
                }
            } catch (InstantiationException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't instanciate class : "
                            + implementationClass, e);
                }
            } catch (IllegalAccessException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't access class : " + implementationClass, e);
                }
            } catch (ClassNotFoundException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't found class : " + implementationClass, e);
                }
            }
        }

        return saver;
    }

    /**
     * Replace $HOME in path.
     * 
     * @param path path to replace home
     * @return path without $HOME
     */
    protected static String getHomeReplacement(String path) {

        String localPath = path;

        // replace $HOME by user.home
        if (localPath != null) {
            // replace / in configuration file by
            // default system file.separator
            // FIXME find better method in 2.0
            localPath = localPath.replace("/", File.separator);

            // correct bug on windows system for example
            // \ in path are interpreted on regex
            String escapedUserHome = System.getProperty("user.home")
                    .replaceAll("\\\\", "\\\\\\\\");
            localPath = localPath.replaceFirst("\\$HOME", escapedUserHome);
        }

        return localPath;
    }

    /**
     * Get configuration idle time in ms.
     * 
     * @return idle time
     */
    public static long getIdleTime() {

        if (idleTime <= 0) {

            // check that init has been done
            init();

            String idleTimeInS = props.getProperty("jtimer.ui.idletime");
            if (idleTimeInS == null) {
                idleTime = DEFAULT_IDLE_TIME;
            } else {
                try {
                    idleTime = Long.parseLong(idleTimeInS) * 1000L;
                } catch (NumberFormatException e) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get idle time as number, using default idle time",
                                        e);
                    }
                    idleTime = DEFAULT_IDLE_TIME;
                }
            }

            if (log.isInfoEnabled()) {
                log.info("Idle time set to : " + idleTime + "ms");
            }
        }

        return idleTime;
    }
}
