/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ws.xmlrpc;

import java.net.ConnectException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xmlrpc.XmlRpcException;
import org.chorem.jtimer.data.DataViolationException;
import org.chorem.jtimer.data.TimerDataManager;
import org.chorem.jtimer.entities.TimerProject;
import org.chorem.jtimer.entities.TimerTask;
import org.chorem.jtimer.entities.TimerTaskHelper;
import org.chorem.jtimer.ws.ConnectionDataHandler;
import org.chorem.jtimer.ws.ProjectManagement;
import org.chorem.jtimer.ws.exception.WebServiceException;

/**
 * ChoremXMLRPCClient.
 * 
 * @author chatellier
 * @version $Revision: 2687 $
 * 
 * Last update : $Date: 2008-06-13 17:56:15 +0200 (ven., 13 juin 2008)$
 * By : $Author: echatellier $
 */
public class ChoremXMLRPCClient extends AbstractXMLRPCClient implements
        ProjectManagement {

    /** log. */
    private static Log log = LogFactory.getLog(ChoremXMLRPCClient.class);

    /** Endpoint. */
    protected String endpoint;

    /** Resource name. */
    protected String resourceName;

    /** Map des task vers les topiaId. */
    protected Map<List<String>, String> taskToTopiaId;

    /** Connection handler. */
    protected ConnectionDataHandler connectionDataHandler;

    /** Data manager */
    protected TimerDataManager dataManager;

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#setDataManager(org.chorem.jtimer.data.TimerDataManager)
     */
    @Override
    public void setDataManager(TimerDataManager dataManager) {
        this.dataManager = dataManager;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#setEndpoint(java.lang.String)
     */
    @Override
    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    /*
     * @see org.chorem.jtimer.ws.xmlrpc.AbstractXMLRPCClient#getEndpoint()
     */
    @Override
    public String getEndpoint() {
        return endpoint;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#getResourceName()
     */
    @Override
    public String getResourceName() {
        return resourceName;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#setResourceName(java.lang.String)
     */
    @Override
    public void setResourceName(String resourceName) {
        this.resourceName = resourceName;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#setConnectionDataHandler(org.chorem.jtimer.ws.ConnectionDataHandler)
     */
    @Override
    public void setConnectionDataHandler(
            ConnectionDataHandler connectionDataHandler) {
        this.connectionDataHandler = connectionDataHandler;
    }

    /**
     * Redefine get to prefix service name
     * 
     * @param serviceName service name
     * @param args service params
     * @return service result
     * @throws XmlRpcException when xml-rpc exception
     */
    protected Object get(String serviceName, Object... args)
            throws XmlRpcException {
        Object result = null;
        try {
            result = super.get(resourceName + "." + serviceName, args);
        } catch (XmlRpcException e) {
            if (e.getCause() instanceof ConnectException) {
                if (log.isTraceEnabled()) {
                    log.trace("Can't connect through xmlrpc to " + endpoint, e);
                } else if (log.isErrorEnabled()) {
                    log.error("Can't connect through xmlrpc to " + endpoint);
                }
            }
            throw e;
        }
        return result;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#getLoginAndIdsMap()
     */
    @Override
    public Map<String, String> getIdAndLoginsMap() throws WebServiceException {
        // method name
        final String methodName = "getEmployees";

        // ws call
        Map<String, String> idAndLogins = null;
        try {
            Object idAndLoginsObject = get(methodName);

            if (idAndLoginsObject instanceof Map) {
                idAndLogins = (Map<String, String>) idAndLoginsObject;
            }
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }

            throw new WebServiceException("Can't call " + methodName + "!", e);
        }

        return idAndLogins;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#login(java.lang.String, java.lang.String)
     */
    public String login(String login, String password)
            throws WebServiceException {

        // method name
        final String methodName = "login";

        // ws call
        String topiaId = null;
        try {
            Object topiaIdObject = get(methodName, login, password);

            if (topiaIdObject instanceof String) {
                topiaId = (String) topiaIdObject;
            }
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }

            throw new WebServiceException("Can't call " + methodName + "!", e);
        }

        return topiaId;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#getUserProjects(java.lang.String)
     */
    public List<TimerProject> getUserProjects(String userId)
            throws WebServiceException {
        // method name
        // this is called getTasks in chorem
        final String methodName = "getTasks";

        // map of timer name > timer project instance
        Map<String, TimerProject> result = new HashMap<String, TimerProject>();
        taskToTopiaId = new HashMap<List<String>, String>();

        try {
            Object oMapTopiaidTaskPath = get(methodName, userId);

            if (oMapTopiaidTaskPath instanceof Map) {

                Map<String, Object[]> mapResult = (Map<String, Object[]>) oMapTopiaidTaskPath;
                for (String topiaId : mapResult.keySet()) {

                    // ? can't get list<string> from ws ?
                    Object[] oTaskNames = mapResult.get(topiaId);
                    List<String> lTaskNames = new ArrayList<String>();
                    for (Object oTaskName : oTaskNames) {

                        // FIXME tmp bugfix, if remote task
                        // have / in their name
                        String taskName = (String) oTaskName;
                        taskName = taskName.replaceAll("/", "-");

                        lTaskNames.add(taskName);
                    }

                    if (log.isDebugEnabled()) {
                        log.debug(" task found (" + topiaId + ") : "
                                + lTaskNames);
                    }

                    // ask this task in tree
                    addTask(result, lTaskNames);

                    // remember assoc between task path and topiaIds
                    taskToTopiaId.put(lTaskNames, topiaId);
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Result found but not Map type : "
                            + oMapTopiaidTaskPath);
                }
            }

        } catch (XmlRpcException e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }

            throw new WebServiceException("Can't call " + methodName + "!", e);
        }

        return new ArrayList<TimerProject>(result.values());
    }

    /**
     * Add new task in current tree.
     * 
     * @param mapOfProjects map of already seen projects
     * @param taskNames task name path
     */
    protected void addTask(Map<String, TimerProject> mapOfProjects,
            List<String> taskNames) {

        // project name
        String projectName = taskNames.get(0);
        if (log.isDebugEnabled()) {
            log.debug("Manage project : " + projectName);
        }

        // found this project in current map
        TimerProject project = mapOfProjects.get(projectName);
        if (project == null) {
            project = new TimerProject();
            // prefix name to set it synchronized
            project.setName(TimerProject.SYNCHRONIZED_PROJECT_NAME_PREFIX
                    + projectName);

            // add creation timestamp
            project.setCreationDate(new Date());

            mapOfProjects.put(projectName, project);
        }

        // add it
        addTask(project, taskNames.subList(1, taskNames.size()));
    }

    /**
     * Add task in task subtask.
     * 
     * @param task task to add into
     * @param taskNames task names
     */
    protected void addTask(TimerTask task, List<String> taskNames) {

        if (taskNames.size() > 0) {

            String taskName = taskNames.get(0);

            TimerTask currentTask = null;
            for (TimerTask subtask : task.getSubTasks()) {
                if (subtask.getName().equals(taskName)) {
                    currentTask = subtask;
                }
            }

            // task doesn't exist
            if (currentTask == null) {
                currentTask = new TimerTask();
                currentTask.setName(taskName);

                // Fix creation date
                currentTask.setCreationDate(new Date());

                task.addTask(currentTask);
            }

            addTask(currentTask, taskNames.subList(1, taskNames.size()));
        }

    }

    /**
     * Get last update date from remote server.
     * 
     * @param userId user id
     * @return date of last update or <tt>null</tt>
     */
    protected Date getLastUpdate(String userId) {

        // method name
        final String methodName = "getLastUpdate";

        // date
        Date result = null;

        try {
            Object tmpResult = get(methodName, userId);

            if (tmpResult instanceof Date) {

                result = (Date) tmpResult;

                if (log.isDebugEnabled()) {
                    log.debug("getLastUpdate() = " + result);
                }
            }

        } catch (XmlRpcException e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }
        }

        return result;
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#synchronize(java.lang.String, java.util.List)
     */
    @Override
    public void synchronize(String userId, List<TimerProject> projects)
            throws WebServiceException {

        // get last update date
        Date lastUpdateDate = getLastUpdate(userId);

        // if date is null, take 1st january 1970 to force init
        if (lastUpdateDate == null) {
            lastUpdateDate = new Date(0);
        }

        for (TimerProject project : projects) {
            if (project.isSynchronized()) {

                // get project name without synchonized prefix name
                String nonSynchonizedProjectName = project.getName().substring(
                        TimerProject.SYNCHRONIZED_PROJECT_NAME_PREFIX.length());
                List<String> pathNames = new ArrayList<String>(1);
                pathNames.add(nonSynchonizedProjectName);

                // call recursive task sync
                synchronizeTask(userId, project, lastUpdateDate, pathNames);
            }
        }
    }

    /**
     * Synchronize a single task.
     * 
     * @param userId user id
     * @param task task to sync
     * @param lastUpdateDate last update date (can be null)
     * @param currentTaskPath current task path
     * @throws WebServiceException when call fail
     */
    protected void synchronizeTask(String userId, TimerTask task,
            Date lastUpdateDate, List<String> currentTaskPath)
            throws WebServiceException {

        // log
        if (log.isDebugEnabled()) {
            log.debug("Manage task path : " + currentTaskPath);
        }

        Date firstDate = TimerTaskHelper.getTaskFirstDateOfTiming(task);
        Date lastDate = TimerTaskHelper.getTaskLastDateOfTiming(task);

        // if task has timing
        if (firstDate != null && lastDate != null) {

            Calendar firstCalendar = Calendar.getInstance();
            firstCalendar.setTime(firstDate);
            Calendar lastCalendar = Calendar.getInstance();
            firstCalendar.setTime(lastDate);

            Calendar lastUpdateCalendar = Calendar.getInstance();
            lastUpdateCalendar.setTimeInMillis(lastUpdateDate.getTime());
            lastUpdateCalendar.add(Calendar.DAY_OF_YEAR, 1);

            Calendar yesterdayCalendar = Calendar.getInstance();
            yesterdayCalendar.add(Calendar.DAY_OF_YEAR, -1);

            Calendar fromDay = firstCalendar.compareTo(lastUpdateCalendar) < 0 ? lastUpdateCalendar
                    : firstCalendar;
            Calendar toDay = lastCalendar.compareTo(yesterdayCalendar) < 0 ? lastCalendar
                    : yesterdayCalendar;

            // define date interval
            if (log.isDebugEnabled()) {
                log.debug(" Sync task time : " + task.getName());
                log.debug("  from " + fromDay.get(Calendar.DAY_OF_MONTH) + "/"
                        + (fromDay.get(Calendar.MONTH) + 1) + "/"
                        + fromDay.get(Calendar.YEAR));
                log.debug("  to " + toDay.get(Calendar.DAY_OF_MONTH) + "/"
                        + (toDay.get(Calendar.MONTH) + 1) + "/"
                        + toDay.get(Calendar.YEAR));
            }

            // call syncTask on each day
            for (; fromDay.compareTo(toDay) <= 0; fromDay.add(
                    Calendar.DAY_OF_YEAR, 1)) {

                long timeOfCurrentDay = task.getTime(fromDay.getTime());
                if (log.isDebugEnabled()) {
                    log.debug("time of day "
                            + fromDay.get(Calendar.DAY_OF_MONTH) + "/"
                            + (fromDay.get(Calendar.MONTH) + 1) + "/"
                            + fromDay.get(Calendar.YEAR) + " = "
                            + timeOfCurrentDay + " (calendar = " + fromDay
                            + ")");
                }
                if (timeOfCurrentDay > 0) {
                    String topiaId = taskToTopiaId.get(currentTaskPath);

                    if (topiaId == null) {
                        String superTopiaId = taskToTopiaId.get(currentTaskPath
                                .subList(0, currentTaskPath.size() - 1));
                        topiaId = addTask(userId, superTopiaId, task.getName());

                        // save topiaId of added task
                        if (topiaId != null) {
                            taskToTopiaId.put(currentTaskPath, topiaId);
                        }

                        // log
                        if (log.isDebugEnabled()) {
                            log.debug("Add task : " + currentTaskPath
                                    + " topiaId = " + topiaId);
                        }
                    }

                    if (topiaId != null) {
                        if (log.isDebugEnabled()) {
                            log.debug(" sending time for  " + topiaId + " = "
                                    + timeOfCurrentDay);
                        }
                        syncTask(userId, topiaId, fromDay.getTime(),
                                timeOfCurrentDay * 1000);
                    } else {
                        if (log.isFatalEnabled()) {
                            log.fatal("Can't get topiaId for task : "
                                    + task.getName());
                            log.fatal("Chorem error ?");
                        }
                    }
                }
            }
        } else {
            // task has no timing, but add it on remote SI
            // not add all task (only if has subtasks)
            if (task.getSubTasks() != null && !task.getSubTasks().isEmpty()) {
                String topiaId = taskToTopiaId.get(currentTaskPath);

                if (topiaId == null) {
                    String superTopiaId = taskToTopiaId.get(currentTaskPath
                            .subList(0, currentTaskPath.size() - 1));
                    topiaId = addTask(userId, superTopiaId, task.getName());

                    // save topiaId of added task
                    if (topiaId != null) {
                        taskToTopiaId.put(currentTaskPath, topiaId);
                    }

                    // log
                    if (log.isDebugEnabled()) {
                        log.debug("Add task : " + currentTaskPath
                                + " topiaId = " + topiaId);
                    }
                }
            }
        }

        // make recursive call on subtask
        for (TimerTask subTask : task.getSubTasks()) {

            // list path for childreen
            List<String> childTaskPath = new ArrayList<String>(currentTaskPath);
            childTaskPath.add(subTask.getName());

            synchronizeTask(userId, subTask, lastUpdateDate, childTaskPath);
        }

    }

    /**
     * Add task on remote server.
     * 
     * @param userId user id
     * @param superTopiaId parent task topiaId
     * @param name name of task to add
     * @return topiaId of created task
     */
    protected String addTask(String userId, String superTopiaId, String name) {

        // method name
        final String methodName = "addTask";

        // topiaId of added task
        String topiaId = null;

        try {
            Object oTopiaId = get(methodName, userId, superTopiaId, name);

            if (oTopiaId instanceof String) {

                topiaId = (String) oTopiaId;

                if (log.isDebugEnabled()) {
                    log.debug(methodName + "() = " + topiaId);
                }
            }

        } catch (XmlRpcException e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }
        }

        return topiaId;
    }

    /**
     * Sync a task, call xmlrpc.
     * 
     * @param userId user id
     * @param topiaId task topiaId
     * @param dateOfDay date
     * @param timeOfDay time in seconds
     * @throws WebServiceException when call fail
     */
    protected void syncTask(String userId, String topiaId, Date dateOfDay,
            long timeOfDay) throws WebServiceException {

        // method name
        final String methodName = "syncTask";

        try {
            // debug
            if (log.isDebugEnabled()) {
                log.debug(methodName + "(" + userId + ", " + topiaId + ", "
                        + dateOfDay + ", " + timeOfDay + ")");
            }

            get(methodName, userId, topiaId, dateOfDay, (int) timeOfDay);

        } catch (XmlRpcException e) {
            if (log.isDebugEnabled()) {
                log.debug("Exception on xml-rpc service call (" + methodName
                        + ")", e);
            }

            throw new WebServiceException("Can't call " + methodName + "!", e);
        }
    }

    /*
     * @see org.chorem.jtimer.ws.ProjectManagement#syncTask(java.lang.String, org.chorem.jtimer.entities.TimerTask, java.util.Date, long)
     */
    @Override
    public void syncTask(String userId, TimerTask task, Date dateOfDay,
            long timeOfDay) throws WebServiceException {

        if (task != null
                && TimerTaskHelper.getTaskProject(task).isSynchronized()) {

            // init task to topiaId association
            if (taskToTopiaId == null) {
                getUserProjects(userId);
            }

            // build task path
            List<String> taskPath = TimerTaskHelper.getTaskPath(task);

            // search for topiaId
            String topiaId = taskToTopiaId.get(taskPath);

            if (topiaId != null) {
                // sync task
                syncTask(userId, topiaId, dateOfDay, timeOfDay);

            } else {
                // iterate over all parent tasks and add them if needed
                // toIndex is exclusive
                for (int toIndex = 1; toIndex < taskPath.size(); toIndex++) {
                    String superTopiaId = taskToTopiaId.get(taskPath.subList(0,
                            toIndex));

                    if (superTopiaId != null) {
                        topiaId = taskToTopiaId.get(taskPath.subList(0,
                                toIndex + 1));

                        if (topiaId == null) {
                            topiaId = addTask(userId, superTopiaId, taskPath
                                    .get(toIndex));

                            // save topiaId of added task
                            if (topiaId != null) {
                                taskToTopiaId.put(taskPath.subList(0,
                                        toIndex + 1), topiaId);
                            } else {
                                // add task fail ?
                                if (log.isWarnEnabled()) {
                                    log.warn("Add new task failed on chorem");
                                }
                            }
                        }
                    }
                }

                // task should have been previously added 
                topiaId = taskToTopiaId.get(taskPath);
                if (topiaId != null) {
                    syncTask(userId, topiaId, dateOfDay, timeOfDay);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Try to add time on an unknown synchronized task");
                    }
                }
            }
        }
    }

    /**
     * Synchronization (from server).
     */
    protected void synchronization() {
        // log
        if (log.isInfoEnabled()) {
            log.info("Synchronization");
        }

        WSDaemon webServicesDeamon = new WSDaemon(this, this.dataManager);
        webServicesDeamon.setConnectionDataHandler(connectionDataHandler);

        Timer timer = new Timer();

        Calendar date = Calendar.getInstance();
        date.set(Calendar.HOUR_OF_DAY, 0);
        date.set(Calendar.MINUTE, 0);
        date.set(Calendar.SECOND, 0);
        date.set(Calendar.MILLISECOND, 0); // start one timer at startup

        // Schedule to run every day in midnight
        // task,firstTime,period
        timer.schedule(webServicesDeamon, date.getTime(), // at date
                1000 * 60 * 60 * 24 // every day
                );
    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#addProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void addProject(TimerProject project) {
    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#addTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void addTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#dataLoaded(java.util.Collection)
     */
    @Override
    public void dataLoaded(Collection<TimerProject> projects) {
        synchronization();
    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preDeleteProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void preDeleteProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preDeleteTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preDeleteTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#deleteProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void deleteProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#deleteTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void deleteTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#modifyProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void modifyProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#modifyTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void modifyTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#preChangeClosedState(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preChangeClosedState(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#postChangeClosedState(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void postChangeClosedState(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#postMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void moveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#postMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void postMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void preMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#preMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preMoveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#startTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void startTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#stopTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void stopTask(TimerTask task) {

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkSetAnnotation(TimerTask task, Date date, String annotation) {

        TimerProject project = TimerTaskHelper.getTaskProject(task);

        // TODO allow for today
        if (project.isSynchronized()) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkSetAnnotation won't pass");
            }
            throw new DataViolationException("Can't set annotation",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setAnnotation(TimerTask task, Date date, String annotation) {
        if (task != null
                && TimerTaskHelper.getTaskProject(task).isSynchronized()) {

            // TODO: assert that calendar date is before today

            if (log.isDebugEnabled()) {
                log.debug(task.getName() + " (" + date + ": annotation="
                        + annotation + ")");
            }

            // not implemented so raise no error
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkSetTaskTime(TimerTask task, Date date, Long value) {

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setTaskTime(TimerTask task, Date date, Long time) {

        if (task != null
                && TimerTaskHelper.getTaskProject(task).isSynchronized()) {

            // check if day is passed
            // in this case, sync
            Calendar calendar = Calendar.getInstance();
            calendar.set(Calendar.HOUR_OF_DAY, 0);
            calendar.set(Calendar.MINUTE, 0);
            calendar.set(Calendar.SECOND, 0);
            calendar.set(Calendar.MILLISECOND, 0);

            if (calendar.getTime().before(date)) {
                if (log.isDebugEnabled()) {
                    log.debug("Today modification, can't synchronize");
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Sync task " + task.getName());
                    log.debug(" date = " + date);
                    log.debug(" value =" + time);
                }

                try {
                    String login = connectionDataHandler
                            .getConnectionInformation(this).getLogin();
                    if (login != null) {
                        syncTask(login, task, date, time * 1000);
                    } else {
                        // login not found so raise no error
                    }
                } catch (WebServiceException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * Synchronized projects are added by synchronization.
     */
    @Override
    public void checkAddProject(TimerProject project) {

    }

    /**
     * {@inheritDoc}
     * 
     * can add, even in sync projects
     */
    @Override
    public void checkAddTask(TimerTask parent, TimerTask task) {

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkChangeClosedState(TimerTask task) {

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkDeleteProject(TimerProject project) {

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkDeleteTask(TimerTask task) {

        TimerProject project = TimerTaskHelper.getTaskProject(task);

        if (project.isSynchronized()) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkDeleteTask won't pass");
            }
            throw new DataViolationException("Can't delete task projet",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkModifyProject(TimerProject project) {

        if (project.isSynchronized()) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkModifyProject won't pass");
            }
            throw new DataViolationException("Can't modify projet",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkModifyTask(TimerTask task) {

        TimerProject project = TimerTaskHelper.getTaskProject(task);

        if (project.isSynchronized()) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkModifyTask won't pass");
            }
            throw new DataViolationException("Can't modify task",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkMoveTask(TimerTask destination, Collection<TimerTask> tasksToMove) {

        boolean syncProject = false;
        // destination (single)
        TimerProject project1 = TimerTaskHelper.getTaskProject(destination);
        if (project1.isSynchronized()) {
            syncProject = true;
        }

        // sources (multiples)
        for (TimerTask taskToMove : tasksToMove) {
            TimerProject project2 = TimerTaskHelper.getTaskProject(taskToMove);
    
            if (project2.isSynchronized()) {
                syncProject = true;
            }
        }

        if (syncProject) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkMoveTask won't pass");
            }
            throw new DataViolationException("Can't modify task",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void checkMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

        TimerProject project1 = TimerTaskHelper.getTaskProject(destinationTask);

        if (project1.isSynchronized()) {
            if (log.isDebugEnabled()) {
                log.debug("Project is synchronized, checkMergeTask won't pass");
            }
            throw new DataViolationException("Can't modify task",
                    "vetoable.ws.chorem.cant.modify.synchronized.project");
        }

        for (TimerTask otherTask : otherTasks) {
            TimerProject otherProject = TimerTaskHelper
                    .getTaskProject(otherTask);
            if (otherProject.isSynchronized()) {
                if (log.isDebugEnabled()) {
                    log
                            .debug("Project is synchronized, checkMergeTask won't pass");
                }
                throw new DataViolationException("Can't modify task",
                        "vetoable.ws.chorem.cant.modify.synchronized.project");
            }
        }
    }
}
