/* *##%
 * Copyright (C) 2009 Code Lutin, Chatellier Eric
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.tasks;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.Timer;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;

import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.JTimerFactory;
import org.jdesktop.application.Action;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceManager;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.SingleFrameApplication;

/**
 * Modal idle dialog showed to user when idle has been detected.
 * 
 * Composed of an unique blocking show method.
 * 
 * Also composed of three resume option :
 * <ul>
 *  <li>Stop task
 *  <li>Continue (with idle time summed)
 *  <li>Resume (without idle time summed)
 * </ul>
 * 
 * @author chatellier
 * @version $Revision: 2694 $
 * 
 * Last update : $Date: 2009-11-16 10:21:54 +0100 (lun., 16 nov. 2009) $
 * By : $Author$
 */
public class IdleDialog extends JDialog {

    /** serialVersionUID. */
    private static final long serialVersionUID = 7669429291708466753L;

    /** log */
    private static Log log = LogFactory.getLog(IdleDialog.class);

    /** Mutex object (multiples running tasks are waiting on it) */
    protected static Object mutex = new Object();

    /** Singleton dialog instance. */
    protected static IdleDialog idleDialog;

    /** Resume option. */
    protected static int lastResumeOption;

    /** Parent application. */
    protected SingleFrameApplication application;

    /** I18N resource map. */
    protected ResourceMap resourceMap;

    /** Timer (for idle duration scheduling). */
    protected Timer timer;

    /** Current Idle duration refresh task. */
    protected UpdateIdleTime updateIdleTime;

    /** Timestamp when idle starts. */
    protected long idleStartTimestamp;

    /** Duration label. */
    protected JLabel idleDurationLabel;

    /** Revert option after idle detect. */
    public static final int REVERT = 0;

    /** Continue option after idle detect. */
    public static final int CONTINUE = 1;

    /** Resume option after idle detect. */
    public static final int RESUME = 2;

    /**
     * IdleDialog constructor.
     * 
     * Protected to force use of show static method.
     * 
     * @param parent parent application
     */
    protected IdleDialog(SingleFrameApplication application) {
        // don't make reference on other parent
        // windows, cause idle to unlock some task
        // if parent window is hidden by systray
        super();

        // init resources map
        this.application = application;
        ApplicationContext ctxt = application.getContext();
        ResourceManager mgr = ctxt.getResourceManager();
        resourceMap = mgr.getResourceMap(IdleDialog.class);

        setName("idleFrame");
        setTitle(resourceMap.getString("idleTitle"));
        setResizable(false);
        setModal(true);
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

        getRootPane().setLayout(new BorderLayout(1, 1));
        getRootPane().add(getMainComponent(), BorderLayout.CENTER);

        // timer
        timer = new Timer();
    }

    /**
     * Build main component UI.
     * 
     * @return component ui.
     */
    private JComponent getMainComponent() {
        JPanel mainComponent = new JPanel(new GridBagLayout());

        // label
        JLabel idleIcon = new JLabel(resourceMap.getIcon("idleIcon"));
        mainComponent.add(idleIcon, new GridBagConstraints(0, 0, 1, 5, 0, 1,
                GridBagConstraints.NORTH, GridBagConstraints.NONE, new Insets(
                        10, 5, 10, 10), 0, 0));

        // label
        JLabel idleLabel = new JLabel(resourceMap.getString("idleMessage", Long
                .valueOf(JTimerFactory.getIdleTime() / (60 * 1000))));
        mainComponent.add(idleLabel, new GridBagConstraints(1, 0, 3, 1, 1, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(5, 0, 0, 3), 0, 0));

        idleDurationLabel = new JLabel(" ");
        mainComponent.add(idleDurationLabel, new GridBagConstraints(1, 1, 3, 1,
                1, 0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(5, 0, 3, 5), 0, 0));

        // separator
        mainComponent.add(new JSeparator(), new GridBagConstraints(1, 2, 3, 1,
                1, 0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(0, 0, 0, 5), 0, 0));

        // label
        JLabel idleRestart = new JLabel(resourceMap.getString("idleRestart"));
        mainComponent.add(idleRestart, new GridBagConstraints(1, 3, 3, 1, 1, 0,
                GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(3, 0, 0, 5), 0, 0));

        JButton revertButton = new JButton();
        revertButton.setHorizontalAlignment(SwingConstants.LEFT);
        revertButton.setAction(application.getContext().getActionMap(this).get(
                "chooseRevertOption"));
        mainComponent.add(revertButton, new GridBagConstraints(1, 4, 1, 1, 1,
                0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(3, 0, 5, 5), 0, 0));

        JButton continueButton = new JButton();
        continueButton.setHorizontalAlignment(SwingConstants.LEFT);
        continueButton.setAction(application.getContext().getActionMap(this)
                .get("chooseContinueOption"));
        mainComponent.add(continueButton, new GridBagConstraints(2, 4, 1, 1, 1,
                0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(3, 0, 5, 5), 0, 0));

        JButton resumeButton = new JButton();
        resumeButton.setHorizontalAlignment(SwingConstants.LEFT);
        resumeButton.setAction(application.getContext().getActionMap(this).get(
                "chooseResumeOption"));
        mainComponent.add(resumeButton, new GridBagConstraints(3, 4, 1, 1, 1,
                0, GridBagConstraints.CENTER, GridBagConstraints.HORIZONTAL,
                new Insets(3, 0, 5, 5), 0, 0));

        return mainComponent;
    }

    /**
     * Init dialog idleDialog instance.
     * 
     * @param parent parent reference
     */
    public static void init(SingleFrameApplication parent) {
        idleDialog = new IdleDialog(parent);
    }

    /**
     * Revert button action.
     */
    @Action
    public void chooseRevertOption() {
        lastResumeOption = REVERT;
        idleEnded();
    }

    /**
     * Continue button action.
     */
    @Action
    public void chooseContinueOption() {
        lastResumeOption = CONTINUE;
        idleEnded();
    }

    /**
     * Resume button action.
     */
    @Action
    public void chooseResumeOption() {
        lastResumeOption = RESUME;
        idleEnded();
    }

    /**
     * Unblock all waiting threads on {@link #mutex}.
     */
    protected void idleEnded() {
        synchronized (mutex) {
            mutex.notifyAll();
        }
        setVisible(false);
    }

    @Override
    public void setVisible(boolean b) {

        if (b) {
            updateIdleTime = new UpdateIdleTime();
            timer.schedule(updateIdleTime, 0, 1000 * 60 /* every minutes */);
        } else {
            updateIdleTime.cancel();
            timer.purge();
        }

        super.setVisible(b);
    }

    /**
     * Called function on idle detect.
     * 
     * This function is thread safe and show only one dialog even if
     * function is called multiples times.
     *
     * @param idleStartTimestamp time stamp when idle start
     * @return idle option
     */
    public static int showIdleDialog(long idleStartTimestamp) {

        // only the first call must display dialog
        // (can happen if multiples task are running)
        boolean mustShow = false;
        synchronized (idleDialog) {
            mustShow = !idleDialog.isVisible();
        }

        // the first frame will be blocked by show modal
        // blocking status...
        if (mustShow) {
            idleDialog.idleStartTimestamp = idleStartTimestamp;
            idleDialog.application.show(idleDialog);
        } else {
            // ...the others by wait();
            synchronized (mutex) {
                try {
                    mutex.wait();
                } catch (InterruptedException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Thread inturrupted", e);
                    }
                }
            }
        }

        return lastResumeOption;
    }

    /**
     * Task to update idle time duration while dialog is showed.
     */
    protected class UpdateIdleTime extends java.util.TimerTask {

        /*
         * @see java.util.TimerTask#run()
         */
        @Override
        public void run() {
            if (log.isDebugEnabled()) {
                log.debug("Update idle duration");
            }
            String duration = DurationFormatUtils.formatDuration(
                    System.currentTimeMillis() - idleStartTimestamp, "HH:mm");
            idleDurationLabel.setText(resourceMap.getString("idleDuration", duration));
        }
    }
}
