/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.data;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.entities.TimerAlert;
import org.chorem.jtimer.entities.TimerProject;
import org.chorem.jtimer.entities.TimerTask;

/**
 * Gere les donnees. Des objets peuvent s'enregistrer pour etre notifies des
 * changements de donnees.
 * 
 * @author chatellier
 * @version $Revision: 2687 $
 * 
 * Last update : $Date: 2008-06-19 10:17:31 +0200 (jeu., 19 juin 2008)$
 * By : $Author: echatellier $
 */
public class TimerDataManager {

    /** log. */
    private static Log log = LogFactory.getLog(TimerDataManager.class);
    
    /** Project list. */
    protected List<TimerProject> projectList;

    /** For change notification */
    protected Collection<DataEventListener> dataEventListeners;

    /** For change notification */
    protected Collection<VetoableDataEventListener> vetoableDataEventListeners;

    /**
     * Constructor.
     */
    public TimerDataManager() {

        // init data list
        projectList = new ArrayList<TimerProject>();

        // init support list
        dataEventListeners = new ArrayList<DataEventListener>();
        vetoableDataEventListeners = new ArrayList<VetoableDataEventListener>();
    }

    /**
     * Add listener.
     * 
     * @param listener listener
     */
    public synchronized void addDataEventListener(DataEventListener listener) {
        dataEventListeners.add(listener);
    }

    /**
     * Remove listener.
     * 
     * @param listener listener
     */
    public synchronized void removeDataEventListener(DataEventListener listener) {
        dataEventListeners.remove(listener);
    }

    /**
     * Add vetoable listener.
     * 
     * @param listener listener
     */
    public synchronized void addVetoableDataEventListener(
            VetoableDataEventListener listener) {
        vetoableDataEventListeners.add(listener);
    }

    /**
     * Remove vetoable listener.
     * 
     * @param listener listener
     */
    public synchronized void removeVetoableDataEventListener(
            VetoableDataEventListener listener) {
        vetoableDataEventListeners.remove(listener);
    }

    /**
     * Add single project.
     * 
     * @param project a project
     */
    public synchronized void addProject(TimerProject project) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkAddProject(project);
        }

        projectList.add(project);

        // fire data event
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().addProject(project);
        }
    }

    /**
     * Add single task.
     * 
     * @param parent parent task
     * @param task task to add
     */
    public synchronized void addTask(TimerTask parent, TimerTask task) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkAddTask(parent, task);
        }

        parent.addTask(task);

        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().addTask(task);
        }
    }

    /**
     * Add many projects.
     * 
     * @param projects project collection
     */
    public synchronized void addAllProjects(Collection<TimerProject> projects) {
        if (projects != null) {
            projectList.clear();
            projectList.addAll(projects);

            // send notification
            Iterator<DataEventListener> itDataEventListener = dataEventListeners
                    .iterator();
            while (itDataEventListener.hasNext()) {
                itDataEventListener.next().dataLoaded(projects);
            }
        }
    }

    /**
     * Get projects list.
     * 
     * synchronized to prevent manipulation during save.
     * 
     * @return list of projects
     */
    public synchronized List<TimerProject> getProjectsList() {
        return projectList;
    }

    /**
     * Change time for the given date.
     * 
     * @param task the task to change time
     * @param date date to change time
     * @param value new time in seconds
     */
    public synchronized void changeTaskTime(TimerTask task, Date date,
            long value) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkSetTaskTime(task, date, value);
        }

        task.setTime(date, Long.valueOf(value));

        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().setTaskTime(task, date, value);
        }
    }

    /**
     * Start task.
     * 
     * @param task task to start
     */
    public synchronized void startTask(TimerTask task) {
        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().startTask(task);
        }
    }

    /**
     * Stop task.
     * 
     * @param task task to stop
     */
    public synchronized void stopTask(TimerTask task) {
        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().stopTask(task);
        }
    }

    /**
     * Delete task.
     * 
     * @param task task to delete
     */
    public synchronized void deleteTask(TimerTask task) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkDeleteTask(task);
        }

        // send notification (before)
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().preDeleteTask(task);
        }

        // task deletion
        task.getParent().getSubTasks().remove(task);

        // send notification
        itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().deleteTask(task);
        }
    }

    /**
     * Delete project.
     * 
     * @param project project to delete
     */
    public synchronized void deleteProject(TimerProject project) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkDeleteProject(project);
        }

        // send notification (before)
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().preDeleteProject(project);
        }
        
        projectList.remove(project);

        // send notification
        itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().deleteProject(project);
        }
    }

    /**
     * Close project.
     * 
     * @param project project to close
     */
    public synchronized void changeProjectCloseState(TimerProject project) {

        // fire vetoable event
        //Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
        //        .iterator();
        //while (itVetoableDataEventListener.hasNext()) {
        //    itVetoableDataEventListener.next().checkChangeClosedState(project);
        //}

        //// send notification
        //Iterator<DataEventListener> itDataEventListener = dataEventListeners
        //       .iterator();
        //while (itDataEventListener.hasNext()) {
        //    itDataEventListener.next().preChangeClosedState(project);
        //}

        //project.setClosed(!project.isClosed());

        //// send notification
        //itDataEventListener = dataEventListeners
        //       .iterator();
        //while (itDataEventListener.hasNext()) {
        //    itDataEventListener.next().postChangeClosedState(project);
        //}

        changeTaskCloseState(project);
    }

    /**
     * Close task.
     * 
     * @param task task to close
     */
    public synchronized void changeTaskCloseState(TimerTask task) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkChangeClosedState(task);
        }

        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().preChangeClosedState(task);
        }

        task.setClosed(!task.isClosed());

        // send notification
        itDataEventListener = dataEventListeners.iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().postChangeClosedState(task);
        }
    }

    /**
     * Edit project.
     * 
     * @param project project to edit
     * @param newProjectName new project name
     */
    public synchronized void editProject(TimerProject project,
            String newProjectName) {

        // fire vetoable event
        TimerProject newProject = project.clone();
        newProject.setName(newProjectName);
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkModifyProject(newProject);
        }

        project.setName(newProjectName);

        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().modifyProject(project);
        }
    }

    /**
     * Edit task.
     * 
     * @param task task to edit
     * @param newTaskName new task name
     */
    public synchronized void editTask(TimerTask task, String newTaskName) {

        // fire vetoable event
        TimerTask newTask = task.clone();
        newTask.setName(newTaskName);
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkChangeClosedState(newTask);
        }

        task.setName(newTaskName);

        // send notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().modifyTask(task);
        }
    }

    /**
     * Move task.
     * 
     * @param destination task to move to
     * @param tasksToMove tasks to move
     */
    public synchronized void moveTask(TimerTask destination, Collection<TimerTask> tasksToMove) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkMoveTask(destination, tasksToMove);
        }

        for (TimerTask taskToMove : tasksToMove) {

            // send notification (pre)
            Iterator<DataEventListener> itDataEventListener = dataEventListeners
                    .iterator();
            while (itDataEventListener.hasNext()) {
                itDataEventListener.next().preMoveTask(taskToMove);
            }

            // move task
            TimerTask actualParent = taskToMove.getParent();
            actualParent.getSubTasks().remove(taskToMove);
            destination.addTask(taskToMove);

            // send notification (post)
            itDataEventListener = dataEventListeners.iterator();
            while (itDataEventListener.hasNext()) {
                itDataEventListener.next().moveTask(taskToMove);
            }
        }
    }

    /**
     * Merge tasks.
     * 
     * @param destinationTask task where task will be merged
     * @param otherTasks task to merge in first task
     */
    public synchronized void mergeTasks(TimerTask destinationTask, List<TimerTask> otherTasks) {
        
        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkMergeTasks(destinationTask, otherTasks);
        }
        
        // send pre notification
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().preMergeTasks(destinationTask, otherTasks);
        }
        
        for (TimerTask otherTask : otherTasks) {
            mergeTwoTasks(destinationTask, otherTask);
        }
        
        // send post notification
        itDataEventListener = dataEventListeners.iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().postMergeTasks(destinationTask, otherTasks);
        }
    }
    
    /**
     * Merge two task together.
     * 
     * @param destinationTask task where task will be merged
     * @param otherTask task to merge in first task
     */
    protected void mergeTwoTasks(TimerTask destinationTask, TimerTask otherTask) {

        if (log.isDebugEnabled()) {
            log.debug("Merging two task : " + destinationTask.getName() + " and " + otherTask.getName());
        }

        // task is modified during merge, deep clone it
        
        // make sub task list copy (concurrency)
        Collection<TimerTask> otherTaskSubTasks = new ArrayList<TimerTask>(otherTask.getSubTasks());
        for (TimerTask otherTaskSubTask : otherTaskSubTasks) {

            // first case to care about, a task with same
            // same already exists in destination
            TimerTask sameTaskNameTask = null;
            for (TimerTask destSubTask : destinationTask.getSubTasks()) {
                if (destSubTask.getName().equals(otherTaskSubTask.getName())) {
                    sameTaskNameTask = destSubTask;
                }
            }
            
            // no similar task found
            if (sameTaskNameTask == null) {
                if (log.isDebugEnabled()) {
                    log.debug("Moving task " + otherTaskSubTask.getName() + " to " + destinationTask.getName());
                }
                // just move
                moveTask(destinationTask, Collections.singleton(otherTaskSubTask));
            }
            else {
                // task must be merged
                if (log.isDebugEnabled()) {
                    log.debug("Sub-merging of " + sameTaskNameTask.getName() + " and " + otherTaskSubTask.getName());
                }
                
                // recursive merge
                mergeTwoTasks(sameTaskNameTask, otherTaskSubTask);
                
                // TODO possible bug here, task times may not be saved
            }
        }
        
        // copy otherTask times to current task
        for (Entry<Date, Long> times : otherTask.getAllDaysAndTimes().entrySet()) {
            Long currentDuration = destinationTask.getTime(times.getKey());
            currentDuration += times.getValue();
            destinationTask.setTime(times.getKey(), currentDuration);
        }
        
        // copy annotations
        for (Entry<Date, String> note : otherTask.getAllDaysAnnotations().entrySet()) {            
            Date noteKey = note.getKey();
            while (destinationTask.getAllDaysAnnotations().containsKey(noteKey)) {
                // les deux taches ont des notes au meme moments
                // on ne deplace à la prochaine seconde
                
                if (log.isDebugEnabled()) {
                    log.debug("Annotation collision detected, try next second");
                }
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(noteKey);
                calendar.add(Calendar.SECOND, 1);
                noteKey = calendar.getTime();
            }
            destinationTask.addAnnotation(noteKey, note.getValue());
        }
        
        // copy alerts
        for (TimerAlert alert : otherTask.getAlerts()) {
            destinationTask.addAlert(alert);
        }
        
        // finally otherTask still exist
        // empty, but still exist
        deleteTask(otherTask);
    }
    
    /**
     * Add annotation on task for specified calendar, and send event.
     * 
     * @param task task
     * @param date day of annotation
     * @param annotation annotation
     */
    public synchronized void addAnnotation(TimerTask task, Date date,
            String annotation) {

        // fire vetoable event
        Iterator<VetoableDataEventListener> itVetoableDataEventListener = vetoableDataEventListeners
                .iterator();
        while (itVetoableDataEventListener.hasNext()) {
            itVetoableDataEventListener.next().checkSetAnnotation(task, date,
                    annotation);
        }

        task.addAnnotation(date, annotation);

        // send event
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().setAnnotation(task, date, annotation);
        }
    }

    /**
     * Notify alert modification.
     * 
     * @param task task where alert has been modified
     */
    public synchronized void modifyAlert(TimerTask task) {

        // send event
        Iterator<DataEventListener> itDataEventListener = dataEventListeners
                .iterator();
        while (itDataEventListener.hasNext()) {
            itDataEventListener.next().modifyTask(task);
        }

    }
}
