/* *##%
 * Copyright (C) 2008, 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.treetable.dnd;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.TransferHandler;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.data.DataViolationException;
import org.chorem.jtimer.data.TimerDataManager;
import org.chorem.jtimer.entities.TimerTask;
import org.chorem.jtimer.ui.treetable.ProjectsAndTasksTable;
import org.jdesktop.application.Application;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceManager;
import org.jdesktop.application.ResourceMap;

/**
 * Transfert handler used to transfert task in table.
 * 
 * @author chatellier
 * @version $Revision: 2607 $
 * 
 * Last update : $Date: 2009-06-18 17:19:19 +0200 (jeu. 18 juin 2009) $ By : $Author: echatellier $
 */
public class TimerTaskTransferHandler extends TransferHandler {

    /** serialVersionUID. */
    private static final long serialVersionUID = 5433321973795969278L;

    /** log. */
    private static Log log = LogFactory.getLog(TimerTaskTransferHandler.class);

    /** I18n resources map */
    protected ResourceMap resourceMap;
    
    /** Data manager. */
    protected TimerDataManager dataManager;

    /**
     * Constructor.
     * 
     * @param application
     * @param dataManager data manager
     */
    public TimerTaskTransferHandler(Application application, TimerDataManager dataManager) {
        this.dataManager = dataManager;
        
        // init resources map
        ApplicationContext ctxt = application.getContext();
        ResourceManager mgr = ctxt.getResourceManager();
        resourceMap = mgr.getResourceMap(TimerTaskTransferHandler.class);
    }

    @Override
    public boolean canImport(JComponent cp, DataFlavor[] df) {

        for (int i = 0; i < df.length; i++) {
            if (df[i].equals(TimerTaskTranferable.myData)) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean importData(JComponent component, Transferable transferable) {

        boolean confirmImport = false;

        if (transferable.isDataFlavorSupported(TimerTaskTranferable.myData)) {
            try {

                ProjectsAndTasksTable treeTable = (ProjectsAndTasksTable) component;
                // elements here, task can be move to task or projects
                List<TimerTask> selectedTasks = treeTable.getSelectedElements();

                if (selectedTasks != null && !selectedTasks.isEmpty()) {
                    TimerTask destinationTask = selectedTasks.get(0);
                    Object myObject = transferable
                            .getTransferData(TimerTaskTranferable.myData);
                    TimerTask movedTask = (TimerTask) myObject;

                    // can't move task to itself
                    if (movedTask != destinationTask) {
                        String title = resourceMap.getString("move.confirmation.title");
                        String message = resourceMap.getString("move.confirmation.message", movedTask.getName(), destinationTask.getName());
                        int answer = JOptionPane.showConfirmDialog(component,
                                message, title,
                                JOptionPane.YES_NO_OPTION,
                                JOptionPane.QUESTION_MESSAGE);
    
                        if (answer == JOptionPane.YES_OPTION) {
                            confirmImport = true;
                        }
                    }
                }
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Exception while transfering task", e);
                }
            } catch (UnsupportedFlavorException e) {
                if (log.isErrorEnabled()) {
                    log.error("Exception while transfering task", e);
                }
            }
        }
        return confirmImport;

    }

    /**
     * Get selected task in tree, and build Transferable object for it.
     * 
     * @param cp component
     * @return transferable instance for selected task
     */
    @Override
    protected Transferable createTransferable(JComponent cp) {

        Transferable transferable = null;

        ProjectsAndTasksTable treeTable = (ProjectsAndTasksTable) cp;
        // only task can be moved !
        List<TimerTask> selectedTasks = treeTable.getSelectedTasks();
        if (selectedTasks != null && !selectedTasks.isEmpty()) {
            TimerTask task = selectedTasks.get(0);
            transferable = new TimerTaskTranferable(task);
        }

        return transferable;
    }

    @Override
    protected void exportDone(JComponent cp, Transferable transferable, int type) {
        if (log.isDebugEnabled()) {
            log.debug("Transfert done");
        }
        
        if (type == TransferHandler.MOVE) {

            try {
                ProjectsAndTasksTable treeTable = (ProjectsAndTasksTable) cp;
                // elements here, task can be move to task or projects
                List<TimerTask> selectedTasks = treeTable.getSelectedElements();

                if (selectedTasks != null && !selectedTasks.isEmpty()) {
                    TimerTask destinationTask = selectedTasks.get(0);
                    Object myObject = transferable
                            .getTransferData(TimerTaskTranferable.myData);
                    TimerTask movedTask = (TimerTask) myObject;

                    try {
                        dataManager.moveTask(movedTask, destinationTask);
                    }
                    catch(DataViolationException e) {
                        String title = resourceMap.getString("action.invalidActionTitle");
                        String message = resourceMap.getString(e.getExceptionKey());
                        if (StringUtils.isEmpty(message)) {
                            message = resourceMap.getString("action.missingErrorMessage", e.getExceptionKey());
                        }
                        JOptionPane.showMessageDialog(cp, message, title, JOptionPane.ERROR_MESSAGE);
                    }
                }
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Exception while transfering task", e);
                }
            } catch (UnsupportedFlavorException e) {
                if (log.isErrorEnabled()) {
                    log.error("Exception while transfering task", e);
                }
            }
        }
    }

    @Override
    public int getSourceActions(JComponent component) {
        return MOVE;
    }
}
