/* *##%
 * Copyright (C) 2008, 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.ws;

import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.entities.ConnectionData;
import org.chorem.jtimer.ws.ConnectionDataHandler;
import org.chorem.jtimer.ws.ProjectManagement;
import org.chorem.jtimer.ws.exception.WebServiceException;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.LocalStorage;
import org.jdesktop.application.ResourceManager;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.SingleFrameApplication;

/**
 * Swing ConnectionInformationHandler.
 * 
 * @author chatellier
 * @version $Revision: 2607 $
 * 
 * Last update : $Date: 2009-06-18 17:19:19 +0200 (jeu. 18 juin 2009) $ By : $Author: echatellier $
 */
public class SwingConnectionInformationHandler extends Dialog implements
        ConnectionDataHandler, ActionListener, ListSelectionListener {

    /** serialVersionUID */
    private static final long serialVersionUID = -4178930926937567471L;

    /** log */
    private static Log log = LogFactory
            .getLog(SwingConnectionInformationHandler.class);

    /** Displayed list */
    protected JList list;

    /** Ok button */
    protected JButton okButton;

    /** Clicked button */
    protected JButton selectedButton;

    /** Local storage */
    protected LocalStorage localStorage;

    /** I18n resources map */
    protected ResourceMap resourceMap;

    /**
     * Data displayed in graphical list.
     */
    protected class LoginListData implements Comparable<LoginListData> {

        /** login */
        protected String login;
        /** id */
        protected String id;

        /**
         * Constructor.
         * 
         * @param login login
         * @param id id
         */
        public LoginListData(String login, String id) {
            this.login = login;
            this.id = id;
        }

        /**
         * Get id;
         * 
         * @return id
         */
        public String getId() {
            return id;
        }

        /**
         * toString only display login.
         * 
         * @return login
         */
        public String toString() {
            return login;
        }

        /*
         * @see java.lang.Comparable#compareTo(java.lang.Object)
         */
        @Override
        public int compareTo(LoginListData o) {

            return login.compareTo(o.login);
        }
    }

    /**
     * Constructor.
     * 
     * @param application parent application reference
     */
    public SwingConnectionInformationHandler(SingleFrameApplication application) {
        super(application.getMainFrame(), true);

        // set name for swing application framework
        super.setName(SwingConnectionInformationHandler.class.getName());

        // init resources map
        ApplicationContext ctxt = application.getContext();
        ResourceManager mgr = ctxt.getResourceManager();
        resourceMap = mgr
                .getResourceMap(SwingConnectionInformationHandler.class);

        localStorage = application.getContext().getLocalStorage();
    }

    /**
     * Build storage name (common place).
     * 
     * @param resourceName resource name
     * @return storage name
     */
    protected String getStorageName(String resourceName) {

        // add unique name, that pass refactor 
        // data are stored in xml format by reflection
        return ConnectionDataHandler.class.getName() + "." + resourceName
                + ".xml";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.chorem.jtimer.ws.ConnectionInformationHandler#getConnectionInformation(java.lang.String)
     */
    @Override
    public ConnectionData getConnectionInformation(
            ProjectManagement managementService) {

        // try to get file information
        ConnectionData data = null;
        try {
            data = (ConnectionData) localStorage
                    .load(getStorageName(managementService.getResourceName()));

            if (data != null) {
                if (log.isInfoEnabled()) {
                    log.info("found user connection information  (login="
                            + data.getLogin() + ")");
                }
            } else {
                data = createNewConnectionInformation(managementService);
            }
        } catch (IOException e) {

            if (log.isDebugEnabled()) {
                log.debug("Connection information not found for current user, ask it",
                                e);
            }

            data = createNewConnectionInformation(managementService);
        }

        return data;
    }

    /**
     * Create, store, and return connection data.
     * 
     * @param managementService interface on remote service
     * @return correct connection data
     */
    protected ConnectionData createNewConnectionInformation(
            ProjectManagement managementService) {
        if (log.isInfoEnabled()) {
            log.info("Connection information not found for current user, ask it");
        }

        ConnectionData data = askUserForInformation(managementService);

        if (data != null) {
            try {
                localStorage.save(data, getStorageName(managementService
                        .getResourceName()));
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't save info for user", e);
                }
            }
        }

        return data;
    }

    /**
     * Add user for data (swing prompt).
     * 
     * @param managementService interface on remote service
     * @return data specified by user
     */
    protected ConnectionData askUserForInformation(
            ProjectManagement managementService) {

        // data to return
        ConnectionData data = null;

        // try to get users list on remote service

        try {
            Map<String, String> idLogins = managementService
                    .getIdAndLoginsMap();

            List<LoginListData> v = new ArrayList<LoginListData>();
            for (Entry<String, String> entry : idLogins.entrySet()) {
                v.add(new LoginListData(entry.getValue(), entry.getKey()));
            }

            // sort list in login order
            Collections.sort(v);
            buildUI(managementService);
            list.setListData(v.toArray());

            // for dialog, call is blocking when modal
            setVisible(true);

            // not cancel, ie ok ;)
            if (okButton.equals(selectedButton)) {

                LoginListData loginData = (LoginListData) list
                        .getSelectedValue();
                if (loginData.getId() != null) {
                    data = new ConnectionData();
                    data.setLogin(loginData.getId());
                }
            }
        } catch (WebServiceException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't contact remote service", e);
            }
        }

        return data;
    }

    /**
     * build ui.
     * 
     * @param managementService management service
     */
    protected void buildUI(ProjectManagement managementService) {

        // set title
        String title = resourceMap.getString("askInformationTitle");
        super.setTitle(title);

        // build UI component
        setLayout(new GridBagLayout());

        // image
        JLabel labelInfo = new JLabel(resourceMap.getString(
                "askInformationMessage", managementService.getResourceName()));
        add(labelInfo, new GridBagConstraints(0, 0, 2, 1, 0, 0, GridBagConstraints.CENTER,
                GridBagConstraints.HORIZONTAL, new Insets(5, 5, 0, 5), 0, 0));

        // image
        list = new JList();
        list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        list.addListSelectionListener(this);
        JScrollPane jsp = new JScrollPane(list);
        add(jsp, new GridBagConstraints(0, 1, 2, 1, 1, 1, GridBagConstraints.CENTER,
                GridBagConstraints.BOTH, new Insets(5, 5, 5, 5), 0, 0));

        // image
        okButton = new JButton(resourceMap.getString("ok"));
        okButton.addActionListener(this);
        okButton.setEnabled(false);
        add(okButton, new GridBagConstraints(0, 2, 1, 1, 1, 0, GridBagConstraints.EAST, GridBagConstraints.NONE,
                new Insets(0, 5, 5, 5), 10, 0));

        // image
        JButton cancelButton = new JButton(resourceMap.getString("cancel"));
        cancelButton.addActionListener(this);
        add(cancelButton, new GridBagConstraints(1, 2, 1, 1, 0, 0, GridBagConstraints.EAST, GridBagConstraints.NONE,
                new Insets(0, 0, 5, 0), 10, 0));

        // fixe size
        Dimension dim = getToolkit().getScreenSize();
        setSize(450, 420);
        setLocation((dim.width - getSize().width) / 2,
                (dim.height - getSize().height) / 2);
    }

    /*
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    @Override
    public void actionPerformed(ActionEvent e) {

        // all action cause close ;)
        selectedButton = (JButton) e.getSource();
        setVisible(false);

    }

    /*
     * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    @Override
    public void valueChanged(ListSelectionEvent e) {

        okButton.setEnabled(true);
    }
}
