/* *##%
 * Copyright (C) 2008, 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.systray;

import java.awt.AWTException;
import java.awt.Image;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.JTimer;
import org.chorem.jtimer.entities.TimerTask;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceManager;
import org.jdesktop.application.ResourceMap;

/**
 * Manager for systray icon.
 * 
 * @author chatellier
 * @version $Revision: 2607 $
 * 
 * Last update : $Date: 2009-06-18 17:19:19 +0200 (jeu. 18 juin 2009) $ By : $Author: echatellier $
 */
public class AWTSystray extends SystrayManager implements ActionListener,
        MouseListener, PropertyChangeListener {

    /** Log */
    private static Log log = LogFactory.getLog(AWTSystray.class);

    /** I18n resources map */
    protected ResourceMap resourceMap;

    /** Tray icon */
    protected TrayIcon trayIcon;

    /** Idle image */
    protected Image idleImage;
    /** Running image */
    protected Image runningImage;
    /** Idle detect image */
    protected Image idleDetectImage;

    /** Reference how many tasks are running */
    protected int nbTasksRunning = 0;

    /** popup menu instance */
    protected JPopupMenu popup;

    /** Menu show */
    protected JMenuItem showItem;

    /**
     * Constructor.
     * 
     * Protected to be called only by factory.
     * 
     * @param parent parent
     */
    protected AWTSystray(JTimer parent) {

        super(parent);

        // init resources map
        ApplicationContext ctxt = parent.getContext();
        ResourceManager mgr = ctxt.getResourceManager();
        resourceMap = mgr.getResourceMap(AWTSystray.class);

        // load an image
        // use FQN to not conflict with annotation
        idleImage = resourceMap.getImageIcon("idleImage").getImage();
        runningImage = resourceMap.getImageIcon("runningImage").getImage();
        idleDetectImage = resourceMap.getImageIcon("idleDetectImage")
                .getImage();
    }

    /**
     * Install try icon into systray.
     */
    public void install() {

        // get the SystemTray instance
        SystemTray tray = SystemTray.getSystemTray();

        // create a action listener to listen for default action executed on
        // the tray icon
        // create a popup menu
        popup = new JPopupMenu();

        // show
        showItem = new JMenuItem(resourceMap.getString("hideMenuText"));
        // showItem.setName("trayShowMenu"); seems don't work with awt
        // components
        showItem.addActionListener(this);
        showItem.setActionCommand("showHide");
        popup.add(showItem);
        popup.addSeparator();

        // quit
        JMenuItem quitItem = new JMenuItem(resourceMap
                .getString("quitMenuText"));
        // defaultItem.setName("quit");
        quitItem.addActionListener(this);
        quitItem.setActionCommand("quit");
        popup.add(quitItem);

        // / ... add other items
        // construct a TrayIcon

        trayIcon = new TrayIcon(idleImage, resourceMap
                .getString("tooltipIdleText"), null);
        trayIcon.setImageAutoSize(true);
        trayIcon.addMouseListener(this);

        // add the tray image
        try {
            tray.add(trayIcon);
        } catch (AWTException e) {

            // add listener
            tray.addPropertyChangeListener("trayIcons", this);

            // exception is throw if systray is currently missing
            // not big deal, we will set it when it'll be ready
            if (log.isDebugEnabled()) {
                log.debug("Error while setting system tray", e);
            }
        }

        // call to super install
        super.install();
    }

    /*
     * @see org.chorem.jtimer.ui.systray.SystrayManager#startTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void startTask(TimerTask task) {
        startStopTask(task, true);
    }

    /*
     * @see org.chorem.jtimer.ui.systray.SystrayManager#stopTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void stopTask(TimerTask task) {
        startStopTask(task, false);
    }

    /**
     * Common code for start or stop task.
     * 
     * @param task task
     * @param start start(true) or stop(false) task
     */
    protected void startStopTask(TimerTask task, boolean start) {
        // increment or decrement task
        if (start) {
            ++nbTasksRunning;
        } else {
            --nbTasksRunning;
        }

        // display message
        String message = null;
        if (nbTasksRunning == 0) {
            message = resourceMap.getString("tooltipIdleText");

            trayIcon.setImage(idleImage);
        } else {
            trayIcon.setImage(runningImage);
            if (nbTasksRunning == 1) {
                message = resourceMap.getString("tooltipRunningTaskText",
                        nbTasksRunning);
            } else {
                message = resourceMap.getString("tooltipRunningTasksText",
                        nbTasksRunning);
            }
        }
        trayIcon.setToolTip(message);

    }

    /*
     * @see org.chorem.jtimer.ui.systray.SystrayManager#postIdleDetect()
     */
    @Override
    public void postIdleDetect() {

        if (log.isDebugEnabled()) {
            log.debug("Post idle detect");
        }

        if (nbTasksRunning == 0) {
            trayIcon.setImage(idleImage);
        } else {
            trayIcon.setImage(runningImage);
        }

    }

    /*
     * @see org.chorem.jtimer.ui.systray.SystrayManager#preIdleDetect()
     */
    @Override
    public void preIdleDetect() {

        if (log.isDebugEnabled()) {
            log.debug("Pre idle detect");
        }

        trayIcon.setImage(idleDetectImage);
    }

    /*
     * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt) {

        // get event name
        String event = evt.getPropertyName();

        if (log.isDebugEnabled()) {
            log.debug("Property change on system tray : " + event);
        }

        // systemTray event is only available in jdk7
        if (event.equals("trayIcons")) {

            if (log.isDebugEnabled()) {
                log.debug("Systray event : " + evt.getPropertyName());
            }

            SystemTray tray = SystemTray.getSystemTray();

            // tray become available
            if (tray != null) {
                try {
                    tray.add(trayIcon);
                } catch (AWTException e) {
                    if (log.isErrorEnabled()) {
                        log.error("System tray seems to become available, but can't set it !",
                                        e);
                    }
                }
            }
        }
    }

    /*
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {

        String actionCommand = e.getActionCommand();

        if ("showHide".equals(actionCommand)) {
            if (parent.getMainFrame().isVisible()) {
                // on la cache
                parent.hide();
                showItem.setText(resourceMap.getString("showMenuText"));
            } else {
                // sinon on la montre
                parent.show();
                showItem.setText(resourceMap.getString("hideMenuText"));
            }
        }

        if ("quit".equals(actionCommand)) {
            parent.quit(e);
        }
    }

    /*
     * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseClicked(MouseEvent e) {

        // ne fait pas de double affichage si plus de clics
        // et ne change pas l'affichage si clic droit
        if (e.getClickCount() == 1 && e.getButton() == MouseEvent.BUTTON1) {

            // si la fenetre est affichee
            if (parent.getMainFrame().isVisible()) {
                showItem.setText(resourceMap.getString("showMenuText"));

                // on la cache
                parent.hide();

            } else {

                showItem.setText(resourceMap.getString("hideMenuText"));

                // sinon on la montre
                // correct iconified bug
                // http://www.java-forums.org/awt-swing/7000-problem-setvisible-linux.html
                //parent.getMainFrame().setExtendedState(JFrame.NORMAL);
                parent.show();

            }
        }

    }

    /*
     * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseEntered(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseExited(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    @Override
    public void mousePressed(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseReleased(MouseEvent e) {

        // use sun workarround http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6285881
        if (e.getClickCount() == 1 && e.getButton() == MouseEvent.BUTTON3) {
            popup.setLocation(e.getX(), e.getY());
            popup.setInvoker(popup);
            popup.setVisible(true);
        }
    }

    /*
     * @see java.awt.event.WindowListener#windowClosing(java.awt.event.WindowEvent)
     */
    @Override
    public void windowClosing(WindowEvent e) {

        // hide window (without exiting)
        parent.hide();
        showItem.setText(resourceMap.getString("showMenuText"));
    }
}
