/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.kafka.common.utils.ByteBufferUnmapper;
import org.apache.kafka.common.utils.OperatingSystem;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.storage.internals.log.IndexEntry;
import org.apache.kafka.storage.internals.log.IndexOffsetOverflowException;
import org.apache.kafka.storage.internals.log.IndexSearchType;
import org.apache.kafka.storage.internals.log.StorageAction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractIndex
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(AbstractIndex.class);
    protected final ReentrantLock lock = new ReentrantLock();
    private final long baseOffset;
    private final int maxIndexSize;
    private final boolean writable;
    private volatile File file;
    private volatile long length;
    private volatile MappedByteBuffer mmap;
    private volatile int maxEntries;
    private volatile int entries;

    public AbstractIndex(File file, long baseOffset, int maxIndexSize, boolean writable) throws IOException {
        Objects.requireNonNull(file);
        this.file = file;
        this.baseOffset = baseOffset;
        this.maxIndexSize = maxIndexSize;
        this.writable = writable;
        this.createAndAssignMmap();
        this.maxEntries = this.mmap.limit() / this.entrySize();
        this.entries = this.mmap.position() / this.entrySize();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createAndAssignMmap() throws IOException {
        boolean newlyCreated = this.file.createNewFile();
        RandomAccessFile raf = this.writable ? new RandomAccessFile(this.file, "rw") : new RandomAccessFile(this.file, "r");
        try {
            if (newlyCreated) {
                if (this.maxIndexSize < this.entrySize()) {
                    throw new IllegalArgumentException("Invalid max index size: " + this.maxIndexSize);
                }
                raf.setLength(AbstractIndex.roundDownToExactMultiple(this.maxIndexSize, this.entrySize()));
            }
            long length = raf.length();
            MappedByteBuffer mmap = AbstractIndex.createMappedBuffer(raf, newlyCreated, length, this.writable, this.entrySize());
            this.length = length;
            this.mmap = mmap;
        }
        finally {
            Utils.closeQuietly((AutoCloseable)raf, (String)("index " + this.file.getName()));
        }
    }

    public abstract void sanityCheck();

    public abstract void truncateTo(long var1);

    protected abstract void truncate();

    protected abstract int entrySize();

    protected abstract IndexEntry parseEntry(ByteBuffer var1, int var2);

    public boolean isFull() {
        return this.entries >= this.maxEntries;
    }

    public File file() {
        return this.file;
    }

    public int maxEntries() {
        return this.maxEntries;
    }

    public int entries() {
        return this.entries;
    }

    public long length() {
        return this.length;
    }

    public int maxIndexSize() {
        return this.maxIndexSize;
    }

    public long baseOffset() {
        return this.baseOffset;
    }

    public void updateParentDir(File parentDir) {
        this.file = new File(parentDir, this.file.getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean resize(int newSize) throws IOException {
        this.lock.lock();
        try {
            boolean bl;
            int roundedNewSize = AbstractIndex.roundDownToExactMultiple(newSize, this.entrySize());
            if (this.length == (long)roundedNewSize) {
                log.debug("Index {} was not resized because it already has size {}", (Object)this.file.getAbsolutePath(), (Object)roundedNewSize);
                boolean bl2 = false;
                return bl2;
            }
            RandomAccessFile raf = new RandomAccessFile(this.file, "rw");
            try {
                int position = this.mmap.position();
                if (OperatingSystem.IS_WINDOWS || OperatingSystem.IS_ZOS) {
                    this.safeForceUnmap();
                }
                raf.setLength(roundedNewSize);
                this.length = roundedNewSize;
                this.mmap = raf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, roundedNewSize);
                this.maxEntries = this.mmap.limit() / this.entrySize();
                this.mmap.position(position);
                log.debug("Resized {} to {}, position is {} and limit is {}", new Object[]{this.file.getAbsolutePath(), roundedNewSize, this.mmap.position(), this.mmap.limit()});
                bl = true;
            }
            catch (Throwable throwable) {
                Utils.closeQuietly((AutoCloseable)raf, (String)("index file " + this.file.getName()));
                throw throwable;
            }
            Utils.closeQuietly((AutoCloseable)raf, (String)("index file " + this.file.getName()));
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void renameTo(File f) throws IOException {
        try {
            Utils.atomicMoveWithFallback((Path)this.file.toPath(), (Path)f.toPath(), (boolean)false);
        }
        finally {
            this.file = f;
        }
    }

    public void flush() {
        this.lock.lock();
        try {
            this.mmap.force();
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean deleteIfExists() throws IOException {
        this.closeHandler();
        return Files.deleteIfExists(this.file.toPath());
    }

    public void trimToValidSize() throws IOException {
        this.lock.lock();
        try {
            this.resize(this.entrySize() * this.entries);
        }
        finally {
            this.lock.unlock();
        }
    }

    public int sizeInBytes() {
        return this.entrySize() * this.entries;
    }

    @Override
    public void close() throws IOException {
        this.trimToValidSize();
        this.closeHandler();
    }

    public void closeHandler() {
        this.lock.lock();
        try {
            this.safeForceUnmap();
        }
        finally {
            this.lock.unlock();
        }
    }

    public void reset() throws IOException {
        this.truncate();
        this.resize(this.maxIndexSize);
    }

    public int relativeOffset(long offset) {
        OptionalInt relativeOffset = this.toRelative(offset);
        return relativeOffset.orElseThrow(() -> new IndexOffsetOverflowException("Integer overflow for offset: " + offset + " (" + this.file.getAbsoluteFile() + ")"));
    }

    public boolean canAppendOffset(long offset) {
        return this.toRelative(offset).isPresent();
    }

    protected final MappedByteBuffer mmap() {
        return this.mmap;
    }

    protected final int warmEntries() {
        return 8192 / this.entrySize();
    }

    protected void safeForceUnmap() {
        if (this.mmap != null) {
            try {
                this.forceUnmap();
            }
            catch (Throwable t) {
                log.error("Error unmapping index {}", (Object)this.file, (Object)t);
            }
        }
    }

    public void forceUnmap() throws IOException {
        try {
            ByteBufferUnmapper.unmap((String)this.file.getAbsolutePath(), (ByteBuffer)this.mmap);
        }
        finally {
            this.mmap = null;
        }
    }

    protected void incrementEntries() {
        ++this.entries;
    }

    protected void truncateToEntries0(int entries) {
        this.entries = entries;
        this.mmap.position(entries * this.entrySize());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected final <T, E extends Exception> T maybeLock(Lock lock, StorageAction<T, E> action) throws E {
        if (OperatingSystem.IS_WINDOWS || OperatingSystem.IS_ZOS) {
            lock.lock();
        }
        try {
            T t = action.execute();
            return t;
        }
        finally {
            if (OperatingSystem.IS_WINDOWS || OperatingSystem.IS_ZOS) {
                lock.unlock();
            }
        }
    }

    protected int largestLowerBoundSlotFor(ByteBuffer idx, long target, IndexSearchType searchEntity) {
        return this.indexSlotRangeFor(idx, target, searchEntity, SearchResultType.LARGEST_LOWER_BOUND);
    }

    protected int smallestUpperBoundSlotFor(ByteBuffer idx, long target, IndexSearchType searchEntity) {
        return this.indexSlotRangeFor(idx, target, searchEntity, SearchResultType.SMALLEST_UPPER_BOUND);
    }

    private static int roundDownToExactMultiple(int number, int factor) {
        return factor * (number / factor);
    }

    private static MappedByteBuffer createMappedBuffer(RandomAccessFile raf, boolean newlyCreated, long length, boolean writable, int entrySize) throws IOException {
        MappedByteBuffer idx = writable ? raf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, length) : raf.getChannel().map(FileChannel.MapMode.READ_ONLY, 0L, length);
        if (newlyCreated) {
            idx.position(0);
        } else {
            idx.position(AbstractIndex.roundDownToExactMultiple(idx.limit(), entrySize));
        }
        return idx;
    }

    private int indexSlotRangeFor(ByteBuffer idx, long target, IndexSearchType searchEntity, SearchResultType searchResultType) {
        if (this.entries == 0) {
            return -1;
        }
        int firstHotEntry = Math.max(0, this.entries - 1 - this.warmEntries());
        if (this.compareIndexEntry(this.parseEntry(idx, firstHotEntry), target, searchEntity) < 0) {
            return this.binarySearch(idx, target, searchEntity, searchResultType, firstHotEntry, this.entries - 1);
        }
        if (this.compareIndexEntry(this.parseEntry(idx, 0), target, searchEntity) > 0) {
            switch (searchResultType) {
                case LARGEST_LOWER_BOUND: {
                    return -1;
                }
                case SMALLEST_UPPER_BOUND: {
                    return 0;
                }
            }
        }
        return this.binarySearch(idx, target, searchEntity, searchResultType, 0, firstHotEntry);
    }

    private int binarySearch(ByteBuffer idx, long target, IndexSearchType searchEntity, SearchResultType searchResultType, int begin, int end) {
        int lo = begin;
        int hi = end;
        while (lo < hi) {
            int mid = lo + hi + 1 >>> 1;
            IndexEntry found = this.parseEntry(idx, mid);
            int compareResult = this.compareIndexEntry(found, target, searchEntity);
            if (compareResult > 0) {
                hi = mid - 1;
                continue;
            }
            if (compareResult < 0) {
                lo = mid;
                continue;
            }
            return mid;
        }
        switch (searchResultType) {
            case LARGEST_LOWER_BOUND: {
                return lo;
            }
            case SMALLEST_UPPER_BOUND: {
                if (lo == this.entries - 1) {
                    return -1;
                }
                return lo + 1;
            }
        }
        throw new IllegalStateException("Unexpected searchResultType " + (Object)((Object)searchResultType));
    }

    private int compareIndexEntry(IndexEntry indexEntry, long target, IndexSearchType searchEntity) {
        int result;
        switch (searchEntity) {
            case KEY: {
                result = Long.compare(indexEntry.indexKey(), target);
                break;
            }
            case VALUE: {
                result = Long.compare(indexEntry.indexValue(), target);
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected IndexSearchType: " + (Object)((Object)searchEntity));
            }
        }
        return result;
    }

    private OptionalInt toRelative(long offset) {
        long relativeOffset = offset - this.baseOffset;
        if (relativeOffset < 0L || relativeOffset > Integer.MAX_VALUE) {
            return OptionalInt.empty();
        }
        return OptionalInt.of((int)relativeOffset);
    }

    private static enum SearchResultType {
        LARGEST_LOWER_BOUND,
        SMALLEST_UPPER_BOUND;

    }
}

