/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gwtwidgets.server.spring;

import java.util.Map;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.handler.AbstractUrlHandlerMapping;

import com.google.gwt.user.client.rpc.RemoteService;

/**
 * The GWTHandler implements a Spring {@link HandlerMapping} which maps RPC from
 * URLs to {@link RemoteService} implementations. It does so by wrapping service
 * beans with a {@link GWTRPCServiceExporter} dynamically proxying all
 * {@link RemoteService} interfaces implemented by the service and delegating
 * RPC to these interfaces to the service. It is possible to use custom implementations
 * of the {@link GWTRPCServiceExporter}, see {@link #setServiceExporterFactory(RPCServiceExporterFactory)}.
 * 
 * @author George Georgovassilis, g.georgovassilis[at]gmail.com
 * 
 */
public class GWTHandler extends AbstractUrlHandlerMapping implements HandlerMapping, InitializingBean{

	// temporary mapping, void after bean initialisation
	private Map<String, Object> _mapping;
	protected RPCServiceExporterFactory factory = new DefaultRPCServiceExporterFactory();
	protected boolean disableResponseCaching = false;
	
	private RPCServiceExporter initServiceInstance(RPCServiceExporter exporter, Object service, Class<RemoteService>[] serviceInterfaces) {
		try {
			exporter.setResponseCachingDisabled(disableResponseCaching);
			exporter.setServletContext(getServletContext());
			exporter.setService(service);
			exporter.setServiceInterfaces(serviceInterfaces);
			exporter.afterPropertiesSet();
			return exporter;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Set a mapping between URLs and services
	 * 
	 * @param mapping
	 */
	public void setMappings(Map<String, Object> mapping) {
		this._mapping = mapping;
	}

	/**
	 * Invoked automatically by Spring after initialisation.
	 */
	public void afterPropertiesSet() throws Exception {
		for (Map.Entry<String, Object> entry : _mapping.entrySet()) {
			RPCServiceExporter exporter = factory.create();
			registerHandler(entry.getKey(), initServiceInstance(exporter, entry.getValue(), ReflectionUtils
					.getExposedInterfaces(entry.getValue().getClass())));
		}
		this._mapping = null;
	}

	/**
	 * Optionally, a {@link RPCServiceExporterFactory} can be injected if a different implementation or
	 * setup is required. Note that after initialization, the following sequence of invocations will
	 * be performed on the {@code serviceExporter} :<br><br>
	 * <code>
	 *		exporter.setServletContext();<br>
	 *		exporter.setService();<br>
	 *		exporter.setServiceInterfaces();<br>
	 *		exporter.afterPropertiesSet();<br>
	 *</code>
	 * @param factory
	 */
	public void setServiceExporterFactory(RPCServiceExporterFactory factory) {
		this.factory = factory;
	}

	/**
	 * Can be used to explicitly disable caching of RPC responses in the client by
	 * modifying the HTTP headers of the response.
	 * @param disableResponseCaching
	 */
	public void setDisableResponseCaching(boolean disableResponseCaching) {
		this.disableResponseCaching = disableResponseCaching;
	}
	

}
