/*
	* Copyright (C) 2002-2024 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.floats;

import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.HashCommon;
import static it.unimi.dsi.fastutil.HashCommon.arraySize;
import static it.unimi.dsi.fastutil.HashCommon.maxFill;
import java.util.Map;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import it.unimi.dsi.fastutil.booleans.BooleanCollection;
import it.unimi.dsi.fastutil.booleans.AbstractBooleanCollection;
import it.unimi.dsi.fastutil.booleans.BooleanIterator;
import it.unimi.dsi.fastutil.booleans.BooleanSpliterator;
import it.unimi.dsi.fastutil.booleans.BooleanSpliterators;
import it.unimi.dsi.fastutil.booleans.BooleanConsumer;
import java.util.Comparator;
import it.unimi.dsi.fastutil.booleans.BooleanListIterator;
import it.unimi.dsi.fastutil.objects.AbstractObjectSortedSet;
import it.unimi.dsi.fastutil.objects.ObjectListIterator;
import it.unimi.dsi.fastutil.objects.ObjectBidirectionalIterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterators;
import it.unimi.dsi.fastutil.objects.ObjectSortedSet;

/**
 * A type-specific linked hash map with with a fast, small-footprint implementation.
 *
 * <p>
 * Instances of this class use a hash table to represent a map. The table is filled up to a
 * specified <em>load factor</em>, and then doubled in size to accommodate new entries. If the table
 * is emptied below <em>one fourth</em> of the load factor, it is halved in size; however, the table
 * is never reduced to a size smaller than that at creation time: this approach makes it possible to
 * create maps with a large capacity in which insertions and deletions do not cause immediately
 * rehashing. Moreover, halving is not performed when deleting entries from an iterator, as it would
 * interfere with the iteration process.
 *
 * <p>
 * Note that {@link #clear()} does not modify the hash table size. Rather, a family of
 * {@linkplain #trim() trimming methods} lets you control the size of the table; this is
 * particularly useful if you reuse instances of this class.
 *
 * <p>
 * Entries returned by the type-specific {@link #entrySet()} method implement the suitable
 * type-specific {@link it.unimi.dsi.fastutil.Pair Pair} interface; only values are mutable.
 *
 * <p>
 * Iterators generated by this map will enumerate pairs in the same order in which they have been
 * added to the map (addition of pairs whose key is already present in the map does not change the
 * iteration order). Note that this order has nothing in common with the natural order of the keys.
 * The order is kept by means of a doubly linked list, represented <i>via</i> an array of longs
 * parallel to the table.
 *
 * <p>
 * This class implements the interface of a sorted map, so to allow easy access of the iteration
 * order: for instance, you can get the first key in iteration order with {@code firstKey()} without
 * having to create an iterator; however, this class partially violates the
 * {@link java.util.SortedMap} contract because all submap methods throw an exception and
 * {@link #comparator()} returns always {@code null}.
 *
 * <p>
 * Additional methods, such as {@code getAndMoveToFirst()}, make it easy to use instances of this
 * class as a cache (e.g., with LRU policy).
 *
 * <p>
 * The iterators provided by the views of this class using are type-specific
 * {@linkplain java.util.ListIterator list iterators}, and can be started at any element <em>which
 * is a key of the map</em>, or a {@link NoSuchElementException} exception will be thrown. If,
 * however, the provided element is not the first or last key in the map, the first access to the
 * list index will require linear time, as in the worst case the entire key set must be scanned in
 * iteration order to retrieve the positional index of the starting key. If you use just the methods
 * of a type-specific {@link it.unimi.dsi.fastutil.BidirectionalIterator}, however, all operations
 * will be performed in constant time.
 *
 * @see Hash
 * @see HashCommon
 */
public class Float2BooleanLinkedOpenHashMap extends AbstractFloat2BooleanSortedMap implements java.io.Serializable, Cloneable, Hash {
	private static final long serialVersionUID = 0L;
	private static final boolean ASSERTS = false;
	/** The array of keys. */
	protected transient float[] key;
	/** The array of values. */
	protected transient boolean[] value;
	/** The mask for wrapping a position counter. */
	protected transient int mask;
	/** Whether this map contains the key zero. */
	protected transient boolean containsNullKey;
	/**
	 * The index of the first entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int first = -1;
	/**
	 * The index of the last entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int last = -1;
	/**
	 * For each entry, the next and the previous entry in iteration order, stored as
	 * {@code ((prev & 0xFFFFFFFFL) << 32) | (next & 0xFFFFFFFFL)}. The first entry contains predecessor
	 * -1, and the last entry contains successor -1.
	 */
	protected transient long[] link;
	/** The current table size. */
	protected transient int n;
	/** Threshold after which we rehash. It must be the table size times {@link #f}. */
	protected transient int maxFill;
	/** We never resize below this threshold, which is the construction-time {#n}. */
	protected final transient int minN;
	/** Number of entries in the set (including the key zero, if present). */
	protected int size;
	/** The acceptable load factor. */
	protected final float f;
	/** Cached set of entries. */
	protected transient FastSortedEntrySet entries;
	/** Cached set of keys. */
	protected transient FloatSortedSet keys;
	/** Cached collection of values. */
	protected transient BooleanCollection values;

	/**
	 * Creates a new hash map.
	 *
	 * <p>
	 * The actual table size will be the least power of two greater than {@code expected}/{@code f}.
	 *
	 * @param expected the expected number of elements in the hash map.
	 * @param f the load factor.
	 */

	public Float2BooleanLinkedOpenHashMap(final int expected, final float f) {
		if (f <= 0 || f >= 1) throw new IllegalArgumentException("Load factor must be greater than 0 and smaller than 1");
		if (expected < 0) throw new IllegalArgumentException("The expected number of elements must be nonnegative");
		this.f = f;
		minN = n = arraySize(expected, f);
		mask = n - 1;
		maxFill = maxFill(n, f);
		key = new float[n + 1];
		value = new boolean[n + 1];
		link = new long[n + 1];
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 *
	 * @param expected the expected number of elements in the hash map.
	 */
	public Float2BooleanLinkedOpenHashMap(final int expected) {
		this(expected, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map with initial expected {@link Hash#DEFAULT_INITIAL_SIZE} entries and
	 * {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 */
	public Float2BooleanLinkedOpenHashMap() {
		this(DEFAULT_INITIAL_SIZE, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Float2BooleanLinkedOpenHashMap(final Map<? extends Float, ? extends Boolean> m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 */
	public Float2BooleanLinkedOpenHashMap(final Map<? extends Float, ? extends Boolean> m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Float2BooleanLinkedOpenHashMap(final Float2BooleanMap m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given
	 * type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 */
	public Float2BooleanLinkedOpenHashMap(final Float2BooleanMap m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @param f the load factor.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Float2BooleanLinkedOpenHashMap(final float[] k, final boolean[] v, final float f) {
		this(k.length, f);
		if (k.length != v.length) throw new IllegalArgumentException("The key array and the value array have different lengths (" + k.length + " and " + v.length + ")");
		for (int i = 0; i < k.length; i++) this.put(k[i], v[i]);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor using the elements of
	 * two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Float2BooleanLinkedOpenHashMap(final float[] k, final boolean[] v) {
		this(k, v, DEFAULT_LOAD_FACTOR);
	}

	private int realSize() {
		return containsNullKey ? size - 1 : size;
	}

	/**
	 * Ensures that this map can hold a certain number of keys without rehashing.
	 *
	 * @param capacity a number of keys; there will be no rehashing unless the map {@linkplain #size()
	 *            size} exceeds this number.
	 */
	public void ensureCapacity(final int capacity) {
		final int needed = arraySize(capacity, f);
		if (needed > n) rehash(needed);
	}

	private void tryCapacity(final long capacity) {
		final int needed = (int)Math.min(1 << 30, Math.max(2, HashCommon.nextPowerOfTwo((long)Math.ceil(capacity / f))));
		if (needed > n) rehash(needed);
	}

	private boolean removeEntry(final int pos) {
		final boolean oldValue = value[pos];
		size--;
		fixPointers(pos);
		shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	private boolean removeNullEntry() {
		containsNullKey = false;
		final boolean oldValue = value[n];
		size--;
		fixPointers(n);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	@Override
	public void putAll(Map<? extends Float, ? extends Boolean> m) {
		if (f <= .5) ensureCapacity(m.size()); // The resulting map will be sized for m.size() elements
		else tryCapacity(size() + m.size()); // The resulting map will be tentatively sized for size() + m.size()
												// elements
		super.putAll(m);
	}

	private int find(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) return containsNullKey ? n : -(n + 1);
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return -(pos + 1);
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return pos;
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return -(pos + 1);
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return pos;
		}
	}

	private void insert(final int pos, final float k, final boolean v) {
		if (pos == n) containsNullKey = true;
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size + 1, f));
		if (ASSERTS) checkTable();
	}

	@Override
	public boolean put(final float k, final boolean v) {
		final int pos = find(k);
		if (pos < 0) {
			insert(-pos - 1, k, v);
			return defRetValue;
		}
		final boolean oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/**
	 * Shifts left entries with the specified hash code, starting at the specified position, and empties
	 * the resulting free entry.
	 *
	 * @param pos a starting position.
	 */
	protected final void shiftKeys(int pos) {
		// Shift entries with the same hash.
		int last, slot;
		float curr;
		final float[] key = this.key;
		final boolean value[] = this.value;
		for (;;) {
			pos = ((last = pos) + 1) & mask;
			for (;;) {
				if ((Float.floatToRawIntBits(curr = key[pos]) == 0)) {
					key[last] = (0);
					return;
				}
				slot = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(curr)) & mask;
				if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
				pos = (pos + 1) & mask;
			}
			key[last] = curr;
			value[last] = value[pos];
			fixPointers(pos, last);
		}
	}

	@Override

	public boolean remove(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey) return removeNullEntry();
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return defRetValue;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return removeEntry(pos);
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return defRetValue;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return removeEntry(pos);
		}
	}

	private boolean setValue(final int pos, final boolean v) {
		final boolean oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/**
	 * Removes the mapping associated with the first key in iteration order.
	 * 
	 * @return the value previously associated with the first key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public boolean removeFirstBoolean() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = first;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			first = (int)link[pos];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
		}
		size--;
		final boolean v = value[pos];
		if (pos == n) {
			containsNullKey = false;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	/**
	 * Removes the mapping associated with the last key in iteration order.
	 * 
	 * @return the value previously associated with the last key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public boolean removeLastBoolean() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = last;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			last = (int)(link[pos] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
		}
		size--;
		final boolean v = value[pos];
		if (pos == n) {
			containsNullKey = false;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	private void moveIndexToFirst(final int i) {
		if (size == 1 || first == i) return;
		if (last == i) {
			last = (int)(link[i] >>> 32);
			// Special case of SET_NEXT(link[last], -1);
			link[last] |= -1 & 0xFFFFFFFFL;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[first] ^= ((link[first] ^ ((i & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[i] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
		first = i;
	}

	private void moveIndexToLast(final int i) {
		if (size == 1 || last == i) return;
		if (first == i) {
			first = (int)link[i];
			// Special case of SET_PREV(link[first], -1);
			link[first] |= (-1 & 0xFFFFFFFFL) << 32;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[last] ^= ((link[last] ^ (i & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[i] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
		last = i;
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * first position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public boolean getAndMoveToFirst(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return value[n];
			}
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return defRetValue;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) {
			moveIndexToFirst(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return defRetValue;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) {
				moveIndexToFirst(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * last position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public boolean getAndMoveToLast(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return value[n];
			}
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return defRetValue;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) {
			moveIndexToLast(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return defRetValue;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) {
				moveIndexToLast(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the first position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public boolean putAndMoveToFirst(final float k, final boolean v) {
		int pos;
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			float curr;
			final float[] key = this.key;
			// The starting point.
			if (!(Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) {
				if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
				while (!(Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[first] ^= ((link[first] ^ ((pos & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[pos] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
			first = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the last position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public boolean putAndMoveToLast(final float k, final boolean v) {
		int pos;
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			float curr;
			final float[] key = this.key;
			// The starting point.
			if (!(Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) {
				if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
				while (!(Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	@Override

	public boolean get(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) return containsNullKey ? value[n] : defRetValue;
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return defRetValue;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return defRetValue;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return value[pos];
		}
	}

	@Override

	public boolean containsKey(final float k) {
		if ((Float.floatToRawIntBits(k) == 0)) return containsNullKey;
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return false;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return true;
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return false;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return true;
		}
	}

	@Override
	public boolean containsValue(final boolean v) {
		final float key[] = this.key;
		final boolean value[] = this.value;
		if (containsNullKey && ((value[n]) == (v))) return true;
		for (int i = n; i-- != 0;) if (!(Float.floatToRawIntBits(key[i]) == 0) && ((value[i]) == (v))) return true;
		return false;
	}

	/** {@inheritDoc} */
	@Override

	public boolean getOrDefault(final float k, final boolean defaultValue) {
		if ((Float.floatToRawIntBits(k) == 0)) return containsNullKey ? value[n] : defaultValue;
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return defaultValue;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return defaultValue;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return value[pos];
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean putIfAbsent(final float k, final boolean v) {
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		insert(-pos - 1, k, v);
		return defRetValue;
	}

	/** {@inheritDoc} */
	@Override

	public boolean remove(final float k, final boolean v) {
		if ((Float.floatToRawIntBits(k) == 0)) {
			if (containsNullKey && ((v) == (value[n]))) {
				removeNullEntry();
				return true;
			}
			return false;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return false;
		if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr)) && ((v) == (value[pos]))) {
			removeEntry(pos);
			return true;
		}
		while (true) {
			if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return false;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr)) && ((v) == (value[pos]))) {
				removeEntry(pos);
				return true;
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean replace(final float k, final boolean oldValue, final boolean v) {
		final int pos = find(k);
		if (pos < 0 || !((oldValue) == (value[pos]))) return false;
		value[pos] = v;
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public boolean replace(final float k, final boolean v) {
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		final boolean oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/** {@inheritDoc} */
	@Override
	public boolean computeIfAbsent(final float k, final java.util.function.DoublePredicate mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		final boolean newValue = mappingFunction.test(k);
		insert(-pos - 1, k, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public boolean computeIfAbsent(final float key, final Float2BooleanFunction mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(key);
		if (pos >= 0) return value[pos];
		if (!mappingFunction.containsKey(key)) return defRetValue;
		final boolean newValue = mappingFunction.get(key);
		insert(-pos - 1, key, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public boolean computeIfAbsentNullable(final float k, final java.util.function.DoubleFunction<? extends Boolean> mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		final Boolean newValue = mappingFunction.apply(k);
		if (newValue == null) return defRetValue;
		final boolean v = (newValue).booleanValue();
		insert(-pos - 1, k, v);
		return v;
	}

	/** {@inheritDoc} */
	@Override
	public boolean computeIfPresent(final float k, final java.util.function.BiFunction<? super Float, ? super Boolean, ? extends Boolean> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		final Boolean newValue = remappingFunction.apply(Float.valueOf(k), Boolean.valueOf(value[pos]));
		if (newValue == null) {
			if ((Float.floatToRawIntBits(k) == 0)) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue).booleanValue();
	}

	/** {@inheritDoc} */
	@Override
	public boolean compute(final float k, final java.util.function.BiFunction<? super Float, ? super Boolean, ? extends Boolean> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		final Boolean newValue = remappingFunction.apply(Float.valueOf(k), pos >= 0 ? Boolean.valueOf(value[pos]) : null);
		if (newValue == null) {
			if (pos >= 0) {
				if ((Float.floatToRawIntBits(k) == 0)) removeNullEntry();
				else removeEntry(pos);
			}
			return defRetValue;
		}
		boolean newVal = (newValue).booleanValue();
		if (pos < 0) {
			insert(-pos - 1, k, newVal);
			return newVal;
		}
		return value[pos] = newVal;
	}

	/** {@inheritDoc} */
	@Override
	public boolean merge(final float k, final boolean v, final java.util.function.BiFunction<? super Boolean, ? super Boolean, ? extends Boolean> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);

		final int pos = find(k);
		if (pos < 0) {
			if (pos < 0) insert(-pos - 1, k, v);
			else value[pos] = v;
			return v;
		}
		final Boolean newValue = remappingFunction.apply(Boolean.valueOf(value[pos]), Boolean.valueOf(v));
		if (newValue == null) {
			if ((Float.floatToRawIntBits(k) == 0)) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue).booleanValue();
	}

	/* Removes all elements from this map.
	 *
	 * <p>To increase object reuse, this method does not change the table size.
	 * If you want to reduce the table size, you must use {@link #trim()}.
	 *
	 */
	@Override
	public void clear() {
		if (size == 0) return;
		size = 0;
		containsNullKey = false;
		Arrays.fill(key, (0));
		first = last = -1;
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public boolean isEmpty() {
		return size == 0;
	}

	/**
	 * The entry class for a hash map does not record key and value, but rather the position in the hash
	 * table of the corresponding entry. This is necessary so that calls to
	 * {@link java.util.Map.Entry#setValue(Object)} are reflected in the map
	 */
	final class MapEntry implements Float2BooleanMap.Entry, Map.Entry<Float, Boolean>, FloatBooleanPair {
		// The table index this entry refers to, or -1 if this entry has been deleted.
		int index;

		MapEntry(final int index) {
			this.index = index;
		}

		MapEntry() {
		}

		@Override
		public float getFloatKey() {
			return key[index];
		}

		@Override
		public float leftFloat() {
			return key[index];
		}

		@Override
		public boolean getBooleanValue() {
			return value[index];
		}

		@Override
		public boolean rightBoolean() {
			return value[index];
		}

		@Override
		public boolean setValue(final boolean v) {
			final boolean oldValue = value[index];
			value[index] = v;
			return oldValue;
		}

		@Override
		public FloatBooleanPair right(final boolean v) {
			value[index] = v;
			return this;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Float getKey() {
			return Float.valueOf(key[index]);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Boolean getValue() {
			return Boolean.valueOf(value[index]);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Boolean setValue(final Boolean v) {
			return Boolean.valueOf(setValue((v).booleanValue()));
		}

		@SuppressWarnings("unchecked")
		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			Map.Entry<Float, Boolean> e = (Map.Entry<Float, Boolean>)o;
			return (Float.floatToRawIntBits(key[index]) == Float.floatToRawIntBits((e.getKey()).floatValue())) && ((value[index]) == ((e.getValue()).booleanValue()));
		}

		@Override
		public int hashCode() {
			return it.unimi.dsi.fastutil.HashCommon.float2int(key[index]) ^ (value[index] ? 1231 : 1237);
		}

		@Override
		public String toString() {
			return key[index] + "=>" + value[index];
		}
	}

	/**
	 * Modifies the {@link #link} vector so that the given entry is removed. This method will complete
	 * in constant time.
	 *
	 * @param i the index of an entry.
	 */
	protected void fixPointers(final int i) {
		if (size == 0) {
			first = last = -1;
			return;
		}
		if (first == i) {
			first = (int)link[i];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
			return;
		}
		if (last == i) {
			last = (int)(link[i] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
			return;
		}
		final long linki = link[i];
		final int prev = (int)(linki >>> 32);
		final int next = (int)linki;
		link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
	}

	/**
	 * Modifies the {@link #link} vector for a shift from s to d.
	 * <p>
	 * This method will complete in constant time.
	 *
	 * @param s the source position.
	 * @param d the destination position.
	 */
	protected void fixPointers(int s, int d) {
		if (size == 1) {
			first = last = d;
			// Special case of SET_UPPER_LOWER(link[d], -1, -1)
			link[d] = -1L;
			return;
		}
		if (first == s) {
			first = d;
			link[(int)link[s]] ^= ((link[(int)link[s]] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[d] = link[s];
			return;
		}
		if (last == s) {
			last = d;
			link[(int)(link[s] >>> 32)] ^= ((link[(int)(link[s] >>> 32)] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[d] = link[s];
			return;
		}
		final long links = link[s];
		final int prev = (int)(links >>> 32);
		final int next = (int)links;
		link[prev] ^= ((link[prev] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[d] = links;
	}

	/**
	 * Returns the first key of this map in iteration order.
	 *
	 * @return the first key in iteration order.
	 */
	@Override
	public float firstFloatKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[first];
	}

	/**
	 * Returns the last key of this map in iteration order.
	 *
	 * @return the last key in iteration order.
	 */
	@Override
	public float lastFloatKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[last];
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Float2BooleanSortedMap tailMap(float from) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Float2BooleanSortedMap headMap(float to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Float2BooleanSortedMap subMap(float from, float to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just returns {@code null}.
	 */
	@Override
	public FloatComparator comparator() {
		return null;
	}

	/**
	 * A list iterator over a linked map.
	 *
	 * <p>
	 * This class provides a list iterator over a linked hash map. The constructor runs in constant
	 * time.
	 */
	private abstract class MapIterator<ConsumerType> {
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#previous()} (or
		 * {@code null} if no previous entry exists).
		 */
		int prev = -1;
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#next()} (or
		 * {@code null} if no next entry exists).
		 */
		int next = -1;
		/**
		 * The last entry that was returned (or -1 if we did not iterate or used
		 * {@link java.util.Iterator#remove()}).
		 */
		int curr = -1;
		/**
		 * The current index (in the sense of a {@link java.util.ListIterator}). Note that this value is not
		 * meaningful when this iterator has been created using the nonempty constructor.
		 */
		int index = -1;

		@SuppressWarnings("unused")
		abstract void acceptOnIndex(final ConsumerType action, final int index);

		protected MapIterator() {
			next = first;
			index = 0;
		}

		private MapIterator(final float from) {
			if ((Float.floatToRawIntBits(from) == 0)) {
				if (Float2BooleanLinkedOpenHashMap.this.containsNullKey) {
					next = (int)link[n];
					prev = n;
					return;
				} else throw new NoSuchElementException("The key " + from + " does not belong to this map.");
			}
			if ((Float.floatToRawIntBits(key[last]) == Float.floatToRawIntBits(from))) {
				prev = last;
				index = size;
				return;
			}
			// The starting point.
			int pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(from)) & mask;
			// There's always an unused entry.
			while (!(Float.floatToRawIntBits(key[pos]) == 0)) {
				if ((Float.floatToRawIntBits(key[pos]) == Float.floatToRawIntBits(from))) {
					// Note: no valid index known.
					next = (int)link[pos];
					prev = pos;
					return;
				}
				pos = (pos + 1) & mask;
			}
			throw new NoSuchElementException("The key " + from + " does not belong to this map.");
		}

		public boolean hasNext() {
			return next != -1;
		}

		public boolean hasPrevious() {
			return prev != -1;
		}

		private final void ensureIndexKnown() {
			if (index >= 0) return;
			if (prev == -1) {
				index = 0;
				return;
			}
			if (next == -1) {
				index = size;
				return;
			}
			int pos = first;
			index = 1;
			while (pos != prev) {
				pos = (int)link[pos];
				index++;
			}
		}

		public int nextIndex() {
			ensureIndexKnown();
			return index;
		}

		public int previousIndex() {
			ensureIndexKnown();
			return index - 1;
		}

		public int nextEntry() {
			if (!hasNext()) throw new NoSuchElementException();
			curr = next;
			next = (int)link[curr];
			prev = curr;
			if (index >= 0) index++;
			return curr;
		}

		public int previousEntry() {
			if (!hasPrevious()) throw new NoSuchElementException();
			curr = prev;
			prev = (int)(link[curr] >>> 32);
			next = curr;
			if (index >= 0) index--;
			return curr;
		}

		public void forEachRemaining(final ConsumerType action) {
			while (hasNext()) {
				curr = next;
				next = (int)link[curr];
				prev = curr;
				if (index >= 0) index++;
				acceptOnIndex(action, curr);
			}
		}

		public void remove() {
			ensureIndexKnown();
			if (curr == -1) throw new IllegalStateException();
			if (curr == prev) {
				/* If the last operation was a next(), we are removing an entry that preceeds
						   the current index, and thus we must decrement it. */
				index--;
				prev = (int)(link[curr] >>> 32);
			} else next = (int)link[curr];
			size--;
			/* Now we manually fix the pointers. Because of our knowledge of next
				   and prev, this is going to be faster than calling fixPointers(). */
			if (prev == -1) first = next;
			else link[prev] ^= ((link[prev] ^ (next & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			if (next == -1) last = prev;
			else link[next] ^= ((link[next] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			int last, slot, pos = curr;
			curr = -1;
			if (pos == n) {
				Float2BooleanLinkedOpenHashMap.this.containsNullKey = false;
			} else {
				float curr;
				final float[] key = Float2BooleanLinkedOpenHashMap.this.key;
				final boolean value[] = Float2BooleanLinkedOpenHashMap.this.value;
				// We have to horribly duplicate the shiftKeys() code because we need to update next/prev.
				for (;;) {
					pos = ((last = pos) + 1) & mask;
					for (;;) {
						if ((Float.floatToRawIntBits(curr = key[pos]) == 0)) {
							key[last] = (0);
							return;
						}
						slot = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(curr)) & mask;
						if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
						pos = (pos + 1) & mask;
					}
					key[last] = curr;
					value[last] = value[pos];
					if (next == pos) next = last;
					if (prev == pos) prev = last;
					fixPointers(pos, last);
				}
			}
		}

		public int skip(final int n) {
			int i = n;
			while (i-- != 0 && hasNext()) nextEntry();
			return n - i - 1;
		}

		public int back(final int n) {
			int i = n;
			while (i-- != 0 && hasPrevious()) previousEntry();
			return n - i - 1;
		}

		public void set(@SuppressWarnings("unused") Float2BooleanMap.Entry ok) {
			throw new UnsupportedOperationException();
		}

		public void add(@SuppressWarnings("unused") Float2BooleanMap.Entry ok) {
			throw new UnsupportedOperationException();
		}
	}

	private final class EntryIterator extends MapIterator<Consumer<? super Float2BooleanMap.Entry>> implements ObjectListIterator<Float2BooleanMap.Entry> {
		private MapEntry entry;

		public EntryIterator() {
		}

		public EntryIterator(float from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Float2BooleanMap.Entry> action, final int index) {
			action.accept(new MapEntry(index));
		}

		@Override
		public MapEntry next() {
			return entry = new MapEntry(nextEntry());
		}

		@Override
		public MapEntry previous() {
			return entry = new MapEntry(previousEntry());
		}

		@Override
		public void remove() {
			super.remove();
			entry.index = -1; // You cannot use a deleted entry.
		}
	}

	private final class FastEntryIterator extends MapIterator<Consumer<? super Float2BooleanMap.Entry>> implements ObjectListIterator<Float2BooleanMap.Entry> {
		final MapEntry entry = new MapEntry();

		public FastEntryIterator() {
		}

		public FastEntryIterator(float from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Float2BooleanMap.Entry> action, final int index) {
			entry.index = index;
			action.accept(entry);
		}

		@Override
		public MapEntry next() {
			entry.index = nextEntry();
			return entry;
		}

		@Override
		public MapEntry previous() {
			entry.index = previousEntry();
			return entry;
		}
	}

	private final class MapEntrySet extends AbstractObjectSortedSet<Float2BooleanMap.Entry> implements FastSortedEntrySet {
		private static final int SPLITERATOR_CHARACTERISTICS = ObjectSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public ObjectBidirectionalIterator<Float2BooleanMap.Entry> iterator() {
			return new EntryIterator();
		}

		/**
		 * {@inheritDoc}
		 *
		 * <p>
		 * There isn't a way to split efficiently while still preserving order for a linked data structure,
		 * so this implementation is just backed by the iterator. Thus, this spliterator is not well
		 * optimized for parallel streams.
		 *
		 * <p>
		 * Note, contrary to the specification of {@link java.util.SortedSet}, this spliterator does not,
		 * report {@link java.util.Spliterators.SORTED}. This is because iteration order is based on
		 * insertion order, not natural ordering.
		 */
		@Override
		public ObjectSpliterator<Float2BooleanMap.Entry> spliterator() {
			return ObjectSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Float2BooleanLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		@Override
		public Comparator<? super Float2BooleanMap.Entry> comparator() {
			return null;
		}

		@Override
		public ObjectSortedSet<Float2BooleanMap.Entry> subSet(Float2BooleanMap.Entry fromElement, Float2BooleanMap.Entry toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Float2BooleanMap.Entry> headSet(Float2BooleanMap.Entry toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Float2BooleanMap.Entry> tailSet(Float2BooleanMap.Entry fromElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public Float2BooleanMap.Entry first() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Float2BooleanLinkedOpenHashMap.this.first);
		}

		@Override
		public Float2BooleanMap.Entry last() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Float2BooleanLinkedOpenHashMap.this.last);
		}

		@Override

		public boolean contains(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Float)) return false;
			if (e.getValue() == null || !(e.getValue() instanceof Boolean)) return false;
			final float k = ((Float)(e.getKey())).floatValue();
			final boolean v = ((Boolean)(e.getValue())).booleanValue();
			if ((Float.floatToRawIntBits(k) == 0)) return Float2BooleanLinkedOpenHashMap.this.containsNullKey && ((value[n]) == (v));
			float curr;
			final float[] key = Float2BooleanLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return false;
			if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return ((value[pos]) == (v));
			// There's always an unused entry.
			while (true) {
				if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return false;
				if ((Float.floatToRawIntBits(k) == Float.floatToRawIntBits(curr))) return ((value[pos]) == (v));
			}
		}

		@Override

		public boolean remove(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Float)) return false;
			if (e.getValue() == null || !(e.getValue() instanceof Boolean)) return false;
			final float k = ((Float)(e.getKey())).floatValue();
			final boolean v = ((Boolean)(e.getValue())).booleanValue();
			if ((Float.floatToRawIntBits(k) == 0)) {
				if (containsNullKey && ((value[n]) == (v))) {
					removeNullEntry();
					return true;
				}
				return false;
			}
			float curr;
			final float[] key = Float2BooleanLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if ((Float.floatToRawIntBits(curr = key[pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask]) == 0)) return false;
			if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
				if (((value[pos]) == (v))) {
					removeEntry(pos);
					return true;
				}
				return false;
			}
			while (true) {
				if ((Float.floatToRawIntBits(curr = key[pos = (pos + 1) & mask]) == 0)) return false;
				if ((Float.floatToRawIntBits(curr) == Float.floatToRawIntBits(k))) {
					if (((value[pos]) == (v))) {
						removeEntry(pos);
						return true;
					}
				}
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public void clear() {
			Float2BooleanLinkedOpenHashMap.this.clear();
		}

		/**
		 * Returns a type-specific list iterator on the elements in this set, starting from a given element
		 * of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Float2BooleanMap.Entry> iterator(final Float2BooleanMap.Entry from) {
			return new EntryIterator(from.getFloatKey());
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from the first
		 * element. Please see the class documentation for implementation details.
		 *
		 * @return a type-specific list iterator starting at the first element.
		 */
		@Override
		public ObjectListIterator<Float2BooleanMap.Entry> fastIterator() {
			return new FastEntryIterator();
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from a given
		 * element of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Float2BooleanMap.Entry> fastIterator(final Float2BooleanMap.Entry from) {
			return new FastEntryIterator(from.getFloatKey());
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final Consumer<? super Float2BooleanMap.Entry> consumer) {
			final long[] link = Float2BooleanLinkedOpenHashMap.this.link;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(new MapEntry(curr));
			}
		}

		/** {@inheritDoc} */
		@Override
		public void fastForEach(final Consumer<? super Float2BooleanMap.Entry> consumer) {
			final MapEntry entry = new MapEntry();
			final long[] link = Float2BooleanLinkedOpenHashMap.this.link;
			for (int i = size, next = first; i-- != 0;) {
				entry.index = next;
				next = (int)link[next];
				consumer.accept(entry);
			}
		}
	}

	@Override
	public FastSortedEntrySet float2BooleanEntrySet() {
		if (entries == null) entries = new MapEntrySet();
		return entries;
	}

	/**
	 * An iterator on keys.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return keys instead of entries.
	 */
	private final class KeyIterator extends MapIterator<FloatConsumer> implements FloatListIterator {
		public KeyIterator(final float k) {
			super(k);
		}

		@Override
		public float previousFloat() {
			return key[previousEntry()];
		}

		public KeyIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final FloatConsumer action, final int index) {
			action.accept(key[index]);
		}

		@Override
		public float nextFloat() {
			return key[nextEntry()];
		}
	}

	private final class KeySet extends AbstractFloatSortedSet {
		private static final int SPLITERATOR_CHARACTERISTICS = FloatSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public FloatListIterator iterator(final float from) {
			return new KeyIterator(from);
		}

		@Override
		public FloatListIterator iterator() {
			return new KeyIterator();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see EntrySet#spliterator()
		 */
		@Override
		public FloatSpliterator spliterator() {
			return FloatSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Float2BooleanLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final FloatConsumer consumer) {
			final long[] link = Float2BooleanLinkedOpenHashMap.this.link;
			final float key[] = Float2BooleanLinkedOpenHashMap.this.key;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(key[curr]);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public boolean contains(float k) {
			return containsKey(k);
		}

		@Override
		public boolean remove(float k) {
			final int oldSize = size;
			Float2BooleanLinkedOpenHashMap.this.remove(k);
			return size != oldSize;
		}

		@Override
		public void clear() {
			Float2BooleanLinkedOpenHashMap.this.clear();
		}

		@Override
		public float firstFloat() {
			if (size == 0) throw new NoSuchElementException();
			return key[first];
		}

		@Override
		public float lastFloat() {
			if (size == 0) throw new NoSuchElementException();
			return key[last];
		}

		@Override
		public FloatComparator comparator() {
			return null;
		}

		@Override
		public FloatSortedSet tailSet(float from) {
			throw new UnsupportedOperationException();
		}

		@Override
		public FloatSortedSet headSet(float to) {
			throw new UnsupportedOperationException();
		}

		@Override
		public FloatSortedSet subSet(float from, float to) {
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public FloatSortedSet keySet() {
		if (keys == null) keys = new KeySet();
		return keys;
	}

	/**
	 * An iterator on values.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return values instead of entries.
	 */
	private final class ValueIterator extends MapIterator<BooleanConsumer> implements BooleanListIterator {
		@Override
		public boolean previousBoolean() {
			return value[previousEntry()];
		}

		public ValueIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final BooleanConsumer action, final int index) {
			action.accept(value[index]);
		}

		@Override
		public boolean nextBoolean() {
			return value[nextEntry()];
		}
	}

	@Override
	public BooleanCollection values() {
		if (values == null) values = new AbstractBooleanCollection() {
			private static final int SPLITERATOR_CHARACTERISTICS = BooleanSpliterators.COLLECTION_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

			@Override
			public BooleanIterator iterator() {
				return new ValueIterator();
			}

			/**
			 * {@inheritDoc}
			 * 
			 * @see EntrySet#spliterator()
			 */
			@Override
			public BooleanSpliterator spliterator() {
				return BooleanSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Float2BooleanLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
			}

			/** {@inheritDoc} */
			@Override
			public void forEach(final BooleanConsumer consumer) {
				final long[] link = Float2BooleanLinkedOpenHashMap.this.link;
				final boolean value[] = Float2BooleanLinkedOpenHashMap.this.value;
				for (int i = size, curr, next = first; i-- != 0;) {
					curr = next;
					next = (int)link[curr];
					consumer.accept(value[curr]);
				}
			}

			@Override
			public int size() {
				return size;
			}

			@Override
			public boolean contains(boolean v) {
				return containsValue(v);
			}

			@Override
			public void clear() {
				Float2BooleanLinkedOpenHashMap.this.clear();
			}
		};
		return values;
	}

	/**
	 * Rehashes the map, making the table as small as possible.
	 *
	 * <p>
	 * This method rehashes the table to the smallest size satisfying the load factor. It can be used
	 * when the set will not be changed anymore, so to optimize access speed and size.
	 *
	 * <p>
	 * If the table size is already the minimum possible, this method does nothing.
	 *
	 * @return true if there was enough memory to trim the map.
	 * @see #trim(int)
	 */
	public boolean trim() {
		return trim(size);
	}

	/**
	 * Rehashes this map if the table is too large.
	 *
	 * <p>
	 * Let <var>N</var> be the smallest table size that can hold <code>max(n,{@link #size()})</code>
	 * entries, still satisfying the load factor. If the current table size is smaller than or equal to
	 * <var>N</var>, this method does nothing. Otherwise, it rehashes this map in a table of size
	 * <var>N</var>.
	 *
	 * <p>
	 * This method is useful when reusing maps. {@linkplain #clear() Clearing a map} leaves the table
	 * size untouched. If you are reusing a map many times, you can call this method with a typical size
	 * to avoid keeping around a very large table just because of a few large transient maps.
	 *
	 * @param n the threshold for the trimming.
	 * @return true if there was enough memory to trim the map.
	 * @see #trim()
	 */
	public boolean trim(final int n) {
		final int l = HashCommon.nextPowerOfTwo((int)Math.ceil(n / f));
		if (l >= this.n || size > maxFill(l, f)) return true;
		try {
			rehash(l);
		} catch (OutOfMemoryError cantDoIt) {
			return false;
		}
		return true;
	}

	/**
	 * Rehashes the map.
	 *
	 * <p>
	 * This method implements the basic rehashing strategy, and may be overridden by subclasses
	 * implementing different rehashing strategies (e.g., disk-based rehashing). However, you should not
	 * override this method unless you understand the internal workings of this class.
	 *
	 * @param newN the new size
	 */

	protected void rehash(final int newN) {
		final float key[] = this.key;
		final boolean value[] = this.value;
		final int mask = newN - 1; // Note that this is used by the hashing macro
		final float newKey[] = new float[newN + 1];
		final boolean newValue[] = new boolean[newN + 1];
		int i = first, prev = -1, newPrev = -1, t, pos;
		final long link[] = this.link;
		final long newLink[] = new long[newN + 1];
		first = -1;
		for (int j = size; j-- != 0;) {
			if ((Float.floatToRawIntBits(key[i]) == 0)) pos = newN;
			else {
				pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(key[i])) & mask;
				while (!(Float.floatToRawIntBits(newKey[pos]) == 0)) pos = (pos + 1) & mask;
			}
			newKey[pos] = key[i];
			newValue[pos] = value[i];
			if (prev != -1) {
				newLink[newPrev] ^= ((newLink[newPrev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				newLink[pos] ^= ((newLink[pos] ^ ((newPrev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				newPrev = pos;
			} else {
				newPrev = first = pos;
				// Special case of SET(newLink[pos], -1, -1);
				newLink[pos] = -1L;
			}
			t = i;
			i = (int)link[i];
			prev = t;
		}
		this.link = newLink;
		this.last = newPrev;
		if (newPrev != -1)
			// Special case of SET_NEXT(newLink[newPrev], -1);
			newLink[newPrev] |= -1 & 0xFFFFFFFFL;
		n = newN;
		this.mask = mask;
		maxFill = maxFill(n, f);
		this.key = newKey;
		this.value = newValue;
	}

	/**
	 * Returns a deep copy of this map.
	 *
	 * <p>
	 * This method performs a deep copy of this hash map; the data stored in the map, however, is not
	 * cloned. Note that this makes a difference only for object keys.
	 *
	 * @return a deep copy of this map.
	 */
	@Override

	public Float2BooleanLinkedOpenHashMap clone() {
		Float2BooleanLinkedOpenHashMap c;
		try {
			c = (Float2BooleanLinkedOpenHashMap)super.clone();
		} catch (CloneNotSupportedException cantHappen) {
			throw new InternalError();
		}
		c.keys = null;
		c.values = null;
		c.entries = null;
		c.containsNullKey = containsNullKey;
		c.key = key.clone();
		c.value = value.clone();
		c.link = link.clone();
		return c;
	}

	/**
	 * Returns a hash code for this map.
	 *
	 * This method overrides the generic method provided by the superclass. Since {@code equals()} is
	 * not overriden, it is important that the value returned by this method is the same value as the
	 * one returned by the overriden method.
	 *
	 * @return a hash code for this map.
	 */
	@Override
	public int hashCode() {
		int h = 0;
		final float key[] = this.key;
		final boolean value[] = this.value;
		for (int j = realSize(), i = 0, t = 0; j-- != 0;) {
			while ((Float.floatToRawIntBits(key[i]) == 0)) i++;
			t = it.unimi.dsi.fastutil.HashCommon.float2int(key[i]);
			t ^= (value[i] ? 1231 : 1237);
			h += t;
			i++;
		}
		// Zero / null keys have hash zero.
		if (containsNullKey) h += (value[n] ? 1231 : 1237);
		return h;
	}

	private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
		final float key[] = this.key;
		final boolean value[] = this.value;
		final EntryIterator i = new EntryIterator();
		s.defaultWriteObject();
		for (int j = size, e; j-- != 0;) {
			e = i.nextEntry();
			s.writeFloat(key[e]);
			s.writeBoolean(value[e]);
		}
	}

	private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();
		n = arraySize(size, f);
		maxFill = maxFill(n, f);
		mask = n - 1;
		final float key[] = this.key = new float[n + 1];
		final boolean value[] = this.value = new boolean[n + 1];
		final long link[] = this.link = new long[n + 1];
		int prev = -1;
		first = last = -1;
		float k;
		boolean v;
		for (int i = size, pos; i-- != 0;) {
			k = s.readFloat();
			v = s.readBoolean();
			if ((Float.floatToRawIntBits(k) == 0)) {
				pos = n;
				containsNullKey = true;
			} else {
				pos = it.unimi.dsi.fastutil.HashCommon.mix(it.unimi.dsi.fastutil.HashCommon.float2int(k)) & mask;
				while (!(Float.floatToRawIntBits(key[pos]) == 0)) pos = (pos + 1) & mask;
			}
			key[pos] = k;
			value[pos] = v;
			if (first != -1) {
				link[prev] ^= ((link[prev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				link[pos] ^= ((link[pos] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				prev = pos;
			} else {
				prev = first = pos;
				// Special case of SET_PREV(newLink[pos], -1);
				link[pos] |= (-1L & 0xFFFFFFFFL) << 32;
			}
		}
		last = prev;
		if (prev != -1)
			// Special case of SET_NEXT(link[prev], -1);
			link[prev] |= -1 & 0xFFFFFFFFL;
		if (ASSERTS) checkTable();
	}

	private void checkTable() {
	}
}
