package fr.ifremer.wao.web.action.authentication;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.Iterables;
import fr.ifremer.wao.entity.ObsProgram;
import fr.ifremer.wao.entity.UserProfile;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.services.AuthenticatedWaoUser;
import fr.ifremer.wao.services.service.administration.InactiveWaoUserException;
import fr.ifremer.wao.services.service.administration.NoRoleAttributedException;
import fr.ifremer.wao.services.service.administration.WaoUsersService;
import fr.ifremer.wao.services.service.administration.WrongCredentialsException;
import fr.ifremer.wao.web.WaoJspActionSupport;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Result;
import org.apache.struts2.convention.annotation.Results;

import java.util.HashMap;
import java.util.Map;

@Results({
        @Result(name="success", type="redirectAction", params = { "namespace", "/${obsProgram.name().toLowerCase()}", "actionName", "news" })
})
public class LoginAction extends WaoJspActionSupport {

    private static final long serialVersionUID = 1L;

    protected transient WaoUsersService service;

    protected boolean credentialsAsked = true;

    protected String login;

    protected String password;

    protected String userProfileId;

    protected boolean userProfileAsked;

    protected Map<String, String> userProfiles;

    protected boolean acceptCguAsked;

    protected boolean cguAccepted;

    protected ObsProgram obsProgram;

    protected WaoUser waoUser;

    public void setLogin(String login) {
        this.login = login;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public void setService(WaoUsersService service) {
        this.service = service;
    }

    public void setUserProfileId(String userProfileId) {
        this.userProfileId = userProfileId;
    }

    public void setCguAccepted(boolean cguAccepted) {
        this.cguAccepted = cguAccepted;
    }

    @Override
    public void validate() {

        try {

            waoUser = service.authenticate(login, password);

            credentialsAsked = false;

            if (cguAccepted) {
                service.acceptCgu(waoUser);
            }

            if (waoUser.getCguAccepted() == null) {
                acceptCguAsked = true;
                addFieldError("cguAccepted", t("wao.ui.form.authentication.error.userMustAcceptCgu"));
            }

            if (waoUser.getUserProfile().size() == 1) {
                userProfileId = Iterables.getOnlyElement(waoUser.getUserProfile()).getTopiaId();
            } else {
                userProfileAsked = true;
                userProfiles = new HashMap<>();
                for (UserProfile userProfile : waoUser.getUserProfile()) {
                    String label = t(userProfile.getUserRole().getI18nKey());
                    label += " " + t(userProfile.getObsProgram().getI18nKey());
                    if (BooleanUtils.isTrue(userProfile.isReadOnly())) {
                        label += " (" + t("wao.ui.form.readOnly") + ")";
                    }
                    userProfiles.put(userProfile.getTopiaId(), label);
                }
            }

        } catch (WrongCredentialsException e) {
            addFieldError("login", t("wao.ui.form.authentication.error.wrongCredentials"));
        } catch (NoRoleAttributedException e) {
            addFieldError("login", t("wao.ui.form.authentication.error.noRoleAttributed"));
        } catch (InactiveWaoUserException e) {
            addFieldError("login", t("wao.ui.form.authentication.error.inactiveWaoUser"));
        }

    }

    @Override
    public String execute() {

        if (StringUtils.isNotBlank(userProfileId)) {

            UserProfile userProfile = waoUser.getUserProfileByTopiaId(userProfileId);
            AuthenticatedWaoUser authenticatedWaoUser = new AuthenticatedWaoUser(waoUser, userProfile);
            session.setAuthenticatedWaoUser(authenticatedWaoUser);

            obsProgram = userProfile.getObsProgram();

            return SUCCESS;

        }

        return INPUT;

    }

    public String getPassword() {
        return password;
    }

    public String getLogin() {
        return login;
    }

    public String getUserProfileId() {
        return userProfileId;
    }

    public boolean isUserProfileAsked() {
        return userProfileAsked;
    }

    public ObsProgram getObsProgram() {
        return obsProgram;
    }

    public boolean isAcceptCguAsked() {
        return acceptCguAsked;
    }

    public boolean isCguAccepted() {
        return cguAccepted;
    }

    public Map<String, String> getUserProfiles() {
        return userProfiles;
    }

    public boolean isCredentialsAsked() {
        return credentialsAsked;
    }
}
