package fr.ifremer.wao.web.action;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Optional;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.entity.ContactState;
import fr.ifremer.wao.entity.ContactStateMotif;
import fr.ifremer.wao.entity.DataReliability;
import fr.ifremer.wao.entity.ObservedDataControl;
import fr.ifremer.wao.entity.SamplingStrategy;
import fr.ifremer.wao.entity.TerrestrialLocation;
import fr.ifremer.wao.entity.TerrestrialLocations;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.entity.WaoUsers;
import fr.ifremer.wao.services.AuthenticatedWaoUser;
import fr.ifremer.wao.services.service.ContactDataInputDateAfterTodayException;
import fr.ifremer.wao.services.service.ContactDataInputDateBeforeObservationEndDateException;
import fr.ifremer.wao.services.service.ContactNotUpdatableException;
import fr.ifremer.wao.services.service.ContactObservationEndDateAfterTodayException;
import fr.ifremer.wao.services.service.ContactObservationEndDateBeforeBeginDateException;
import fr.ifremer.wao.services.service.ContactRestitutionDateBeforeDataInputDateException;
import fr.ifremer.wao.services.service.ContactWithObservedDataControlToCorrectionAskedException;
import fr.ifremer.wao.services.service.ContactsService;
import fr.ifremer.wao.services.service.DuplicatedContactMainObserverInSecondaryObserversException;
import fr.ifremer.wao.services.service.IllegalAcceptationException;
import fr.ifremer.wao.services.service.InvalidContactObservationBeginDateException;
import fr.ifremer.wao.services.service.MismatchContactMainObserverCompanyException;
import fr.ifremer.wao.services.service.MismatchContactSecondaryObserverCompanyException;
import fr.ifremer.wao.services.service.MissingContactCommentAdminException;
import fr.ifremer.wao.services.service.MissingContactCommentException;
import fr.ifremer.wao.services.service.MissingContactDataInputDateException;
import fr.ifremer.wao.services.service.MissingContactDataReliabilityException;
import fr.ifremer.wao.services.service.MissingContactMainObserverException;
import fr.ifremer.wao.services.service.MissingContactMammalsInfoException;
import fr.ifremer.wao.services.service.MissingContactNbObservantsException;
import fr.ifremer.wao.services.service.MissingContactObservationBeginDateException;
import fr.ifremer.wao.services.service.MissingContactObservationEndDateException;
import fr.ifremer.wao.services.service.MissingContactObservedDataControlException;
import fr.ifremer.wao.services.service.MissingContactRestitutionException;
import fr.ifremer.wao.services.service.MissingContactStateMotifException;
import fr.ifremer.wao.services.service.MissingContactTerrestrialLocationException;
import fr.ifremer.wao.services.service.UnknownContactIdException;
import fr.ifremer.wao.services.service.UnwantedContactContactStateMotifException;
import fr.ifremer.wao.services.service.UpdateContactCommand;
import fr.ifremer.wao.services.service.administration.ReferentialService;
import fr.ifremer.wao.services.service.administration.WaoUsersService;
import fr.ifremer.wao.web.WaoJspActionSupport;
import org.apache.struts2.convention.annotation.Result;
import org.apache.struts2.convention.annotation.Results;

import java.util.Date;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * Created on 4/6/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 4.0
 */
@Results({@Result(name = "success", type = "redirectAction", params = {"actionName", "contacts"})})
public class EditContactAction extends WaoJspActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    /**
     * Id of contact to edit.
     */
    protected Optional<String> optionalContactId = Optional.absent();

    protected Map<String, String> observers;

    protected Map<String, String> terrestrialLocations;

    protected Map<ContactState, String> contactStates;

    protected Map<String, String> contactStateMotives;

    protected Map<ObservedDataControl, String> observedDataControls;

    protected Map<DataReliability, String> dataReliabilities;

    protected Map<SamplingStrategy, String> samplingStrategies;

    protected transient ContactsService service;

    protected transient WaoUsersService waoUsersService;

    protected transient ReferentialService referentialService;

    protected UpdateContactCommand updateContactCommand;

    public void setService(ContactsService service) {
        this.service = service;
    }

    public void setWaoUsersService(WaoUsersService waoUsersService) {
        this.waoUsersService = waoUsersService;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public void setContactId(String contactId) {
        this.optionalContactId = Optional.fromNullable(Strings.emptyToNull(contactId));
    }

    public String getContactId() {
        return optionalContactId.orNull();
    }

    public UpdateContactCommand getUpdateContactCommand() {
        if (updateContactCommand == null) {
            prepare();
        }
        return updateContactCommand;
    }

    public Map<String, String> getObservers() {
        if (observers == null) {
            prepare();
        }
        return observers;
    }

    public Map<ContactState, String> getContactStates() {
        if (contactStates == null) {
            prepare();
        }
        return contactStates;
    }

    public Map<String, String> getContactStateMotives() {
        if (contactStateMotives == null) {
            prepare();
        }
        return contactStateMotives;
    }

    public Map<ObservedDataControl, String> getObservedDataControls() {
        if (observedDataControls == null) {
            prepare();
        }
        return observedDataControls;
    }

    public Map<DataReliability, String> getDataReliabilities() {
        if (dataReliabilities == null) {
            prepare();
        }
        return dataReliabilities;
    }

    public Map<String, String> getTerrestrialLocations() {
        if (terrestrialLocations == null) {
            prepare();
        }
        return terrestrialLocations;
    }

    public Map<SamplingStrategy, String> getSamplingStrategies() {
        if (samplingStrategies == null) {
            prepare();
        }
        return samplingStrategies;
    }

    public String getDateTimePlaceholder() {
        return formatDateTime(new Date());
    }

    public String getDatePlaceholder() {
        return formatDate(new Date());
    }

    @Override
    public void prepare() {

        try {
            updateContactCommand = service.newUpdateContactCommand(getAuthenticatedWaoUser(), optionalContactId);
        } catch (UnknownContactIdException e) {
            addActionError(t("wao.ui.error.unknownContactId"));
        }

        List<WaoUser> waoUsers = waoUsersService.getActiveWaoUsers(Optional.of(updateContactCommand.getContact().getSampleRow().getCompany().getTopiaId()));
        Set<WaoUser> sortedWaoUsers = Sets.newTreeSet(WaoUsers.fullNameComparator());
        sortedWaoUsers.addAll(waoUsers);
        observers = new LinkedHashMap<>();
        for (WaoUser waoUser : sortedWaoUsers) {
            observers.put(waoUser.getTopiaId(), waoUser.getFullName());
        }
        contactStates = new EnumMap<>(ContactState.class);
        for (ContactState contactState : ContactState.getAllowedStates(getAuthenticatedWaoUser().getObsProgram())) {
            contactStates.put(contactState, WaoUtils.l(getLocale(), contactState));
        }

        contactStateMotives = new TreeMap<>();
        for (ContactStateMotif contactStateMotif : referentialService.getAllContactStateMotifs(null)) {
            contactStateMotives.put(contactStateMotif.getTopiaId(), contactStateMotif.getName());
        }

        observedDataControls = new EnumMap<>(ObservedDataControl.class);
        for (ObservedDataControl observedDataControl : ObservedDataControl.values()) {
            observedDataControls.put(observedDataControl, WaoUtils.l(getLocale(), observedDataControl));
        }

        dataReliabilities = new EnumMap<>(DataReliability.class);
        for (DataReliability dataReliability : DataReliability.values()) {
            dataReliabilities.put(dataReliability, WaoUtils.l(getLocale(), dataReliability));
        }

        samplingStrategies = new EnumMap<>(SamplingStrategy.class);
        for (SamplingStrategy samplingStrategy : SamplingStrategy.values()) {
            samplingStrategies.put(samplingStrategy, WaoUtils.l(getLocale(), samplingStrategy));
        }

        terrestrialLocations = new LinkedHashMap<>();
        List<TerrestrialLocation> terrestrialLocationInDistricts = referentialService.getTerrestrialLocationInDistricts(updateContactCommand.getContact().getSampleRow().getTerrestrialLocations());
        Set<TerrestrialLocation> sortedTerrestrialLocations = new TreeSet<>(TerrestrialLocations.nameComparator());
        sortedTerrestrialLocations.addAll(terrestrialLocationInDistricts);
        for (TerrestrialLocation terrestrialLocation : sortedTerrestrialLocations) {
            terrestrialLocations.put(terrestrialLocation.getTopiaId(), terrestrialLocation.getDescription());
        }
    }

    @Override
    public void validate() {

        service.preValidate(updateContactCommand);

        AuthenticatedWaoUser authenticatedWaoUser = session.getAuthenticatedWaoUser();

        try {
            service.validate(authenticatedWaoUser, updateContactCommand);
        } catch (ContactNotUpdatableException e) {
            session.addErrorMessages(t("wao.ui.contacts.validation.failure.not.updatable"));

        } catch (UnwantedContactContactStateMotifException e) {
            addFieldError("updateContactCommand.contact.contactStateMotif", t("wao.ui.form.Contact.error.unwantedContactStateMotif"));

        } catch (MissingContactNbObservantsException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState());
            addFieldError("updateContactCommand.contact.nbObservants", t("wao.ui.form.Contact.error.missingObserver", state));

        } catch (MissingContactObservationEndDateException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState());
            addFieldError("updateContactCommand.contact.observationEndDate", t("wao.ui.form.Contact.error.missingObservationEndDate", state));

        } catch (ContactDataInputDateBeforeObservationEndDateException e) {
            addFieldError("updateContactCommand.contact.dataInputDate", t("wao.ui.form.Contact.error.dataInputDateBeforeObservationEndDate"));

        } catch (InvalidContactObservationBeginDateException e) {
            addFieldError("updateContactCommand.contact.observationBeginDate", t("wao.ui.form.Contact.error.invalidObservationBeginDate"));

        } catch (MissingContactRestitutionException e) {
            addFieldError("updateContactCommand.contact.restitution", t("wao.ui.form.Contact.error.missingRestitution"));

        } catch (ContactDataInputDateAfterTodayException e) {
            addFieldError("updateContactCommand.contact.dataInputDate", t("wao.ui.form.Contact.error.dataInputDateAfterToday"));

        } catch (MissingContactCommentException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState());
            addFieldError("updateContactCommand.contact.comment", t("wao.ui.form.Contact.error.missingComment", state));

        } catch (MismatchContactMainObserverCompanyException e) {
            String companyName = e.getCompany().getName();
            String observerLogin = e.getObserver().getLogin();
            addFieldError("updateContactCommand.contact.mainObserver", t("wao.ui.form.Contact.error.mismatchCompanyForObserver", observerLogin, companyName));

        } catch (ContactRestitutionDateBeforeDataInputDateException e) {
            addFieldError("updateContactCommand.contact.restitution", t("wao.ui.form.Contact.error.transmissionDateBeforeDataInputDate"));

        } catch (ContactObservationEndDateBeforeBeginDateException e) {
            addFieldError("updateContactCommand.contact.observationEndDate", t("wao.ui.form.Contact.error.observationEndDateBeforeBeginDate"));

        } catch (DuplicatedContactMainObserverInSecondaryObserversException e) {
            addFieldError("updateContactCommand.contact.secondaryObservers", t("wao.ui.form.Contact.error.duplicatedMainObserverInSecondaryObservers"));

        } catch (MissingContactDataReliabilityException e) {
            addFieldError("updateContactCommand.contact.dataReliability", t("wao.ui.form.Contact.error.missingDataReliability"));

        } catch (MismatchContactSecondaryObserverCompanyException e) {
            String companyName = e.getCompany().getName();
            String observerLogin = e.getObserver().getLogin();
            addFieldError("updateContactCommand.contact.secondaryObservers", t("wao.ui.form.Contact.error.mismatchCompanyForObserver", observerLogin, companyName));

        } catch (MissingContactObservedDataControlException e) {
            addFieldError("updateContactCommand.contact.observedDataControl", t("wao.ui.form.Contact.error.missingObservedDataControl"));

        } catch (ContactWithObservedDataControlToCorrectionAskedException e) {
            addFieldError("updateContactCommand.contact.observedDataControl", t("wao.ui.form.Contact.error.observedDataControlToCorrectionAsked"));

        } catch (MissingContactStateMotifException e) {
            addFieldError("updateContactCommand.contact.contactStateMotif", t("wao.ui.form.Contact.error.missingContactStateMotif"));

        } catch (MissingContactDataInputDateException e) {
            addFieldError("updateContactCommand.contact.dataInputDate", t("wao.ui.form.Contact.error.missingDataInputDate"));

        } catch (MissingContactMainObserverException e) {
            addFieldError("updateContactCommand.contact.mainObserver", t("wao.ui.form.Contact.error.missingMainObserver"));

        } catch (MissingContactCommentAdminException e) {
            String dataReliability = WaoUtils.l(getLocale(), e.getContact().getDataReliability());
            addFieldError("updateContactCommand.contact.commentAdmin", t("wao.ui.form.Contact.error.missingCommentAdmin", dataReliability));

        } catch (ContactObservationEndDateAfterTodayException e) {
            addFieldError("updateContactCommand.contact.observationEndDate", t("wao.ui.form.Contact.error.observationEndDateAfterToday"));

        } catch (MissingContactObservationBeginDateException e) {
            String state = WaoUtils.l(getLocale(), e.getContact().getContactState());
            addFieldError("updateContactCommand.contact.observationBeginDate", t("wao.ui.form.Contact.error.missingObservationBeginDate", state));

        } catch (MissingContactMammalsInfoException e) {
            addFieldError("updateContactCommand.contact.mammalsInfo", t("wao.ui.form.Contact.error.missingMammalsInfo"));

        } catch (MissingContactTerrestrialLocationException e) {
            throw new UnsupportedOperationException();
        } catch (IllegalAcceptationException e) {
            throw new IllegalStateException("should never occur", e);
        }
    }

    @Override
    public String execute() throws Exception {

        service.save(updateContactCommand);

        session.addMessage(t("wao.ui.form.updateContactCommand.success"));

        session.setContactToHighlightId(updateContactCommand.getContact().getTopiaId());
        return SUCCESS;
    }

}