package fr.ifremer.wao.web.action;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Objects;
import com.opensymphony.xwork2.Preparable;
import fr.ifremer.wao.ContactsFilter;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.ContactState;
import fr.ifremer.wao.entity.DCF5Code;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.services.service.ContactsFilterValues;
import fr.ifremer.wao.services.service.ContactsService;
import fr.ifremer.wao.services.service.ObsMerContactsList;
import fr.ifremer.wao.web.WaoJspActionSupport;
import org.apache.commons.lang3.time.DateUtils;
import org.nuiton.i18n.I18n;
import org.nuiton.util.pagination.PaginationParameter;
import org.nuiton.util.pagination.PaginationResult;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Created on 4/3/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 4.0
 */
public class ContactsAction extends WaoJspActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    protected transient ContactsService service;

    protected int pageNumber;

    protected long firstRangePage;

    protected long lastRangePage;

    /**
     * Contacts filter.
     */
    protected ContactsFilter filter;

    /**
     * List of contacts according to the input pager and filter.
     */
    protected ObsMerContactsList contactsList;

    public void setService(ContactsService service) {
        this.service = service;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    public long getFirstRangePage() {
        return firstRangePage;
    }

    public long getLastRangePage() {
        return lastRangePage;
    }

    public ContactsFilter getFilter() {
        if (filter == null) {
            prepare();
        }
        return filter;
    }

    public List<Contact> getContacts() {
        return contactsList.getContacts().getElements();
    }

    public ContactsFilterValues getFilterValues() {
        return contactsList.getFilterValues();
    }

    public PaginationResult<Contact> getPagination() {
        return contactsList.getContacts();
    }

    @Override
    public void prepare() {

        if (filter == null) {
            filter = service.newContactFilter(getAuthenticatedWaoUser());
        }

    }

    @Override
    public void validate() {

        if (filter.getPeriodFrom() != null && filter.getPeriodTo() != null && filter.getPeriodTo().before(filter.getPeriodFrom())) {
            addFieldError("filter.periodTo", I18n.l(getLocale(), "wao.ui.periodToMustBeAfterPeriodFrom"));
        }

    }

    @Override
    public String execute() {

        filter = Objects.firstNonNull(session.getContactsFilter(), filter);

        return applyFilter();

    }

    public String resetFilter() {

        filter = service.newContactFilter(getAuthenticatedWaoUser());

        return applyFilter();

    }

    public String applyFilter() {

        getSession().setContactsFilter(filter);

        PaginationParameter pager = service.newContactsPaginationParameter(pageNumber, 25);
        contactsList = service.getContactsList(getAuthenticatedWaoUser(), filter, pager);

        firstRangePage = getPaginationFirstPage(contactsList.getContacts(), 10);
        lastRangePage = getPaginationLastPage(contactsList.getContacts(), 10, firstRangePage);

        return SUCCESS;

    }

    protected Date now = new Date();

    public String getFilterPeriodFromPlaceholder() {
        String placeholder = formatDate(DateUtils.addMonths(now, 1));
        return placeholder;
    }

    public String getFilterPeriodToPlaceholder() {
        String placeholder = formatDate(DateUtils.addYears(now, 1));
        return placeholder;
    }

    public boolean isFullView() {
        boolean fullView = getAuthenticatedWaoUser().isAdmin() || getAuthenticatedWaoUser().isProfessional();
        return fullView;
    }

    public String getMammalsInfo(Contact contact) {
        return escapeForToolTip(contact.getMammalsInfo());
    }

    public boolean isBoardingDone(Contact contact) {
        return ContactState.OBSERVATION_DONE == contact.getContactState();
    }

    public String getDivision(SampleRow sampleRow) {
        String result = "";
        for (FishingZone zone : sampleRow.getFishingZone()) {
            result += zone.getDistrictCode() + " ";
        }
        return result;
    }

    /**
     * Devrait résoudre un problème que pose le composant ck/tooltip.
     * <p/>
     * Voir ANO#606
     */
    public static String escapeForToolTip(String str) {
        String result = str.replaceAll("\r\n", "\n");
        result = result.replaceAll("\n", "<br />");
        result = result.replaceAll("'", "&rsquo;");
        result = result.replaceAll("\"", "&#34;");
        result = result.replaceAll("%", "&#37;");
        return result;
    }

    public String getContactToHighlightId() {
        return session.getContactToHighlightId();
    }

    public Map<String, String> getDcf5CodesAndDescriptions(SampleRow sampleRow) {
        Map<String, String> dcf5CodesAndDescriptions = new LinkedHashMap<>();
        for (DCF5Code dcf5Code : sampleRow.getdCF5Code()) {
            String key = dcf5Code.getCode();
            String description = dcf5Code.getFishingGearCode() + " - " + WaoUtils.l(getLocale(), dcf5Code.getFishingGearDCF());
            if (dcf5Code.getTargetSpeciesCode() != null) {
                description += " ; " + dcf5Code.getTargetSpeciesCode() + " - " + WaoUtils.l(getLocale(), dcf5Code.getTargetSpeciesDCF());
            }
            dcf5CodesAndDescriptions.put(key, description);
        }
        return dcf5CodesAndDescriptions;
    }

    public boolean isAcceptable(Contact contact) {
        boolean acceptable;
        if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            acceptable = contact.isAcceptableByCompany();
        } else if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            acceptable = contact.isAcceptableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return acceptable;
    }

    public boolean isRefusable(Contact contact) {
        boolean refuseable;
        if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            refuseable = contact.isRefusableByCompany();
        } else if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            refuseable = contact.isRefusableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return refuseable;
    }

    public boolean isUnacceptable(Contact contact) {
        boolean unacceptable;
        if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationCompany()) {
            unacceptable = contact.isUnacceptableByCompany();
        } else if (getAuthenticatedWaoUser().isAuthorizedToChangeValidationProgram()) {
            unacceptable = contact.isUnacceptableByProgram();
        } else {
            throw new IllegalStateException();
        }
        return unacceptable;
    }

}