package fr.ifremer.wao.web;

/*
 * #%L
 * Wao :: Web
 * %%
 * Copyright (C) 2009 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.Interceptor;
import fr.ifremer.wao.WaoApplicationConfig;
import fr.ifremer.wao.WaoTopiaPersistenceContext;
import fr.ifremer.wao.services.WaoApplicationContext;
import fr.ifremer.wao.services.WaoService;
import fr.ifremer.wao.services.WaoServiceContext;
import fr.ifremer.wao.web.action.authentication.LoginAction;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;

import java.beans.PropertyDescriptor;
import java.util.Locale;
import java.util.Set;

public class WaoInterceptor implements Interceptor {

    private static final Log log = LogFactory.getLog(WaoInterceptor.class);

    protected static final ImmutableSet<Class<LoginAction>> ACTIONS_ACCESSIBLE_WITHOUT_LOGIN =
            ImmutableSet.of(LoginAction.class);

    @Override
    public void init() {

        if (log.isInfoEnabled()) {
            log.info("init " + this);
        }

    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        Object action = invocation.getAction();

        if (action instanceof WaoActionSupport) {

            WaoActionSupport waoAction = (WaoActionSupport) action;

            WaoSession waoSession = getWaoSession(invocation);

            if (waoSession.getAuthenticatedWaoUser() == null && ! ACTIONS_ACCESSIBLE_WITHOUT_LOGIN.contains(waoAction.getClass())) {

                return "redirect-to-login";

            } else {

                if (CollectionUtils.isNotEmpty(waoSession.getMessages())) {

                    for (String message : waoSession.getMessages()) {

                        waoAction.addActionMessage(message);

                    }

                    waoSession.getMessages().clear();

                }

                if (CollectionUtils.isNotEmpty(waoSession.getErrorMessages())) {

                    for (String message : waoSession.getErrorMessages()) {

                        waoAction.addActionError(message);

                    }

                    waoSession.getErrorMessages().clear();

                }

                Set<PropertyDescriptor> descriptors =
                        BeanUtil.getDescriptors(
                                action.getClass(),
                                BeanUtil.IS_WRITE_DESCRIPTOR);

                Locale locale = waoAction.getLocale();

                WaoServiceContext serviceContext = null;

                try {

                    serviceContext = newServiceContext(invocation, locale);

                    for (PropertyDescriptor propertyDescriptor : descriptors) {

                        Class<?> propertyType = propertyDescriptor.getPropertyType();

                        Object toInject = null;

                        if (WaoService.class.isAssignableFrom(propertyType)) {

                            Class<? extends WaoService> serviceClass =
                                    (Class<? extends WaoService>) propertyType;

                            toInject = serviceContext.newService(serviceClass);

                        } else if (WaoSession.class.isAssignableFrom(propertyType)) {

                            toInject = waoSession;

                        } else if (WaoApplicationConfig.class.isAssignableFrom(propertyType)) {

                            toInject = getWaoApplicationContext(invocation).getApplicationConfig();

                        }

                        if (toInject != null) {

                            if (log.isTraceEnabled()) {
                                log.trace("injecting " + toInject + " in action " + action);
                            }

                            propertyDescriptor.getWriteMethod().invoke(action, toInject);

                        }
                    }

                    return invocation.invoke();

                } catch (Throwable t) {

                    // TODO brendan 21/07/14 use try with resources
                    if (serviceContext != null) {
                        serviceContext.getPersistenceContext().rollback();
                    }

                    throw t;

                } finally {

                    // TODO brendan 21/07/14 use try with resources
                    if (serviceContext != null) {
                        serviceContext.getPersistenceContext().close();
                    }

                }

            }

        } else {

            // access to a static data (css, image...) just process
            return invocation.invoke();

        }

    }

    protected WaoSession getWaoSession(ActionInvocation invocation) {

        WaoSession session = (WaoSession) invocation.getInvocationContext().getSession().get(WaoSession.SESSION_PARAMETER);

        if (session == null) {

            session = new WaoSession();

            invocation.getInvocationContext().getSession().put(WaoSession.SESSION_PARAMETER, session);

        }

        return session;

    }

    protected WaoApplicationContext getWaoApplicationContext(ActionInvocation invocation) {

        DefaultWaoApplicationContext applicationContext =
                (DefaultWaoApplicationContext) invocation
                        .getInvocationContext()
                        .getApplication()
                        .get(DefaultWaoApplicationContext.APPLICATION_CONTEXT_PARAMETER);

        Preconditions.checkNotNull(applicationContext,
                "application context must be initialized before calling an action");

        return applicationContext;

    }

    protected WaoServiceContext newServiceContext(ActionInvocation invocation, Locale locale) {

        WaoApplicationContext waoApplicationContext = getWaoApplicationContext(invocation);

        WaoTopiaPersistenceContext persistenceContext = waoApplicationContext.newPersistenceContext();

        WaoServiceContext serviceContext =
                waoApplicationContext.newServiceContext(persistenceContext, locale);

        return serviceContext;

    }

    @Override
    public void destroy() {

        if (log.isInfoEnabled()) {
            log.info("destroy " + this);
        }

    }

}
