package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosFrequencyUIModel.java 1564 2014-01-31 10:56:26Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/frequency/BenthosFrequencyUIModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Sets;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.ui.swing.content.operation.catches.FrequencyConfigurationMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIModel;
import org.apache.commons.collections.CollectionUtils;

import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyUIModel extends AbstractTuttiTableUIModel<BenthosBatchRowModel, BenthosFrequencyRowModel, BenthosFrequencyUIModel> {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_BATCH = "batch";

    public static final String PROPERTY_CONFIGURATION_MODE = "configurationMode";

    public static final String PROPERTY_STEP = "step";

    private static final String PROPERTY_MIN_STEP = "minStep";

    private static final String PROPERTY_MAX_STEP = "maxStep";

    public static final String PROPERTY_CAN_GENERATE = "canGenerate";

    public static final String PROPERTY_AUTO_GEN_MODE = "autoGenMode";

    public static final String PROPERTY_RAFALE_MODE = "rafaleMode";

    public static final String PROPERTY_SIMPLE_COUNTING_MODE = "simpleCountingMode";

    public static final String PROPERTY_SIMPLE_COUNT = "simpleCount";

    public static final String PROPERTY_LENGHT_STEP_CARACTERISTIC = "lengthStepCaracteristic";

    public static final String PROPERTY_TOTAL_NUMBER = "totalNumber";

    public static final String PROPERTY_TOTAL_WEIGHT = "totalWeight";

    public static final String PROPERTY_EMPTY_ROWS = "emptyRows";

    public static final String PROPERTY_NEXT_EDITABLE_ROW_INDEX = "nextEditableRowIndex";

    /**
     * Fill mode.
     *
     * @since 0.2
     */
    protected FrequencyConfigurationMode configurationMode;

    /**
     * Batch that contains frequencies.
     *
     * @since 0.2
     */
    protected BenthosBatchRowModel batch;

    /**
     * Default step to increment length step.
     *
     * @since 0.2
     */
    protected float step;

    /**
     * Min step to auto generate length steps.
     *
     * @since 0.2
     */
    protected Float minStep;

    /**
     * Max step to auto generate length steps.
     *
     * @since 0.2
     */
    protected Float maxStep;

    /**
     * Length step caracteristic.
     *
     * @since 0.3
     */
    protected Caracteristic lengthStepCaracteristic;

    /**
     * Number in case of simple counting mode
     *
     * @since 1.0
     */
    protected Integer simpleCount;

    /**
     * Sum of the number of each valid row
     *
     * @since 2.3
     */
    protected Integer totalNumber;

    /**
     * Sum of the weight of each valid row
     *
     * @since 2.3
     */
    protected Float totalWeight;

    /**
     * The index of the next editable row (null if none).
     *
     * @since 2.5
     */
    protected Integer nextEditableRowIndex;

    protected Set<BenthosFrequencyRowModel> emptyRows;

    /**
     * Sample categories model.
     *
     * @since 2.4
     */
    protected final SampleCategoryModel sampleCategoryModel;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    /**
     * Rows with a filled weight.
     *
     * @since 3.0
     */
    protected Set<BenthosFrequencyRowModel> withWeightRows = Sets.newHashSet();

    public BenthosFrequencyUIModel(WeightUnit weightUnit,
                                   SampleCategoryModel sampleCategoryModel) {
        super(BenthosBatchRowModel.class, null, null);
        this.weightUnit = weightUnit;
        this.sampleCategoryModel = sampleCategoryModel;
        setEmptyRows(Sets.<BenthosFrequencyRowModel>newHashSet());
    }

    @Override
    protected BenthosBatchRowModel newEntity() {
        return new BenthosBatchRowModel(weightUnit, sampleCategoryModel);
    }

//    @Override
//    public void setRows(List<BenthosFrequencyRowModel> rows) {
//        super.setRows(rows);
//        setEmptyRows(Sets.<BenthosFrequencyRowModel>newHashSet());
//    }

    public FrequencyConfigurationMode getConfigurationMode() {
        return configurationMode;
    }

    public void setConfigurationMode(FrequencyConfigurationMode configurationMode) {
        Object oldValue = getConfigurationMode();
        this.configurationMode = configurationMode;
        firePropertyChange(PROPERTY_CONFIGURATION_MODE, oldValue, configurationMode);
        firePropertyChange(PROPERTY_AUTO_GEN_MODE, null, isAutoGenMode());
        firePropertyChange(PROPERTY_RAFALE_MODE, null, isRafaleMode());
        firePropertyChange(PROPERTY_SIMPLE_COUNTING_MODE, null, isSimpleCountingMode());
    }

    public float getStep() {
        return step;
    }

    public void setStep(float step) {
        Object oldValue = getStep();
        this.step = step;
        firePropertyChange(PROPERTY_STEP, oldValue, step);
    }

    public Caracteristic getLengthStepCaracteristic() {
        return lengthStepCaracteristic;
    }

    public void setLengthStepCaracteristic(Caracteristic lengthStepCaracteristic) {
        Object oldValue = getLengthStepCaracteristic();
        this.lengthStepCaracteristic = lengthStepCaracteristic;
        firePropertyChange(PROPERTY_LENGHT_STEP_CARACTERISTIC, oldValue, lengthStepCaracteristic);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
    }

    public Float getMinStep() {
        return minStep;
    }

    public void setMinStep(Float minStep) {
        Object oldValue = getMinStep();
        this.minStep = minStep;
        firePropertyChange(PROPERTY_MIN_STEP, oldValue, minStep);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
    }

    public Float getMaxStep() {
        return maxStep;
    }

    public void setMaxStep(Float maxStep) {
        Object oldValue = getMaxStep();
        this.maxStep = maxStep;
        firePropertyChange(PROPERTY_MAX_STEP, oldValue, maxStep);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
    }

    public Integer getSimpleCount() {
        return simpleCount;
    }

    public void setSimpleCount(Integer simpleCount) {
        Object oldValue = getSimpleCount();
        this.simpleCount = simpleCount;
        firePropertyChange(PROPERTY_SIMPLE_COUNT, oldValue, simpleCount);
    }

    public Integer getNextEditableRowIndex() {
        return nextEditableRowIndex;
    }

    public void setNextEditableRowIndex(Integer nextEditableRowIndex) {
        Object oldValue = getNextEditableRowIndex();
        this.nextEditableRowIndex = nextEditableRowIndex;
        firePropertyChange(PROPERTY_NEXT_EDITABLE_ROW_INDEX, oldValue, nextEditableRowIndex);
    }

    public boolean isAutoGenMode() {
        return FrequencyConfigurationMode.AUTO_GEN == configurationMode;
    }

    public boolean isRafaleMode() {
        return FrequencyConfigurationMode.RAFALE == configurationMode;
    }

    public boolean isSimpleCountingMode() {
        return FrequencyConfigurationMode.SIMPLE_COUNTING == configurationMode;
    }

    public boolean isCanGenerate() {
        return minStep != null && maxStep != null && maxStep > minStep && lengthStepCaracteristic != null;
    }

    public BenthosBatchRowModel getBatch() {
        return batch;
    }

    public void setBatch(BenthosBatchRowModel batch) {
        this.batch = batch;
        firePropertyChange(PROPERTY_BATCH, null, batch);
    }

    public float getLengthStep(float lengthStep) {
        int intValue = (int) (lengthStep * 10);
        int intStep = (int) (step * 10);
        int correctIntStep = intValue - (intValue % intStep);
        float result = correctIntStep / 10f;
        return result;
    }

    public Integer getTotalNumber() {
        return totalNumber;
    }

    public void setTotalNumber(Integer totalNumber) {
        Object oldValue = getTotalNumber();
        this.totalNumber = totalNumber;
        firePropertyChange(PROPERTY_TOTAL_NUMBER, oldValue, totalNumber);
    }

    public Float getTotalWeight() {
        return totalWeight;
    }

    public void setTotalWeight(Float totalWeight) {
        Object oldValue = getTotalWeight();
        this.totalWeight = totalWeight;
        firePropertyChange(PROPERTY_TOTAL_WEIGHT, oldValue, totalWeight);
    }

    public Set<BenthosFrequencyRowModel> getEmptyRows() {
        return emptyRows;
    }

    public void setEmptyRows(Set<BenthosFrequencyRowModel> emptyRows) {
        this.emptyRows = emptyRows;
        firePropertyChange(PROPERTY_EMPTY_ROWS, null, emptyRows);
    }

    public void clearWithWeightRows() {
        withWeightRows.clear();
    }

    public int getNbRowsWithWeight() {
        return withWeightRows.size();
    }

    public boolean isAllRowsWithWeight() {
        return withWeightRows.size() == rows.size();
    }

    public boolean isSomeRowsWithWeightAndOtherWithout() {
        boolean result;
        if (CollectionUtils.isEmpty(rows)) {
            // no row
            result = false;
        } else {

            // there is some rows
            int nbRowsWithWeight = getNbRowsWithWeight();
            int nbEmptyRows = emptyRows.size();
            if (nbRowsWithWeight == 0 || (nbEmptyRows + nbRowsWithWeight == rows.size())) {

                // no row with weight or all none empty rows have a weight
                result = false;
            } else {

                // some rows with weight and other without
                result = true;
            }
        }
        return result;
    }

    public void updateRowWithWeight(BenthosFrequencyRowModel row) {

        if (row.getWeight() == null) {
            withWeightRows.remove(row);
        } else {
            withWeightRows.add(row);
        }
    }

    public void updateEmptyRow(BenthosFrequencyRowModel row) {
        if (row.isValid() && row.getNumber() == null && row.getWeight() == null) {
            emptyRows.add(row);
        } else {
            emptyRows.remove(row);
        }
        firePropertyChange(PROPERTY_EMPTY_ROWS, null, emptyRows);
    }

    public Float computeTotalWeight() {
        float result = 0f;
        for (BenthosFrequencyRowModel row : withWeightRows) {
            if (!row.isValid()) {
                continue;
            }
            result += row.getWeight();
        }
        return result;
    }

    public int computeTotalNumber() {
        int result = 0;
        if (rows != null) {
            for (BenthosFrequencyRowModel row : rows) {
                if (!row.isValid()) {
                    continue;
                }
                if (row.getNumber() != null) {
                    result += row.getNumber();
                }
            }
        }
        return result;
    }

    public void recomputeTotalNumberAndWeight() {
        int totalNumber = computeTotalNumber();
        Float totalWeight = computeTotalWeight();
        setTotalNumber(totalNumber);
        setTotalWeight(totalWeight);
    }
}
