package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiActionHelper.java 930 2013-05-08 14:41:35Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.8/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/TuttiActionHelper.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.ui.swing.content.AbstractMainUITuttiAction;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import org.apache.commons.lang3.reflect.ConstructorUtils;

import javax.swing.AbstractButton;
import javax.swing.Action;

import static org.nuiton.i18n.I18n._;

/**
 * Useful method around TuttiUIAction.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.2
 */
public class TuttiActionHelper {

    public static <A extends AbstractTuttiAction> void runInternalAction(A action) {
        Throwable error = null;

        try {
            action.doAction();
            action.postSuccessAction();
        } catch (Throwable e) {
            error = e;
            throw TuttiActionException.propagateError(action, e);
        } finally {
            try {
                if (error != null) {
                    action.postFailedAction(error);
                }
            } finally {

                action.releaseAction();
            }
        }
    }

    public static <A extends AbstractTuttiAction> void runInternalAction(AbstractTuttiUIHandler handler,
                                                                         Class<A> actionName) {
        A action = createLogicAction(handler, actionName);
        runInternalAction(action);
    }

    public static <A extends AbstractTuttiAction> void runAction(A action) {
        TuttiUIAction<A> uiAction = createUIAction(null, action);
        uiAction.actionPerformed(null);
    }

    public static void runAction(AbstractButton button) {
        button.getAction().actionPerformed(null);
    }

    public static <A extends AbstractTuttiAction> void runActionAndWait(A action) {
        TuttiUIAction<A> uiAction = createUIAction(null, action);
        uiAction.launchActionAndWait();
    }

    public static <A extends AbstractTuttiAction> void runActionAndWait(AbstractTuttiUIHandler handler,
                                                                        Class<A> actionName) {
        A logicAction = createLogicAction(handler, actionName);
        runActionAndWait(logicAction);
    }

    public static <A extends AbstractTuttiAction> TuttiUIAction<A> createUIAction(AbstractTuttiUIHandler handler,
                                                                                  Class<A> actionName) {
        return createUIAction(handler, null, actionName);
    }


    public static <A extends AbstractTuttiAction> TuttiUIAction<A> createUIAction(AbstractTuttiUIHandler handler,
                                                                                  AbstractButton abstractButton,
                                                                                  Class<A> actionName) {
        try {

            // create logic action
            A logicAction = createLogicAction(handler, actionName);

            // create ui action
            TuttiUIAction<A> result = createUIAction(abstractButton, logicAction);
            return result;
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.action.create.error", actionName), e);
        }

    }

    public static <A extends AbstractTuttiAction> TuttiUIAction<A> createUIAction(AbstractButton abstractButton,
                                                                                  A logicAction) {

        // create ui action
        TuttiUIAction<A> result = new TuttiUIAction<A>(abstractButton,
                                                       logicAction);
        return result;


    }

    public static <A extends AbstractTuttiAction> A createLogicAction(AbstractTuttiUIHandler handler,
                                                                      Class<A> actionName) {
        try {

            if (AbstractMainUITuttiAction.class.isAssignableFrom(actionName) &&
                handler.getContext().getMainUI() != null) {
                handler = handler.getContext().getMainUI().getHandler();
            }

            // create action
            A result = ConstructorUtils.invokeConstructor(actionName, handler);
            return result;
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.action.create.error", actionName), e);
        }
    }

    public static <A extends AbstractTuttiAction> A getLogicAction(AbstractButton b) {
        Action action = b.getAction();
        Preconditions.checkNotNull(action);
        Preconditions.checkState(action instanceof TuttiUIAction);
        return ((TuttiUIAction<A>) action).getLogicAction();
    }
}
